// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/**
 * The |title| is the text label displayed for the bookmark.
 *
 * The bookmark may point at a location in the PDF or a URI.
 * If it points at a location, |page| indicates which 0-based page it leads to.
 * Optionally, |y| is the y position in that page, in pixel coordinates.
 * If it points at an URI, |uri| is the target for that bookmark.
 *
 * |children| is an array of the |Bookmark|s that are below this in a table of
 * contents tree
 * structure.
 * @typedef {{
 *   title: string,
 *   page: (number | undefined),
 *   y: (number | undefined),
 *   uri: (string | undefined),
 *   children: !Array<!Bookmark>
 * }}
 */
let Bookmark;

(function() {
/** Amount that each level of bookmarks is indented by (px). */
const BOOKMARK_INDENT = 20;

Polymer({
  is: 'viewer-bookmark',

  properties: {
    /** @type {Bookmark} */
    bookmark: {
      type: Object,
      observer: 'bookmarkChanged_',
    },

    depth: {
      type: Number,
      observer: 'depthChanged_'
    },

    /** @private */
    childDepth_: Number,

    /** @private */
    childrenShown_: {
      type: Boolean,
      reflectToAttribute: true,
      value: false,
    },

    /** @type {?HTMLElement} The target for the key bindings below. */
    keyEventTarget: Object,
  },

  behaviors: [Polymer.IronA11yKeysBehavior],

  keyBindings: {'enter': 'onEnter_', 'space': 'onSpace_'},

  /** @override */
  attached: function() {
    this.keyEventTarget = this.$.item;
  },

  /** @private */
  bookmarkChanged_: function() {
    this.$.expand.style.visibility =
        this.bookmark.children.length > 0 ? 'visible' : 'hidden';
  },

  /** @private */
  depthChanged_: function() {
    this.childDepth_ = this.depth + 1;
    this.$.item.style.paddingInlineStart =
        (this.depth * BOOKMARK_INDENT) + 'px';
  },

  /** @private */
  onClick_: function() {
    if (this.bookmark.hasOwnProperty('page')) {
      if (this.bookmark.hasOwnProperty('y')) {
        this.fire('change-page-and-xy', {
          page: this.bookmark.page,
          x: 0,
          y: this.bookmark.y,
          origin: 'bookmark'
        });
      } else {
        this.fire(
            'change-page', {page: this.bookmark.page, origin: 'bookmark'});
      }
    } else if (this.bookmark.hasOwnProperty('uri')) {
      this.fire('navigate', {uri: this.bookmark.uri, newtab: true});
    }
  },

  /**
   * @param {!KeyboardEvent} e
   * @private
   */
  onEnter_: function(e) {
    // Don't allow events which have propagated up from the expand button to
    // trigger a click.
    if (e.detail.keyboardEvent.target != this.$.expand) {
      this.onClick_();
    }
  },

  /**
   * @param {!KeyboardEvent} e
   * @private
   */
  onSpace_: function(e) {
    // cr-icon-button stops propagation of space events, so there's no need
    // to check the event source here.
    this.onClick_();
    // Prevent default space scroll behavior.
    e.detail.keyboardEvent.preventDefault();
  },

  /**
   * @param {!Event} e
   * @private
   */
  toggleChildren_: function(e) {
    this.childrenShown_ = !this.childrenShown_;
    e.stopPropagation();  // Prevent the above onClick_ handler from firing.
  }
});
})();
