// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/permissions/permission_request_manager.h"

#include <algorithm>

#include "base/bind.h"
#include "base/command_line.h"
#include "base/containers/circular_deque.h"
#include "base/feature_list.h"
#include "base/metrics/user_metrics.h"
#include "base/metrics/user_metrics_action.h"
#include "base/strings/string16.h"
#include "base/task/post_task.h"
#include "base/threading/sequenced_task_runner_handle.h"
#include "build/build_config.h"
#include "chrome/browser/permissions/adaptive_quiet_notification_permission_ui_enabler.h"
#include "chrome/browser/permissions/contextual_notification_permission_ui_selector.h"
#include "chrome/browser/permissions/notification_permission_ui_selector.h"
#include "chrome/browser/permissions/permission_decision_auto_blocker_factory.h"
#include "chrome/browser/permissions/permission_uma_util.h"
#include "chrome/browser/permissions/quiet_notification_permission_ui_config.h"
#include "chrome/browser/permissions/quiet_notification_permission_ui_state.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/ui/permission_bubble/permission_prompt.h"
#include "chrome/common/chrome_features.h"
#include "chrome/common/chrome_switches.h"
#include "components/permissions/permission_decision_auto_blocker.h"
#include "components/permissions/permission_request.h"
#include "components/url_formatter/elide_url.h"
#include "content/public/browser/back_forward_cache.h"
#include "content/public/browser/browser_task_traits.h"
#include "content/public/browser/browser_thread.h"
#include "content/public/browser/navigation_handle.h"
#include "content/public/browser/web_contents.h"
#include "url/origin.h"

#if !defined(OS_ANDROID)
#include "chrome/browser/extensions/extension_ui_util.h"
#include "extensions/common/constants.h"
#endif

#if defined(OS_CHROMEOS)
#include "chrome/browser/chromeos/app_mode/web_app/web_kiosk_app_data.h"
#include "chrome/browser/chromeos/app_mode/web_app/web_kiosk_app_manager.h"
#include "components/user_manager/user.h"
#include "components/user_manager/user_manager.h"
#endif

namespace {

bool IsMessageTextEqual(permissions::PermissionRequest* a,
                        permissions::PermissionRequest* b) {
  if (a == b)
    return true;
  if (a->GetMessageTextFragment() == b->GetMessageTextFragment() &&
      a->GetOrigin() == b->GetOrigin()) {
    return true;
  }
  return false;
}

// We only group together media requests. We don't display grouped requests for
// any other permissions at present.
bool ShouldGroupRequests(permissions::PermissionRequest* a,
                         permissions::PermissionRequest* b) {
  if (a->GetOrigin() != b->GetOrigin())
    return false;

  if (a->GetPermissionRequestType() ==
      permissions::PermissionRequestType::PERMISSION_MEDIASTREAM_MIC) {
    return b->GetPermissionRequestType() ==
           permissions::PermissionRequestType::PERMISSION_MEDIASTREAM_CAMERA;
  }

  if (a->GetPermissionRequestType() ==
      permissions::PermissionRequestType::PERMISSION_MEDIASTREAM_CAMERA) {
    return b->GetPermissionRequestType() ==
           permissions::PermissionRequestType::PERMISSION_MEDIASTREAM_MIC;
  }

  return false;
}

}  // namespace

// PermissionRequestManager ----------------------------------------------------

PermissionRequestManager::~PermissionRequestManager() {
  DCHECK(!IsRequestInProgress());
  DCHECK(duplicate_requests_.empty());
  DCHECK(queued_requests_.empty());
}

void PermissionRequestManager::AddRequest(
    permissions::PermissionRequest* request) {
  if (base::CommandLine::ForCurrentProcess()->HasSwitch(
          switches::kDenyPermissionPrompts)) {
    request->PermissionDenied();
    request->RequestFinished();
    return;
  }

  if (is_notification_prompt_cooldown_active_ &&
      request->GetContentSettingsType() == ContentSettingsType::NOTIFICATIONS) {
    // Short-circuit by canceling rather than denying to avoid creating a large
    // number of content setting exceptions on Desktop / disabled notification
    // channels on Android.
    request->Cancelled();
    request->RequestFinished();
    return;
  }

  // TODO(tsergeant): change the UMA to no longer mention bubbles.
  base::RecordAction(base::UserMetricsAction("PermissionBubbleRequest"));

  // TODO(gbillock): is there a race between an early request on a
  // newly-navigated page and the to-be-cleaned-up requests on the previous
  // page? We should maybe listen to DidStartNavigationToPendingEntry (and
  // any other renderer-side nav initiations?). Double-check this for
  // correct behavior on interstitials -- we probably want to basically queue
  // any request for which GetVisibleURL != GetLastCommittedURL.
  const GURL& main_frame_url_ = web_contents()->GetLastCommittedURL();
  bool is_main_frame =
      url::Origin::Create(main_frame_url_)
          .IsSameOriginWith(url::Origin::Create(request->GetOrigin()));

#if defined(OS_CHROMEOS)
  // In web kiosk mode, all permission requests are auto-approved for the origin
  // of the main app.
  if (user_manager::UserManager::IsInitialized() &&
      user_manager::UserManager::Get()->IsLoggedInAsWebKioskApp()) {
    const AccountId& account_id =
        user_manager::UserManager::Get()->GetPrimaryUser()->GetAccountId();
    DCHECK(chromeos::WebKioskAppManager::IsInitialized());
    const chromeos::WebKioskAppData* app_data =
        chromeos::WebKioskAppManager::Get()->GetAppByAccountId(account_id);
    DCHECK(app_data);
    if (url::Origin::Create(request->GetOrigin()) ==
        url::Origin::Create(app_data->install_url())) {
      request->PermissionGranted();
    }
    request->RequestFinished();
    return;
  }
#endif

  // Don't re-add an existing request or one with a duplicate text request.
  permissions::PermissionRequest* existing_request =
      GetExistingRequest(request);
  if (existing_request) {
    // |request| is a duplicate. Add it to |duplicate_requests_| unless it's the
    // same object as |existing_request| or an existing duplicate.
    if (request == existing_request)
      return;
    auto range = duplicate_requests_.equal_range(existing_request);
    for (auto it = range.first; it != range.second; ++it) {
      if (request == it->second)
        return;
    }
    duplicate_requests_.insert(std::make_pair(existing_request, request));
    return;
  }

  if (is_main_frame) {
    if (IsRequestInProgress()) {
      base::RecordAction(
          base::UserMetricsAction("PermissionBubbleRequestQueued"));
    }
  } else {
    base::RecordAction(
        base::UserMetricsAction("PermissionBubbleIFrameRequestQueued"));
  }
  queued_requests_.push_back(request);

  // If we're displaying a quiet permission request, kill it in favor of this
  // permission request.
  if (ShouldCurrentRequestUseQuietUI()) {
    // FinalizeBubble will call ScheduleDequeueRequest on its own.
    FinalizeBubble(permissions::PermissionAction::IGNORED);
  } else {
    ScheduleDequeueRequestIfNeeded();
  }
}

void PermissionRequestManager::UpdateAnchorPosition() {
  if (view_)
    view_->UpdateAnchorPosition();
}

void PermissionRequestManager::DidStartNavigation(
    content::NavigationHandle* navigation_handle) {
  if (!navigation_handle->IsInMainFrame() ||
      navigation_handle->IsSameDocument()) {
    return;
  }

  // Cooldown lasts until the next user-initiated navigation, which is defined
  // as either a renderer-initiated navigation with a user gesture, or a
  // browser-initiated navigation.
  //
  // TODO(crbug.com/952347): This check has to be done at DidStartNavigation
  // time, the HasUserGesture state is lost by the time the navigation commits.
  if (!navigation_handle->IsRendererInitiated() ||
      navigation_handle->HasUserGesture()) {
    is_notification_prompt_cooldown_active_ = false;
  }
}

void PermissionRequestManager::DidFinishNavigation(
    content::NavigationHandle* navigation_handle) {
  if (!navigation_handle->IsInMainFrame() ||
      !navigation_handle->HasCommitted() ||
      navigation_handle->IsSameDocument()) {
    return;
  }

  if (!queued_requests_.empty() || IsRequestInProgress()) {
    // |queued_requests_| and |requests_| will be deleted below, which
    // might be a problem for back-forward cache — the page might be restored
    // later, but the requests won't be.
    // Disable bfcache here if we have any requests here to prevent this
    // from happening.
    web_contents()
        ->GetController()
        .GetBackForwardCache()
        .DisableForRenderFrameHost(
            navigation_handle->GetPreviousRenderFrameHostId(),
            "PermissionRequestManager");
  }

  CleanUpRequests();
}

void PermissionRequestManager::DocumentOnLoadCompletedInMainFrame() {
  // This is scheduled because while all calls to the browser have been
  // issued at DOMContentLoaded, they may be bouncing around in scheduled
  // callbacks finding the UI thread still. This makes sure we allow those
  // scheduled calls to AddRequest to complete before we show the page-load
  // permissions bubble.
  ScheduleDequeueRequestIfNeeded();
}

void PermissionRequestManager::DOMContentLoaded(
    content::RenderFrameHost* render_frame_host) {
  ScheduleDequeueRequestIfNeeded();
}

void PermissionRequestManager::WebContentsDestroyed() {
  // If the web contents has been destroyed, treat the bubble as cancelled.
  CleanUpRequests();

  // The WebContents is going away; be aggressively paranoid and delete
  // ourselves lest other parts of the system attempt to add permission bubbles
  // or use us otherwise during the destruction.
  web_contents()->RemoveUserData(UserDataKey());
  // That was the equivalent of "delete this". This object is now destroyed;
  // returning from this function is the only safe thing to do.
}

void PermissionRequestManager::OnVisibilityChanged(
    content::Visibility visibility) {
  bool tab_was_hidden = tab_is_hidden_;
  tab_is_hidden_ = visibility == content::Visibility::HIDDEN;
  if (tab_was_hidden == tab_is_hidden_)
    return;

  if (tab_is_hidden_) {
    if (view_) {
      switch (view_->GetTabSwitchingBehavior()) {
        case PermissionPrompt::TabSwitchingBehavior::
            kDestroyPromptButKeepRequestPending:
          DeleteBubble();
          break;
        case PermissionPrompt::TabSwitchingBehavior::
            kDestroyPromptAndIgnoreRequest:
          FinalizeBubble(permissions::PermissionAction::IGNORED);
          break;
        case PermissionPrompt::TabSwitchingBehavior::kKeepPromptAlive:
          break;
      }
    }

    return;
  }

  if (!web_contents()->IsDocumentOnLoadCompletedInMainFrame())
    return;

  if (!IsRequestInProgress()) {
    ScheduleDequeueRequestIfNeeded();
    return;
  }

  if (view_) {
    // We switched tabs away and back while a prompt was active.
    DCHECK_EQ(view_->GetTabSwitchingBehavior(),
              PermissionPrompt::TabSwitchingBehavior::kKeepPromptAlive);
  } else if (current_request_ui_to_use_.has_value()) {
    ShowBubble();
  }
}

const std::vector<permissions::PermissionRequest*>&
PermissionRequestManager::Requests() {
  return requests_;
}

PermissionPrompt::DisplayNameOrOrigin
PermissionRequestManager::GetDisplayNameOrOrigin() {
  DCHECK(IsRequestInProgress());
  GURL origin_url = requests_[0]->GetOrigin();

#if !defined(OS_ANDROID)
  if (origin_url.SchemeIs(extensions::kExtensionScheme)) {
    base::string16 extension_name =
        extensions::ui_util::GetEnabledExtensionNameForUrl(
            origin_url, web_contents()->GetBrowserContext());
    if (!extension_name.empty())
      return {extension_name, false /* is_origin */};
  }
#endif  // !defined(OS_ANDROID)

  // Web URLs should be displayed as the origin in the URL.
  return {url_formatter::FormatUrlForSecurityDisplay(
              origin_url, url_formatter::SchemeDisplay::OMIT_CRYPTOGRAPHIC),
          true /* is_origin */};
}

void PermissionRequestManager::Accept() {
  DCHECK(view_);
  std::vector<permissions::PermissionRequest*>::iterator requests_iter;
  for (requests_iter = requests_.begin(); requests_iter != requests_.end();
       requests_iter++) {
    PermissionGrantedIncludingDuplicates(*requests_iter);
  }
  FinalizeBubble(permissions::PermissionAction::GRANTED);
}

void PermissionRequestManager::Deny() {
  DCHECK(view_);

  // Suppress any further prompts in this WebContents, from any origin, until
  // there is a user-initiated navigation. This stops users from getting trapped
  // in request loops where the website automatically navigates cross-origin
  // (e.g. to another subdomain) to be able to prompt again after a rejection.
  if (base::FeatureList::IsEnabled(
          features::kBlockRepeatedNotificationPermissionPrompts) &&
      std::any_of(requests_.begin(), requests_.end(), [](const auto* request) {
        return request->GetContentSettingsType() ==
               ContentSettingsType::NOTIFICATIONS;
      })) {
    is_notification_prompt_cooldown_active_ = true;
  }

  std::vector<permissions::PermissionRequest*>::iterator requests_iter;
  for (requests_iter = requests_.begin();
       requests_iter != requests_.end();
       requests_iter++) {
    PermissionDeniedIncludingDuplicates(*requests_iter);
  }
  FinalizeBubble(permissions::PermissionAction::DENIED);
}

void PermissionRequestManager::Closing() {
  DCHECK(view_);
  std::vector<permissions::PermissionRequest*>::iterator requests_iter;
  for (requests_iter = requests_.begin();
       requests_iter != requests_.end();
       requests_iter++) {
    CancelledIncludingDuplicates(*requests_iter);
  }
  FinalizeBubble(permissions::PermissionAction::DISMISSED);
}

PermissionRequestManager::PermissionRequestManager(
    content::WebContents* web_contents)
    : content::WebContentsObserver(web_contents),
      view_factory_(base::Bind(&PermissionPrompt::Create)),
      view_(nullptr),
      tab_is_hidden_(web_contents->GetVisibility() ==
                     content::Visibility::HIDDEN),
      auto_response_for_test_(NONE),
      notification_permission_ui_selector_(
          std::make_unique<ContextualNotificationPermissionUiSelector>(
              Profile::FromBrowserContext(web_contents->GetBrowserContext()))) {
}

void PermissionRequestManager::ScheduleShowBubble() {
  base::RecordAction(base::UserMetricsAction("PermissionBubbleRequest"));
  base::PostTask(FROM_HERE, {content::BrowserThread::UI},
                 base::BindOnce(&PermissionRequestManager::ShowBubble,
                                weak_factory_.GetWeakPtr()));
}

void PermissionRequestManager::DequeueRequestIfNeeded() {
  if (!web_contents()->IsDocumentOnLoadCompletedInMainFrame() || view_ ||
      queued_requests_.empty() || IsRequestInProgress()) {
    return;
  }

  requests_.push_back(queued_requests_.front());
  queued_requests_.pop_front();

  if (!queued_requests_.empty() &&
      ShouldGroupRequests(requests_.front(), queued_requests_.front())) {
    requests_.push_back(queued_requests_.front());
    queued_requests_.pop_front();
  }

  if (requests_.front()->GetPermissionRequestType() ==
      permissions::PermissionRequestType::PERMISSION_NOTIFICATIONS) {
    notification_permission_ui_selector_->SelectUiToUse(
        requests_.front(),
        base::BindOnce(
            &PermissionRequestManager::OnSelectedUiToUseForNotifications,
            weak_factory_.GetWeakPtr()));
  } else {
    current_request_ui_to_use_ = UiToUse::kNormalUi;
    ScheduleShowBubble();
  }
}

void PermissionRequestManager::ScheduleDequeueRequestIfNeeded() {
  base::SequencedTaskRunnerHandle::Get()->PostTask(
      FROM_HERE,
      base::BindOnce(&PermissionRequestManager::DequeueRequestIfNeeded,
                     weak_factory_.GetWeakPtr()));
}

void PermissionRequestManager::ShowBubble() {
  // There is a race condition where the request might have been removed already
  // so double-checking that there is a request in progress (crbug.com/1041222).
  if (!IsRequestInProgress())
    return;

  DCHECK(!view_);
  DCHECK(web_contents()->IsDocumentOnLoadCompletedInMainFrame());
  DCHECK(current_request_ui_to_use_);

  if (tab_is_hidden_)
    return;

  view_ = view_factory_.Run(web_contents(), this);
  if (!view_)
    return;

  if (!current_request_view_shown_to_user_) {
    PermissionUmaUtil::PermissionPromptShown(requests_);

    if (ShouldCurrentRequestUseQuietUI()) {
      base::RecordAction(base::UserMetricsAction(
          "Notifications.Quiet.PermissionRequestShown"));
    }
  }
  current_request_view_shown_to_user_ = true;
  NotifyBubbleAdded();

  // If in testing mode, automatically respond to the bubble that was shown.
  if (auto_response_for_test_ != NONE)
    DoAutoResponseForTesting();
}

void PermissionRequestManager::DeleteBubble() {
  DCHECK(view_);
  view_.reset();
  NotifyBubbleRemoved();
}

void PermissionRequestManager::FinalizeBubble(
    permissions::PermissionAction permission_action) {
  DCHECK(IsRequestInProgress());

  PermissionUmaUtil::PermissionPromptResolved(
      requests_, web_contents(), permission_action,
      DetermineCurrentRequestUIDispositionForUMA());

  Profile* profile =
      Profile::FromBrowserContext(web_contents()->GetBrowserContext());
  permissions::PermissionDecisionAutoBlocker* autoblocker =
      PermissionDecisionAutoBlockerFactory::GetForProfile(profile);

  auto* adaptive_notification_permission_ui_enabler =
      AdaptiveQuietNotificationPermissionUiEnabler::GetForProfile(profile);
  for (permissions::PermissionRequest* request : requests_) {
    // TODO(timloh): We only support dismiss and ignore embargo for permissions
    // which use PermissionRequestImpl as the other subclasses don't support
    // GetContentSettingsType.
    if (request->GetContentSettingsType() == ContentSettingsType::DEFAULT)
      continue;

    if (request->GetPermissionRequestType() ==
        permissions::PermissionRequestType::PERMISSION_NOTIFICATIONS) {
      adaptive_notification_permission_ui_enabler
          ->RecordPermissionPromptOutcome(permission_action);
    }

    PermissionEmbargoStatus embargo_status =
        PermissionEmbargoStatus::NOT_EMBARGOED;
    if (permission_action == permissions::PermissionAction::DISMISSED) {
      if (autoblocker->RecordDismissAndEmbargo(
              request->GetOrigin(), request->GetContentSettingsType(),
              ShouldCurrentRequestUseQuietUI())) {
        embargo_status = PermissionEmbargoStatus::REPEATED_DISMISSALS;
      }
    } else if (permission_action == permissions::PermissionAction::IGNORED) {
      if (autoblocker->RecordIgnoreAndEmbargo(
              request->GetOrigin(), request->GetContentSettingsType(),
              ShouldCurrentRequestUseQuietUI())) {
        embargo_status = PermissionEmbargoStatus::REPEATED_IGNORES;
      }
    }
    PermissionUmaUtil::RecordEmbargoStatus(embargo_status);
  }
  std::vector<permissions::PermissionRequest*>::iterator requests_iter;
  for (requests_iter = requests_.begin();
       requests_iter != requests_.end();
       requests_iter++) {
    RequestFinishedIncludingDuplicates(*requests_iter);
  }
  requests_.clear();

  notification_permission_ui_selector_->Cancel();

  current_request_view_shown_to_user_ = false;
  current_request_ui_to_use_.reset();
  current_request_quiet_ui_reason_.reset();

  if (view_)
    DeleteBubble();

  ScheduleDequeueRequestIfNeeded();
}

void PermissionRequestManager::CleanUpRequests() {
  for (permissions::PermissionRequest* request : queued_requests_)
    RequestFinishedIncludingDuplicates(request);
  queued_requests_.clear();

  if (IsRequestInProgress())
    FinalizeBubble(permissions::PermissionAction::IGNORED);
}

permissions::PermissionRequest* PermissionRequestManager::GetExistingRequest(
    permissions::PermissionRequest* request) {
  for (permissions::PermissionRequest* existing_request : requests_) {
    if (IsMessageTextEqual(existing_request, request))
      return existing_request;
  }
  for (permissions::PermissionRequest* existing_request : queued_requests_) {
    if (IsMessageTextEqual(existing_request, request))
      return existing_request;
  }
  return nullptr;
}

void PermissionRequestManager::PermissionGrantedIncludingDuplicates(
    permissions::PermissionRequest* request) {
  DCHECK_EQ(request, GetExistingRequest(request))
      << "Only requests in [queued_[frame_]]requests_ can have duplicates";
  request->PermissionGranted();
  auto range = duplicate_requests_.equal_range(request);
  for (auto it = range.first; it != range.second; ++it)
    it->second->PermissionGranted();
}

void PermissionRequestManager::PermissionDeniedIncludingDuplicates(
    permissions::PermissionRequest* request) {
  DCHECK_EQ(request, GetExistingRequest(request))
      << "Only requests in [queued_]requests_ can have duplicates";
  request->PermissionDenied();
  auto range = duplicate_requests_.equal_range(request);
  for (auto it = range.first; it != range.second; ++it)
    it->second->PermissionDenied();
}

void PermissionRequestManager::CancelledIncludingDuplicates(
    permissions::PermissionRequest* request) {
  DCHECK_EQ(request, GetExistingRequest(request))
      << "Only requests in [queued_]requests_ can have duplicates";
  request->Cancelled();
  auto range = duplicate_requests_.equal_range(request);
  for (auto it = range.first; it != range.second; ++it)
    it->second->Cancelled();
}

void PermissionRequestManager::RequestFinishedIncludingDuplicates(
    permissions::PermissionRequest* request) {
  // We can't call GetExistingRequest here, because other entries in requests_,
  // queued_requests_ might already have been deleted.
  DCHECK_EQ(1, std::count(requests_.begin(), requests_.end(), request) +
                   std::count(queued_requests_.begin(), queued_requests_.end(),
                              request))
      << "Only requests in [queued_]requests_ can have duplicates";
  request->RequestFinished();
  // Beyond this point, |request| has probably been deleted.
  auto range = duplicate_requests_.equal_range(request);
  for (auto it = range.first; it != range.second; ++it)
    it->second->RequestFinished();
  // Additionally, we can now remove the duplicates.
  duplicate_requests_.erase(request);
}

void PermissionRequestManager::AddObserver(Observer* observer) {
  observer_list_.AddObserver(observer);
}

void PermissionRequestManager::RemoveObserver(Observer* observer) {
  observer_list_.RemoveObserver(observer);
}

bool PermissionRequestManager::ShouldCurrentRequestUseQuietUI() const {
  if (!IsRequestInProgress())
    return false;

  // ContentSettingImageModel might call into this method if the user switches
  // between tabs while the |notification_permission_ui_selector_| is pending.
  return current_request_ui_to_use_ &&
         *current_request_ui_to_use_ == UiToUse::kQuietUi;
}

PermissionRequestManager::QuietUiReason
PermissionRequestManager::ReasonForUsingQuietUi() const {
  return *current_request_quiet_ui_reason_;
}

bool PermissionRequestManager::IsRequestInProgress() const {
  return !requests_.empty();
}

void PermissionRequestManager::NotifyBubbleAdded() {
  for (Observer& observer : observer_list_)
    observer.OnBubbleAdded();
}

void PermissionRequestManager::NotifyBubbleRemoved() {
  for (Observer& observer : observer_list_)
    observer.OnBubbleRemoved();
}

void PermissionRequestManager::OnSelectedUiToUseForNotifications(
    UiToUse ui_to_use,
    base::Optional<QuietUiReason> quiet_ui_reason) {
  current_request_ui_to_use_ = ui_to_use;
  current_request_quiet_ui_reason_ = quiet_ui_reason;
  ScheduleShowBubble();
}

PermissionPromptDisposition
PermissionRequestManager::DetermineCurrentRequestUIDispositionForUMA() {
#if defined(OS_ANDROID)
  return ShouldCurrentRequestUseQuietUI()
             ? PermissionPromptDisposition::MINI_INFOBAR
             : PermissionPromptDisposition::MODAL_DIALOG;
#else
  return !ShouldCurrentRequestUseQuietUI()
             ? PermissionPromptDisposition::ANCHORED_BUBBLE
             : ReasonForUsingQuietUi() == QuietUiReason::kTriggeredByCrowdDeny
                   ? PermissionPromptDisposition::LOCATION_BAR_RIGHT_STATIC_ICON
                   : PermissionPromptDisposition::
                         LOCATION_BAR_RIGHT_ANIMATED_ICON;
#endif
}

void PermissionRequestManager::DoAutoResponseForTesting() {
  switch (auto_response_for_test_) {
    case ACCEPT_ALL:
      Accept();
      break;
    case DENY_ALL:
      Deny();
      break;
    case DISMISS:
      Closing();
      break;
    case NONE:
      NOTREACHED();
  }
}

WEB_CONTENTS_USER_DATA_KEY_IMPL(PermissionRequestManager)
