// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.preferences;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import android.support.test.filters.SmallTest;

import org.junit.Test;
import org.junit.runner.RunWith;

import org.chromium.base.test.BaseRobolectricTestRunner;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * Test class that verifies that {@link ChromePreferenceKeys} conforms to its constraints:
 * - No keys are both in [keys in use] and in [deprecated keys].
 * - All keys follow the format "Chrome.[Feature].[Key]"
 */
@RunWith(BaseRobolectricTestRunner.class)
public class ChromePreferenceKeysTest {
    /**
     * The important test: verify that keys in {@link ChromePreferenceKeys} are not reused.
     *
     * If a key was used in the past but is not used anymore, it should be in [deprecated keys].
     * Adding the same key to [keys in use] will break this test to warn the developer.
     */
    @Test
    @SmallTest
    public void testKeysAreNotReused() {
        doTestKeysAreNotReused(ChromePreferenceKeys.createKeysInUse(),
                ChromePreferenceKeys.createGrandfatheredKeysInUse(),
                ChromePreferenceKeys.createDeprecatedKeysForTesting(),
                ChromePreferenceKeys.createGrandfatheredPrefixesInUse(),
                ChromePreferenceKeys.createDeprecatedPrefixesForTesting());
    }

    private void doTestKeysAreNotReused(List<String> usedList, List<String> grandfatheredUsedList,
            List<String> deprecatedList, List<KeyPrefix> usedGrandfatheredPrefixList,
            List<KeyPrefix> deprecatedGrandfatheredPrefixList) {
        // Check for duplicate keys in [keys in use].
        Set<String> usedSet = new HashSet<>(usedList);
        assertEquals(usedList.size(), usedSet.size());

        Set<String> grandfatheredUsedSet = new HashSet<>(grandfatheredUsedList);
        assertEquals(grandfatheredUsedList.size(), grandfatheredUsedSet.size());

        Set<String> intersection = new HashSet<>(usedSet);
        intersection.retainAll(grandfatheredUsedSet);
        if (!intersection.isEmpty()) {
            fail("\"" + intersection.iterator().next()
                    + "\" is both in ChromePreferenceKeys' regular and grandfathered "
                    + "[keys in use]");
        }

        Set<String> allKeysInUse = new HashSet<>(usedSet);
        allKeysInUse.addAll(grandfatheredUsedSet);

        // Check for duplicate keys in [deprecated keys].
        Set<String> deprecatedSet = new HashSet<>(deprecatedList);
        assertEquals(deprecatedList.size(), deprecatedSet.size());

        // Check for keys in [deprecated keys] that are now also [keys in use]. This ensures no
        // deprecated keys are reused.
        intersection = new HashSet<>(allKeysInUse);
        intersection.retainAll(deprecatedSet);
        if (!intersection.isEmpty()) {
            fail("\"" + intersection.iterator().next()
                    + "\" is both in ChromePreferenceKeys' [keys in use] and in "
                    + "[deprecated keys]");
        }

        // Check for keys that match a grandfathered prefix, deprecated or not.
        List<KeyPrefix> grandfatheredPrefixes = new ArrayList<>(usedGrandfatheredPrefixList);
        grandfatheredPrefixes.addAll(deprecatedGrandfatheredPrefixList);

        for (String usedKey : usedSet) {
            for (KeyPrefix grandfatheredPrefix : grandfatheredPrefixes) {
                assertFalse(grandfatheredPrefix.hasGenerated(usedKey));
            }
        }
    }

    // Below are tests to ensure that testKeysAreNotReused() works.

    @Test
    @SmallTest
    public void testReuseCheck_emptyLists() {
        doTestKeysAreNotReused(Collections.EMPTY_LIST, Collections.EMPTY_LIST,
                Collections.EMPTY_LIST, Collections.EMPTY_LIST, Collections.EMPTY_LIST);
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testReuseCheck_duplicateKey_used() {
        doTestKeysAreNotReused(Arrays.asList("UsedKey1", "UsedKey1"), Collections.EMPTY_LIST,
                Collections.EMPTY_LIST, Collections.EMPTY_LIST, Collections.EMPTY_LIST);
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testReuseCheck_duplicateKey_grandfathered() {
        doTestKeysAreNotReused(Collections.EMPTY_LIST,
                Arrays.asList("GrandfatheredKey1", "GrandfatheredKey1"), Collections.EMPTY_LIST,
                Collections.EMPTY_LIST, Collections.EMPTY_LIST);
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testReuseCheck_duplicateKey_deprecated() {
        doTestKeysAreNotReused(Collections.EMPTY_LIST, Collections.EMPTY_LIST,
                Arrays.asList("DeprecatedKey1", "DeprecatedKey1"), Collections.EMPTY_LIST,
                Collections.EMPTY_LIST);
    }

    @Test
    @SmallTest
    public void testReuseCheck_noIntersection() {
        doTestKeysAreNotReused(Arrays.asList("UsedKey1", "UsedKey2"),
                Arrays.asList("GrandfatheredKey1", "GrandfatheredKey2"),
                Arrays.asList("DeprecatedKey1", "DeprecatedKey2"),
                Arrays.asList(new KeyPrefix("UsedGrandfatheredFormat1*"),
                        new KeyPrefix("UsedGrandfatheredFormat2*")),
                Arrays.asList(
                        new KeyPrefix("DeprecatedFormat1*"), new KeyPrefix("DeprecatedFormat2*")));
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testReuseCheck_intersectionUsedAndGrandfathered() {
        doTestKeysAreNotReused(Arrays.asList("ReusedKey", "UsedKey1"),
                Arrays.asList("GrandfatheredKey1", "ReusedKey"), Collections.EMPTY_LIST,
                Collections.EMPTY_LIST, Collections.EMPTY_LIST);
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testReuseCheck_intersectionUsedAndDeprecated() {
        doTestKeysAreNotReused(Arrays.asList("UsedKey1", "ReusedKey"), Collections.EMPTY_LIST,
                Arrays.asList("ReusedKey", "DeprecatedKey1"), Collections.EMPTY_LIST,
                Collections.EMPTY_LIST);
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testReuseCheck_intersectionGrandfatheredAndDeprecated() {
        doTestKeysAreNotReused(Collections.EMPTY_LIST,
                Arrays.asList("GrandfatheredKey1", "ReusedKey"),
                Arrays.asList("ReusedKey", "DeprecatedKey1"), Collections.EMPTY_LIST,
                Collections.EMPTY_LIST);
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testReuseCheck_intersectionUsedGrandfatheredFormat_prefix() {
        doTestKeysAreNotReused(Arrays.asList("UsedKey1"), Collections.EMPTY_LIST,
                Collections.EMPTY_LIST, Arrays.asList(new KeyPrefix("UsedKey*")),
                Collections.EMPTY_LIST);
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testReuseCheck_intersectionDeprecatedGrandfatheredFormat_prefix() {
        doTestKeysAreNotReused(Arrays.asList("UsedKey1"), Collections.EMPTY_LIST,
                Collections.EMPTY_LIST, Collections.EMPTY_LIST,
                Arrays.asList(new KeyPrefix("Used*")));
    }

    /**
     * Test that the keys in use (not grandfathered) conform to the format:
     * "Chrome.[Feature].[Key]"
     */
    @Test
    @SmallTest
    public void testKeysConformToFormat() {
        doTestKeysConformToFormat(ChromePreferenceKeys.createKeysInUse());
    }

    /**
     * Old constants grandfathered in are checked to see if they shouldn't be in
     * {@link ChromePreferenceKeys#createKeysInUse()}.
     */
    @Test
    @SmallTest
    public void testGrandfatheredKeysDoNotConformToFormat() {
        doTestKeysDoNotConformToFormat(ChromePreferenceKeys.createGrandfatheredKeysInUse());
    }

    private void doTestKeysConformToFormat(List<String> usedList) {
        Pattern regex = buildValidKeyFormatPattern();
        for (String keyInUse : usedList) {
            assertTrue("\"" + keyInUse + "\" does not conform to format \"Chrome.[Feature].[Key]\"",
                    regex.matcher(keyInUse).matches());
        }
    }

    private void doTestKeysDoNotConformToFormat(List<String> grandfatheredList) {
        Pattern regex = buildValidKeyFormatPattern();
        for (String keyInUse : grandfatheredList) {
            assertFalse("\"" + keyInUse
                            + "\" conforms to format \"Chrome.[Feature].[Key]\", move it to ChromePreferenceKeys.createKeysInUse()",
                    regex.matcher(keyInUse).matches());
        }
    }

    private static Pattern buildValidKeyFormatPattern() {
        String term = "([A-Z][a-z0-9]*)+";
        return Pattern.compile("Chrome\\." + term + "\\." + term + "(\\.\\*)?");
    }

    // Below are tests to ensure that doTestKeysConformToFormat() works.

    private static class TestFormatConstantsClass {
        static final String GRANDFATHERED_IN = "grandfathered_in";
        static final String NEW1 = "Chrome.FeatureOne.Key1";
        static final String NEW2 = "Chrome.Foo.Key";
        static final String BROKEN_PREFIX = "Chrom.Foo.Key";
        static final String MISSING_FEATURE = "Chrome..Key";
        static final String LOWERCASE_KEY = "Chrome.Foo.key";

        static final KeyPrefix PREFIX = new KeyPrefix("Chrome.FeatureOne.KeyPrefix1.*");
        static final KeyPrefix PREFIX_EXTRA_LEVEL =
                new KeyPrefix("Chrome.FeatureOne.KeyPrefix1.ExtraLevel.*");
        static final KeyPrefix PREFIX_MISSING_LEVEL = new KeyPrefix("Chrome.FeatureOne.*");
    }

    @Test
    @SmallTest
    public void testFormatCheck_correct() {
        doTestKeysConformToFormat(
                Arrays.asList(TestFormatConstantsClass.NEW1, TestFormatConstantsClass.NEW2));
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testFormatCheck_invalidFormat() {
        doTestKeysConformToFormat(Arrays.asList(TestFormatConstantsClass.GRANDFATHERED_IN,
                TestFormatConstantsClass.NEW1, TestFormatConstantsClass.NEW2));
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testFormatCheck_brokenPrefix() {
        doTestKeysConformToFormat(Arrays.asList(
                TestFormatConstantsClass.NEW1, TestFormatConstantsClass.BROKEN_PREFIX));
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testFormatCheck_missingFeature() {
        doTestKeysConformToFormat(Arrays.asList(
                TestFormatConstantsClass.NEW1, TestFormatConstantsClass.MISSING_FEATURE));
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testFormatCheck_lowercaseKey() {
        doTestKeysConformToFormat(Arrays.asList(
                TestFormatConstantsClass.NEW1, TestFormatConstantsClass.LOWERCASE_KEY));
    }

    @Test
    @SmallTest
    public void testFormatCheck_prefixCorrect() {
        doTestKeysConformToFormat(Arrays.asList(TestFormatConstantsClass.PREFIX.pattern()));
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testFormatCheck_prefixExtraLevel() {
        doTestKeysConformToFormat(
                Arrays.asList(TestFormatConstantsClass.PREFIX_EXTRA_LEVEL.pattern()));
    }

    @Test(expected = AssertionError.class)
    @SmallTest
    public void testFormatCheck_prefixMissingLevel() {
        doTestKeysConformToFormat(
                Arrays.asList(TestFormatConstantsClass.PREFIX_MISSING_LEVEL.pattern()));
    }
}
