// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/ash/assistant/test/fake_s3_server.h"
#include "base/command_line.h"
#include "base/files/file_path.h"
#include "base/files/file_util.h"
#include "base/path_service.h"
#include "base/process/launch.h"
#include "base/strings/string_number_conversions.h"
#include "base/strings/string_util.h"
#include "base/strings/stringprintf.h"
#include "chromeos/assistant/internal/internal_constants.h"
#include "chromeos/services/assistant/service.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace chromeos {
namespace assistant {

namespace {

// Folder where the S3 communications are stored when running in replay mode.
constexpr const char kTestDataFolder[] =
    "chromeos/assistant/internal/test_data/";

base::FilePath GetExecutableDir() {
  base::FilePath result;
  base::PathService::Get(base::DIR_EXE, &result);
  return result;
}

base::FilePath GetSourceDir() {
  base::FilePath result;
  base::PathService::Get(base::DIR_SOURCE_ROOT, &result);
  return result;
}

std::string GetSanitizedTestName() {
  std::string test_name = base::ToLowerASCII(base::StringPrintf(
      "%s_%s",
      testing::UnitTest::GetInstance()->current_test_info()->test_case_name(),
      testing::UnitTest::GetInstance()->current_test_info()->name()));
  std::string new_test_name;
  base::ReplaceChars(test_name, "/", "_", &new_test_name);
  return new_test_name;
}

std::string GetTestDataFileName() {
  return GetSourceDir()
      .Append(FILE_PATH_LITERAL(kTestDataFolder))
      .Append(FILE_PATH_LITERAL(GetSanitizedTestName() + ".fake_s3.proto"))
      .MaybeAsASCII();
}

const std::string GetAccessTokenFromEnvironmentOrDie() {
  const char* token = std::getenv("TOKEN");
  CHECK(token && strlen(token))
      << "No token found in the environmental variable $TOKEN.\n"
      << kGenerateTokenInstructions;
  return token;
}

std::string FakeS3ModeToString(FakeS3Mode mode) {
  switch (mode) {
    case FakeS3Mode::kProxy:
      return "PROXY";
    case FakeS3Mode::kRecord:
      return "RECORD";
    case FakeS3Mode::kReplay:
      return "REPLAY";
  }
  NOTREACHED();
}

void AppendArgument(base::CommandLine* command_line,
                    const std::string& name,
                    const std::string& value) {
  // Note we can't use |AppendSwitchASCII| as that will add "<name>=<value>",
  // and the fake s3 server binary does not support '='.
  command_line->AppendArg(name);
  command_line->AppendArg(value);
}

}  // namespace

// Selects a port for the fake S3 server to use.
// This will use a file-based lock because different test shards might be trying
// to run fake S3 servers at the same time, and we need to ensure they use
// different ports.
class PortSelector {
 public:
  PortSelector() { SelectPort(); }
  PortSelector(PortSelector&) = delete;
  PortSelector& operator=(PortSelector&) = delete;
  ~PortSelector() {
    lock_file_.Close();
    base::DeleteFileRecursively(GetLockFilePath());
  }

  int port() const { return port_; }

 private:
  // The first port we'll try to use. Randomly chosen to be outside of the range
  // of known ports.
  constexpr static int kStartPort = 23600;
  // Maximum number of ports we'll try before we give up and conclude no ports
  // are available (which really should not happen).
  constexpr static int kMaxAttempts = 20000;

  void SelectPort() {
    for (int offset = 0; offset < kMaxAttempts; offset++) {
      port_ = kStartPort + offset;
      lock_file_ = base::File(GetLockFilePath(), GetFileFlags());
      if (lock_file_.IsValid())
        return;
    }
    CHECK(false) << "Failed to find an available port.";
  }

  base::FilePath GetLockFilePath() const {
    std::string file_name = "port_" + base::NumberToString(port_) + "_lock";
    return GetLockFileDirectory().Append(file_name);
  }
  static base::FilePath GetLockFileDirectory() {
    base::FilePath result;
    bool success = base::GetTempDir(&result);
    EXPECT_TRUE(success);
    return result;
  }

  static int GetFileFlags() {
    return base::File::FLAG_CREATE | base::File::FLAG_EXCLUSIVE_WRITE |
           base::File::FLAG_WRITE;
  }

  // File exclusively opened on the file-system, to ensure no other fake S3
  // server uses the same port.
  base::File lock_file_;
  int port_;
};

FakeS3Server::FakeS3Server()
    : port_selector_(std::make_unique<PortSelector>()) {}

FakeS3Server::~FakeS3Server() {
  Teardown();
}

void FakeS3Server::Setup(FakeS3Mode mode) {
  SetAccessTokenForMode(mode);
  StartS3ServerProcess(mode);
  SetFakeS3ServerURI();
}

void FakeS3Server::Teardown() {
  StopS3ServerProcess();
  UnsetFakeS3ServerURI();
}

std::string FakeS3Server::GetAccessToken() const {
  return access_token_;
}

void FakeS3Server::SetAccessTokenForMode(FakeS3Mode mode) {
  if (mode == FakeS3Mode::kProxy || mode == FakeS3Mode::kRecord) {
    access_token_ = GetAccessTokenFromEnvironmentOrDie();
  }
}

void FakeS3Server::SetFakeS3ServerURI() {
  // Note this must be stored in a local variable, as
  // |Service::OverrideS3ServerUriForTesting| does not take ownership of the
  // |const char *|.
  fake_s3_server_uri_ = "localhost:" + base::NumberToString(port());
  Service::OverrideS3ServerUriForTesting(fake_s3_server_uri_.c_str());
}

void FakeS3Server::UnsetFakeS3ServerURI() {
  Service::OverrideS3ServerUriForTesting(nullptr);
  fake_s3_server_uri_ = "";
}

void FakeS3Server::StartS3ServerProcess(FakeS3Mode mode) {
  base::FilePath fake_s3_server_main =
      GetExecutableDir().Append(FILE_PATH_LITERAL(kFakeS3ServerBinary));

  base::CommandLine command_line(fake_s3_server_main);
  AppendArgument(&command_line, "--port", base::NumberToString(port()));
  AppendArgument(&command_line, "--mode", FakeS3ModeToString(mode));
  AppendArgument(&command_line, "--auth_token", GetAccessToken());
  AppendArgument(&command_line, "--test_data_file", GetTestDataFileName());

  fake_s3_server_ = base::LaunchProcess(command_line, base::LaunchOptions{});
}

void FakeS3Server::StopS3ServerProcess() {
  fake_s3_server_.Terminate(/*exit_code=*/0, /*wait=*/true);
}

int FakeS3Server::port() const {
  return port_selector_->port();
}

}  // namespace assistant
}  // namespace chromeos
