/********************************************************************************

   Fotoxx - edit photos and manage collections  

   Copyright 2007-2023 Michael Cornelison
   source code URL: https://kornelix.net
   contact: mkornelix@gmail.com

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. See https://www.gnu.org/licenses

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
   See the GNU General Public License for more details.

*********************************************************************************

   Fotoxx window and menu build functions

   build_widgets           build widgets and menus for F/G/W/M view modes
   m_viewmode              set current F/G/W/M view mode
   popup_menufunc          image/thumb right-click menu func
   image_Rclick_popup      popup menu for image right-click
   gallery_Lclick_func     thumbnail left-click function
   gallery_Rclick_popup    popup menu for thumbnail right-click
   m_custom_menu           build custom user menu

*********************************************************************************/

#define EX extern                                                                //  enable extern declarations
#include "fotoxx.h"                                                              //  (variables in fotoxx.h are defined)

/********************************************************************************/

GtkWidget   *mFile, *mGallery, *mMap, *mMeta, *mSelect;
GtkWidget   *mEdit, *mRepair, *mRefine, *mEffects, *mWarp, *mComb;
GtkWidget   *mBatch, *mTools, *mHelp, *mDevmenu;
GtkWidget   *popmenu_image, *popmenu_raw, *popmenu_mpo, *popmenu_video;
GtkWidget   *popmenu_thumb, *popmenu_album;


//  initialize widgets and menus for F/G/W/M view modes
//  called from main() before gtk_main() loop is entered

void build_widgets()
{
   Mwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);                                   //  create main window
   gtk_window_set_title(MWIN,Frelease);

   MWhbox = gtk_box_new(HORIZONTAL,0);                                           //  main window top container
   gtk_container_add(GTK_CONTAINER(Mwin),MWhbox);

   MWmenu = gtk_box_new(VERTICAL,0);                                             //  container for main window menus
   gtk_box_pack_start(GTK_BOX(MWhbox),MWmenu,0,1,0);

   MWvbox = gtk_box_new(VERTICAL,0);                                             //  container for F/G/M/W views
   gtk_box_pack_start(GTK_BOX(MWhbox),MWvbox,1,1,0);

   G_SIGNAL(Mwin,"delete-event",delete_event,0);                                 //  connect signals to main window
   G_SIGNAL(Mwin,"destroy",destroy_event,0);
   G_SIGNAL(Mwin,"window-state-event",state_event,0);
   G_SIGNAL(Mwin,"key-press-event",KBpress,0);                                   //  connect KB events to main window
   G_SIGNAL(Mwin,"key-release-event",KBrelease,0);

   //  F view widgets - image file

   Fhbox = gtk_box_new(HORIZONTAL,0);                                            //  top container
   gtk_box_pack_start(GTK_BOX(MWvbox),Fhbox,1,1,0);
   Fvbox = gtk_box_new(VERTICAL,0);                                              //  vbox for image
   gtk_box_pack_start(GTK_BOX(Fhbox),Fvbox,1,1,0);
   Fpanel = gtk_box_new(HORIZONTAL,0);                                           //  panel over image
   gtk_box_pack_start(GTK_BOX(Fvbox),Fpanel,0,0,0);
   gtk_widget_set_size_request(Fpanel,0,20);
   Fpanlab = gtk_label_new("panel");
   gtk_box_pack_start(GTK_BOX(Fpanel),Fpanlab,0,0,0);
   Fpanelshow = 1;                                                               //  panel normally shows
   Fdrawin = gtk_drawing_area_new();                                             //  image drawing area
   gtk_box_pack_start(GTK_BOX(Fvbox),Fdrawin,1,1,0);
   gtk_widget_hide(Fhbox);

   gtk_widget_add_events(Fdrawin,GDK_BUTTON_PRESS_MASK);                         //  connect mouse events to image window
   gtk_widget_add_events(Fdrawin,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(Fdrawin,GDK_BUTTON_MOTION_MASK);
   gtk_widget_add_events(Fdrawin,GDK_POINTER_MOTION_MASK);
   gtk_widget_add_events(Fdrawin,GDK_SCROLL_MASK);
   G_SIGNAL(Fdrawin,"button-press-event",mouse_event,0);                         //  connect signals
   G_SIGNAL(Fdrawin,"button-release-event",mouse_event,0);
   G_SIGNAL(Fdrawin,"motion-notify-event",mouse_event,0);
   G_SIGNAL(Fdrawin,"scroll-event",mouse_event,0);
   G_SIGNAL(Fdrawin,"draw",Fpaint,0);
   drag_drop_dest(Fdrawin,drop_event);                                           //  accept drag-drop file

   //  G view widgets - thumbnail gallery

   Ghbox = gtk_box_new(HORIZONTAL,0);                                            //  top container
   gtk_box_pack_start(GTK_BOX(MWvbox),Ghbox,1,1,0);
   Gvbox = gtk_box_new(VERTICAL,0);                                              //  vbox for gallery
   gtk_box_pack_start(GTK_BOX(Ghbox),Gvbox,1,1,0);
   Gpanel = gtk_box_new(HORIZONTAL,0);                                           //  top panel for [TOP] and navi buttons
   gtk_box_pack_start(GTK_BOX(Gvbox),Gpanel,0,0,2);
   Galbum = gtk_button_new_with_label("Album");                                  //  [Album] button in panel
   gtk_box_pack_start(GTK_BOX(Gpanel),Galbum,0,0,3);
   Gtop = gtk_button_new_with_label("TOP");                                      //  [TOP] button in panel
   gtk_box_pack_start(GTK_BOX(Gpanel),Gtop,0,0,3);
   Gsep = gtk_label_new(0);
   gtk_label_set_markup(GTK_LABEL(Gsep),"<span font=\"sans bold 12\" >@</span>");
   gtk_box_pack_start(GTK_BOX(Gpanel),Gsep,0,0,10);

   Gsep = gtk_separator_new(HORIZONTAL);                                         //  separator line
   gtk_box_pack_start(GTK_BOX(Gvbox),Gsep,0,0,3);
   Gscroll = gtk_scrolled_window_new(0,0);                                       //  scrolled window for gallery
   gtk_scrolled_window_set_policy(SCROLLWIN(Gscroll),NEVER,ALWAYS);
   Gadjust = gtk_scrolled_window_get_vadjustment(SCROLLWIN(Gscroll));
   gtk_box_pack_start(GTK_BOX(Gvbox),Gscroll,1,1,0);
   Gdrawin = gtk_drawing_area_new();                                             //  gallery drawing area
   gtk_container_add(GTK_CONTAINER(Gscroll),Gdrawin);
   gtk_widget_hide(Ghbox);

   gtk_widget_add_events(Gdrawin,GDK_BUTTON_PRESS_MASK);                         //  connect mouse events to gallery window
   gtk_widget_add_events(Gdrawin,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(Gdrawin,GDK_POINTER_MOTION_MASK);
   G_SIGNAL(Gtop,"clicked",navi::newtop,0);
   G_SIGNAL(Galbum,"clicked",navi::newalbum,0);
   G_SIGNAL(Gdrawin,"button-press-event",navi::mouse_event,0);
   G_SIGNAL(Gdrawin,"button-release-event",navi::mouse_event,0);
   G_SIGNAL(Gdrawin,"motion-notify-event",navi::mouse_event,0);
   G_SIGNAL(Gdrawin,"draw",navi::gallery_paint,null);
   drag_drop_source(Gdrawin,navi::gallery_dragfile);                             //  start file drag-drop
   drag_drop_dest(Gdrawin,navi::gallery_dropfile);                               //  accept drag-drop file

   //  M view widgets - internet maps

   Mhbox = gtk_box_new(HORIZONTAL,0);                                            //  top container
   gtk_box_pack_start(GTK_BOX(MWvbox),Mhbox,1,1,0);
   Mvbox = gtk_box_new(VERTICAL,0);                                              //  vbox for net map window
   gtk_box_pack_start(GTK_BOX(Mhbox),Mvbox,1,1,0);
   gtk_widget_hide(Mhbox);

   //  W view widgets - local map files

   Whbox = gtk_box_new(HORIZONTAL,0);                                            //  top container
   gtk_box_pack_start(GTK_BOX(MWvbox),Whbox,1,1,0);
   Wvbox = gtk_box_new(VERTICAL,0);                                              //  vbox for file map window
   gtk_box_pack_start(GTK_BOX(Whbox),Wvbox,1,1,0);
   Wdrawin = gtk_drawing_area_new();                                             //  filemap drawing area
   gtk_box_pack_start(GTK_BOX(Wvbox),Wdrawin,1,1,0);
   gtk_widget_hide(Whbox);

   gtk_widget_add_events(Wdrawin,GDK_BUTTON_PRESS_MASK);                         //  connect mouse events to filemap window
   gtk_widget_add_events(Wdrawin,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(Wdrawin,GDK_BUTTON_MOTION_MASK);
   gtk_widget_add_events(Wdrawin,GDK_POINTER_MOTION_MASK);
   gtk_widget_add_events(Wdrawin,GDK_SCROLL_MASK);
   G_SIGNAL(Wdrawin,"button-press-event",mouse_event,0);                         //  connect signals
   G_SIGNAL(Wdrawin,"button-release-event",mouse_event,0);
   G_SIGNAL(Wdrawin,"motion-notify-event",mouse_event,0);
   G_SIGNAL(Wdrawin,"scroll-event",mouse_event,0);
   G_SIGNAL(Wdrawin,"draw",Fpaint,0);

   //  menu popup text (tool tips) ----------------------------------------
   
   //  main menu buttons
   ch * File_tip = "File: new session, open, rename, copy/move, delete, print";
   ch * Gallery_tip = "Gallery: thumbnails, new/recent, sort, select, bookmarks, albums, slide show";
   ch * Maps_tip = "Maps: net maps, file maps, images by map location";
   ch * custom_menu_tip = "Custom menu: add favorite or frequent functions here";
   ch * prev_next_tip = "Left/right click: previous/next image (also ←/→ keys)";
   ch * zoom_menu_tip = "Left/right click: zoom image/thumbnail in/out (also +/- keys)";
   ch * save_tip = "Save modified image as new file or new file version";
   ch * metadata_tip = "View/edit metadata, manage tags, images by location/date, search images";
   ch * select_tip = "Select image areas to edit separately, copy/paste selections";
   ch * undo_redo_tip = "Left/right click: undo/redo one edit step \n"
                           "  (with A-key: undo/redo all edits) \n"
                           " Middle click: go back to selected edit step";
   ch * edit_tip = "Edit: crop, rotate, retouch, rescale, markup, paint edits, copy areas, plugins";
   ch * repair_tip = "Repair: sharpen, blur, denoise, defog, red eyes, smart erase, adjust color";
   ch * refine_tip = "Refine: edit dist, flatten dist, local contrast, gradients, retinex, saturation, vignette";
   ch * effects_tip = "Effects: cartoon, sketch, paint, emboss, dither, add texture, change colors, custom convolution";
   ch * warp_tip = "Warp: unbend, fix perspective, warp, unwarp closeup, flatten, image transforms";
   ch * combine_tip = "Combine: HDR, HDF, stack, panorama, mashup, image array";
   ch * batch_tip = "Batch: copy, move, convert, RAW convert, add/change metadata, custom scripts";
   ch * tools_tip = "Tools: image index, settings, shortcuts, magnify, find duplicates, grid lines";
   ch * help_tip = "Help: user guide, tutorial, change log, licenses, privacy, about";

   //  file menu
   ch * new_session_tip = "Open a parallel Fotoxx session";
   ch * open_file_tip = "Select and view an image file";
   ch * cycle2files_tip = "Cycle 2 Prior Files";
   ch * cycle3files_tip = "Cycle 3 Prior Files";
   ch * view360_tip = "View 360 degree panorama image";
   ch * rename_tip = "Change file name";
   ch * permissions_tip = "View and change file permissions";
   ch * change_alpha_tip = "Add/Remove image file alpha channel (transparency)";
   ch * blank_image_tip = "Create a blank image";
   ch * copy_move_tip = "Copy or Move image file to new location";
   ch * copyto_desktop_tip = "Copy image file to the desktop";
   ch * copyto_clipboard_tip = "Copy image file to the clipboard";
   ch * deltrash_tip = "Delete or trash an image file";
   ch * convert_adobe_tip = "Convert Adobe document file into jpeg file(s)";
   ch * set_wallpaper_tip = "Set image file as desktop wallpaper (GNOME only)";
   ch * show_on_net_map_tip = "Show image location on Internet map";
   ch * print_tip = "Print the current image file";
   ch * print_calibrated_tip = "Print current image file with adjusted colors";
   ch * quit_tip = "Quit Fotoxx";

   //  gallery menu
   ch * thumbview_tip = "Gallery view with thumbnails and file data";
   ch * metaview_tip = "Gallery view with thumbnails and basic metadata";
   ch * recentfiles_tip = "Gallery of recently viewed image files";
   ch * newfiles_tip = "Gallery of newest image files";
   ch * gallery_sort_tip = "Change gallery sort order";
   ch * gallery_screen_tip = "Select newest, oldest, matching pattern ...";
   ch * gallery_home_tip = "Jump to gallery beginning [home]";
   ch * gallery_end_tip = "Jump to gallery end [end]";
   ch * image_folder_tip = "Get gallery (folder) of current image file";
   ch * folder_tree_tip = "Show folder tree, click any branch for gallery view";
   ch * select_files_tip = "Select input files for subsequent function";
   ch * rename_folder_tip = "Rename current folder (current gallery)";
   ch * add_subfolder_tip = "Add a new subfolder to current gallery";
   ch * bookmarks_tip = "Set and recall bookmarked image locations";
   ch * manage_albums_tip = "Organize images into albums";
   ch * update_albums_tip = "Update albums for new file versions";
   ch * album_mass_update_tip = "Mass update album files";
   ch * gallery2album_tip = "Save current gallery as album";
   ch * slideshow_tip = "Start a slide show";
   
   //  map menu
   ch * net_map_tip = "Open Internet map";
   ch * net_locs_tip = "Go to internet map location, save location";
   ch * download_maps_tip = "Download fle maps from Fotoxx web site";
   ch * file_map_tip = "Open file map";
   ch * choose_map_tip = "Set current file map";
   ch * markers_tip = "Show map markers for all images or current gallery";

   //  metadata menu
   ch * meta_view_main_tip = "Show key metadata for current image or clicked thumbnail";
   ch * meta_view_all_tip = "List all metadata for current image or clicked thumbnail";
   ch * meta_edit_main_tip = "Edit key metadata: tags/geotags/caption/rating ...";
   ch * meta_edit_any_tip = "Edit any metadata item";
   ch * meta_delete_tip = "Remove selected image metadata";
   ch * meta_copy_tip = "Copy metadata from one image to another";
   ch * meta_fix_tip = "Repair malformed metadata preventing metadata edits";
   ch * meta_manage_tags_tip = "Define tags (keywords) used for searching images";
   ch * meta_choose_caps_tip = "Choose metadata shown on image (captions etc.)";
   ch * meta_toggle_caps_tip = "Toggle image metadata display ON/OFF";
   ch * meta_places_dates_tip = "Find all images for a location [date range]";
   ch * meta_timeline_tip = "Show image counts in year/month calendar, select, report";
   ch * search_images_tip = "Find images meeting multiple search criteria";

   //  select area menu
   ch * select_area_tip = "Select objects or areas for separate editing";
   ch * select_find_gap_tip = "Find a gap in an area outline";
   ch * select_blend_tip = "Blend area edits using the mouse";
   ch * select_show_tip = "Show (outline) selected areas";
   ch * select_hide_tip = "Hide selected areas";
   ch * select_enable_tip = "Enable areas for editing";
   ch * select_disable_tip = "Disable areas for editing";
   ch * select_invert_tip = "Invert selected areas (selected ↔ not selected)";
   ch * select_clear_tip = "Erase existing areas";
   ch * select_copy_tip = "Copy area for later pasting into image";
   ch * select_paste_tip = "Paste previously copied area into image";
   ch * select_load_tip = "Open a file and paste as selected area into image";
   ch * select_save_tip = "Save selected area to a file with transparency";

   //  edit menu
   ch * rotate_tip = "Rotate or level image, upright, mirror";
   ch * upright_tip = "Upright a rotated image with one button";
   ch * crop_tip = "Crop (trim) image margins";
   ch * retouch_tip = "Adjust brightness, contrast, color";
   ch * rescale_tip = "Change image pixel dimensions";
   ch * margins_tip = "Add margins around an image";
   ch * markup_tip = "Draw on image: text, line/arrow, box, circle, ellipse";
   ch * color_mode_tip = "Set black-white/color, negative/positive, sepia";
   ch * paint_image_tip = "Paint on image using the mouse";
   ch * paint_edits_tip = "Paint current edit function gradually using the mouse";
   ch * paint_transp_tip = "Paint image transparency using the mouse";
   ch * copy_in_image_tip = "Copy pixels within an image using the mouse";
   ch * copy_from_image_tip = "Copy pixels from another image using the mouse";
   ch * copy_prior_edit_tip = "Copy pixels from prior edits using the mouse";
   ch * plugins_tip = "Edit plugins menu or run a plugin function";

   //  repair menu
   ch * sharpen_tip = "Sharpen the image (various methods)";
   ch * blur_tip = "Blur the image (various methods)";
   ch * denoise_tip = "Suppress noise from low-light photos";
   ch * defog_tip = "Reduce fog/haze in an image or selected area";
   ch * redeyes_tip = "Fix red-eyes from flash photo";
   ch * smart_erase_tip = "Remove unwanted objects in an image";
   ch * remove_halo_tip = "Remove halos left by other edit functions";
   ch * jpeg_artifacts_tip = "Suppress JPEG compression artifacts";
   ch * anti_alias_tip = "Remove jaggies on image feature hard edges";
   ch * adjust_RGB_tip = "Adjust color using RGB or CMY colors";
   ch * adjust_HSL_tip = "Adjust color using HSL colors";
   ch * color_profile_tip = "Convert image to another color profile";
   ch * remove_dust_tip = "Remove dust spots from old scanned photos";
   ch * chromatic1_tip = "Fix color fringes in outer areas of an image";
   ch * chromatic2_tip = "Fix color fringes on dark/bright feature edges";

   //  refine menu
   ch * voodoo1_tip = "1-click auto improve that may be good enough";
   ch * voodoo2_tip = "1-click auto improve that may be good enough";
   ch * edit_dist_tip = "Edit brightness distribution directly";
   ch * flatdist_tip = "Flatten brightness distribution to enhance details";
   ch * localcon_tip = "Rescale RGB levels to increase local contrast";
   ch * gradients_tip = "Magnify brightness gradients to enhance details";
   ch * retinex_tip = "Rescale RGB - reduce color caste and fog/haze";
   ch * saturation_tip = "Adjust color saturation based on brightness";
   ch * soft_focus_tip = "Apply a soft focus effect to an image";
   ch * match_colors_tip = "Match image colors with those of another image";
   ch * brite_ramp_tip = "Add a brightness/color ramp across an image";
   ch * vignette_tip = "Add/remove brightness/color radially from chosen center";

   //  effects menu
   ch * sketch_tip = "Convert image into a simulated sketch";
   ch * cartoon_tip = "Convert image into a cartoon drawing";
   ch * drawing_tip = "Convert image into a line drawing (edge detection)";
   ch * emboss_tip = "Create an embossed or 3D appearance";
   ch * tiles1_tip = "Convert image into square tiles with 3D effect";
   ch * tiles2_tip = "Convert image into irregular tiles matching image";
   ch * painting_tip = "Convert image into a simulated painting";
   ch * texture_tip = "Add texture to an image";
   ch * pattern_tip = "Tile image with a repeating pattern";
   ch * dither_tip = "Convert image into dithered dots";
   ch * engrave_tip = "Convet image into black/white lines";
   ch * mosaic_tip = "Create a mosaic with tiles made from all images";
   ch * add_noise_tip = "Add random noise to an image";
   ch * color_depth_tip = "Reduce image color depth (posterize)";
   ch * shift_colors_tip = "Shift/convert image colors into other colors";
   ch * alien_colors_tip = "Change image colors using an algorithm";
   ch * copy_colors_tip = "Copy selected RGB colors from current image";
   ch * paste_colors_tip = "Paste selected RGB colors from selected image file";
   ch * anykernel_tip = "Process an image using a custom convolution kernel";

   //  warp menu
   ch * unbend_tip = "Remove image curvature, especially in panoramas";
   ch * perspective_tip = "Fix perspective for object photographed from an angle";
   ch * warp_area_tip = "Distort image areas using the mouse";
   ch * warp_curved_tip = "Warp image (curved) using the mouse";
   ch * warp_linear_tip = "Warp image (straight) using the mouse";
   ch * warp_affine_tip = "Warp image (affine) using the mouse";
   ch * area_rescale_tip = "Rescale selected image area";
   ch * unwarp_closeup_tip = "Rectify closeup face photo (baloon face)";
   ch * waves_tip = "Warp an image with a wave pattern";
   ch * twist_tip = "Twist image centered at mouse position";
   ch * sphere_tip = "Make a spherical projection of an image";
   ch * stretch_tip = "Add/remove barrel/pincushion distortion";
   ch * inside_out_tip = "Turn an image inside-out";
   ch * tiny_planet_tip = "Convert an image into a Tiny Planet";
   ch * escher_spiral_tip = "Generate an inward spiraling repeating image";
   ch * flatphoto_tip = "Flatten photo of a curved image";

   //  combine menu
   ch * HDR_tip = "Combine bright/dark images for better detail";
   ch * HDF_tip = "Combine near/far focus images for deeper focus";
   ch * stack_paint_tip = "Combine multiple photos to erase passing people, etc.";
   ch * stack_noise_tip = "Combine multiple noisy photos into a low-noise image";
   ch * stack_versions_tip = "Combine image versions, mouse paint areas/versions";
   ch * stack_split_tip = "Compare two images split by sliding boundary";
   ch * panorama_tip = "Combine images into a panorama";
   ch * show_cim_files_tip = "Show input files for above combine functions";
   ch * mashup_tip = "Arrange images and text in a layout (montage)";
   ch * image_array_tip = "Combine images into a array or matrix of images";

   //  batch menu
   ch * batch_convert_tip = "Rename/convert/rescale/move selected image files";
   ch * batch_copy_move_tip = "Copy/move selected image files to a new folder";
   ch * batch_upright_tip = "Upright rotated image files, selected or ALL";
   ch * batch_deltrash_tip = "Delete or Trash selected image files";
   ch * batch_RAW_tip = "Convert camera RAW files to tiff/png/jpeg";
   ch * batch_overlay_tip = "Add overlay image (copyright ...) to selected images";
   ch * batch_tags_tip = "Add/remove tags for selected image files";
   ch * batch_rename_tags_tip = "Revise tag names for all image files";
   ch * batch_photo_DT_tip = "Set or shift photo date/time for selected image files";
   ch * batch_change_meta_tip = "Add/change/delete metadata for selected image files";
   ch * batch_report_meta_tip = "Report metadata for selected image files";
   ch * batch_meta_mover_tip = "Move metadata between selected keys";
   ch * batch_geotags_tip = "Add/revise geotags for selected image files";
   ch * export_filelist_tip = "Create a file of selected image files";
   ch * export_files_tip = "Export selected image files to a folder";
   ch * edit_script_tip = "Build a custom script with multiple edit functions";
   ch * run_script_tip = "Run custom script to edit the current image file";
   ch * batch_script_tip = "Run custom script to edit selected image files";

   //  tools menu
   ch * index_tip = "Index new image files and make thumbnails";
   ch * quick_index_tip = "Quick incremental index update";
   ch * settings_tip = "User preferences and settings";
   ch * KB_shortcuts_tip = "Show or make Keyboard Shortcuts";
   ch * RGB_dist_tip = "Show RGB brightness distribution";
   ch * magnify_tip = "Magnify image around the mouse position";
   ch * duplicates_tip = "Search all image files and report duplicates";
   ch * measure_image_tip = "Measure distances within an image";
   ch * show_RGB_tip = "Show RGB colors at mouse position";
   ch * popup_text_tip = "Show text strings for edit insertion";
   ch * calib_printer_tip = "Calibrate printer colors";
   ch * grid_settings_tip = "Set grid line counts or spacing";
   ch * toggle_grid_tip = "Toggle grid lines on and off";
   ch * line_color_tip = "Set foreground line color (crop, grid, select, warp)";
   ch * darkbrite_tip = "Highlight darkest and brightest pixels";
   ch * monitor_color_tip = "Chart to adjust monitor color";
   ch * resources_tip = "Show memory and CPU resources used";

   //  help menu
   ch * user_guide_tip = "Read or search the user guide";
   ch * video_tutorial_tip = "Fotoxx video tutorial on YouTube";
   ch * all_edit_funcs_tip = "Overview of all Fotoxx edit functions";
   ch * outboard_programs_tip = "List outboard programs and functions";
   ch * logfile_tip = "View the log file (info and error messages)";
   ch * command_params_tip = "List Fotoxx command line parameters";
   ch * changelog_tip = "Show Fotoxx changes for recent releases";
   ch * copyright_tip = "Fotoxx copyright notice";
   ch * licenses_tip = "Licenses used by Fotoxx: GNU and Pareto";
   ch * source_build_tip = "How to build Fotoxx from source";
   ch * privacy_tip = "Fotoxx privacy policy";
   ch * about_tip = "Show Fotoxx installed version and author contact";
   ch * homepage_tip = "Fotoxx web site (information and downloads)";
   ch * uninstall_tip = "Remove Fotoxx application and data files"; 


   //  build menu table ---------------------------------------------------------

   #define MENU(_topmenu, _text, _icon, _tip, _func, _arg)        \
      me = Nmenus++;                                              \
      if (me >= maxmenus) zappcrash("maxmenus exceeded");         \
      menutab[me].topmenu = _topmenu;                             \
      menutab[me].menu = _text;                                   \
      menutab[me].icon = _icon;                                   \
      menutab[me].desc = _tip;                                    \
      menutab[me].func = _func;                                   \
      if (_arg) menutab[me].arg = _arg;                           \
      else menutab[me].arg = _text;                               \

   int      me;
   Nmenus = 0;

   mFile = create_popmenu();
   MENU(mFile,    "New Session", 0,           new_session_tip,              m_new_session, 0 );
   MENU(mFile,    "Open Image File", 0,       open_file_tip,                m_open_file, 0 );
   MENU(mFile,    "Image Folder", 0,          image_folder_tip,             m_image_folder, 0 );
   MENU(mFile,    "Cycle 2", 0,               cycle2files_tip,              m_cycle2files, 0 );
   MENU(mFile,    "Cycle 3", 0,               cycle3files_tip,              m_cycle3files, 0 );
   MENU(mFile,    "View 360° Pano", 0,        view360_tip,                  m_view360, 0);
   MENU(mFile,    "Rename", 0,                rename_tip,                   m_rename, 0 );
   MENU(mFile,    "Permissions", 0,           permissions_tip,              m_permissions, 0 );
   MENU(mFile,    "Change Alpha", 0,          change_alpha_tip,             m_change_alpha, 0 );
   MENU(mFile,    "Blank Image", 0,           blank_image_tip,              m_blank_image, 0 );
   MENU(mFile,    "Copy/Move", 0,             copy_move_tip,                m_copy_move, 0 );
   MENU(mFile,    "Copy to Desktop", 0,       copyto_desktop_tip,           m_copyto_desktop, 0 );
   MENU(mFile,    "Copy to Clipboard", 0,     copyto_clipboard_tip,         m_copyto_clip, 0 );
   MENU(mFile,    "Delete/Trash", 0,          deltrash_tip,                 m_delete_trash, 0 );
   MENU(mFile,    "Convert Adobe", 0,         convert_adobe_tip,            m_convert_adobe, 0 );
   MENU(mFile,    "Set Wallpaper", 0,         set_wallpaper_tip,            m_wallpaper, 0 );
   MENU(mFile,    "Show on Map", 0,           show_on_net_map_tip,          m_netmap_zoomin, 0 );
   MENU(mFile,    "Print", 0,                 print_tip,                    m_print, 0 );
   MENU(mFile,    "Print Calibrated", 0,      print_calibrated_tip,         m_print_calibrated, 0 );
   MENU(mFile,    "Quit", 0,                  quit_tip,                     m_quit, 0 );

   mGallery = create_popmenu();
   MENU(mGallery,   "Thumb View", 0,          thumbview_tip,                m_thumbview, 0 );
   MENU(mGallery,   "Meta View", 0,           metaview_tip,                 m_metaview, 0 );
   MENU(mGallery,   "Recent Files", 0,        recentfiles_tip,              m_recentfiles, 0 );
   MENU(mGallery,   "Newest Files", 0,        newfiles_tip,                 m_newfiles, 0 );
   MENU(mGallery,   "Gallery Sort", 0,        gallery_sort_tip,             m_gallery_sort, 0);
   MENU(mGallery,   "Gallery Screen", 0,      gallery_screen_tip,           m_gallery_screen, 0);
   MENU(mGallery,   "Gallery Home", 0,        gallery_home_tip,             navi::menufuncx, "Home" );
   MENU(mGallery,   "Gallery End", 0,         gallery_end_tip,              navi::menufuncx, "End" );
   MENU(mGallery,   "Image Folder", 0,        image_folder_tip,             m_image_folder, 0 );
   MENU(mGallery,   "Folder Tree", 0,         folder_tree_tip,              m_folder_tree, 0 );
   MENU(mGallery,   "Select Files", 0,        select_files_tip,             m_select_files, 0 );
   MENU(mGallery,   "Rename Folder", 0,       rename_folder_tip,            m_rename_folder, 0 );
   MENU(mGallery,   "Add Subfolder", 0,       add_subfolder_tip,            m_add_subfolder, 0 );
   MENU(mGallery,   "Bookmarks", 0,           bookmarks_tip,                m_bookmarks, 0 );
   MENU(mGallery,   "Manage Albums", 0,       manage_albums_tip,            m_manage_albums, 0 );
   MENU(mGallery,   "Update Albums", 0,       update_albums_tip,            m_update_albums, 0 );
   MENU(mGallery,   "Album Mass Update", 0,   album_mass_update_tip,        m_album_mass_update, 0 );
   MENU(mGallery,   "Gallery to Album", 0,    gallery2album_tip,            m_gallery2album, 0 );
   MENU(mGallery,   "Slide Show", 0,          slideshow_tip,                m_slideshow, 0 );

   mMap = create_popmenu();
   MENU(mMap,     "Net Map", 0,               net_map_tip,                  m_viewmode, "M");
   MENU(mMap,     "Net Map Locs", 0,          net_locs_tip,                 m_netmap_locs, 0 );
   MENU(mMap,     "Download Maps", 0,         download_maps_tip,            m_download_maps, 0 ); 
   MENU(mMap,     "File Map", 0,              file_map_tip,                 m_load_filemap, "load map");
   MENU(mMap,     "Choose Map", 0,            choose_map_tip,               m_load_filemap, "choose map" );
   MENU(mMap,     "Markers", 0,               markers_tip,                  m_set_map_markers, 0 );

   mMeta = create_popmenu();
   MENU(mMeta,    "View Meta", 0,             meta_view_main_tip,           m_meta_view_short, 0 );
   MENU(mMeta,    "View All Meta", 0,         meta_view_all_tip,            m_meta_view_long, 0 );
   MENU(mMeta,    "Edit Meta", 0,             meta_edit_main_tip,           m_meta_edit_main, 0 );
   MENU(mMeta,    "Edit Any Meta", 0,         meta_edit_any_tip,            m_meta_edit_any, 0 );
   MENU(mMeta,    "Delete Meta", 0,           meta_delete_tip,              m_meta_delete, 0 );
   MENU(mMeta,    "Copy Meta", 0,             meta_copy_tip,                m_meta_copy, 0 );
   MENU(mMeta,    "Fix Meta", 0,              meta_fix_tip,                 m_meta_fix, 0 );
   MENU(mMeta,    "Manage Tags", 0,           meta_manage_tags_tip,         m_meta_manage_tags, 0 );
   MENU(mMeta,    "Choose Captions", 0,       meta_choose_caps_tip,         m_meta_choose_caps, 0 );
   MENU(mMeta,    "Toggle Captions", 0,       meta_toggle_caps_tip,         m_meta_toggle_caps, 0 );
   MENU(mMeta,    "Places/Dates", 0,          meta_places_dates_tip,        m_meta_places_dates, 0 );
   MENU(mMeta,    "Timeline", 0,              meta_timeline_tip,            m_meta_timeline, 0 );
   MENU(mMeta,    "Search", 0,                search_images_tip,            m_search_images, 0 );

   mSelect = create_popmenu();
   MENU(mSelect,     "Select Area", 0,        select_area_tip,              m_select_area, 0 );
   MENU(mSelect,     "Find Gap", 0,           select_find_gap_tip,          m_select_find_gap, 0 );
   MENU(mSelect,     "Area Blend", 0,         select_blend_tip,             m_select_blend, 0 );
   MENU(mSelect,     "Show", 0,               select_show_tip,              m_select_show, 0 );
   MENU(mSelect,     "Hide", 0,               select_hide_tip,              m_select_hide, 0 );
   MENU(mSelect,     "Enable", 0,             select_enable_tip,            m_select_enable, 0 );
   MENU(mSelect,     "Disable", 0,            select_disable_tip,           m_select_disable, 0 );
   MENU(mSelect,     "Invert", 0,             select_invert_tip,            m_select_invert, 0 );
   MENU(mSelect,     "Clear", 0,              select_clear_tip,             m_select_clear, 0 );
   MENU(mSelect,     "Copy", 0,               select_copy_tip,              m_select_copy, 0 );
   MENU(mSelect,     "Paste", 0,              select_paste_tip,             m_select_paste, 0 );
   MENU(mSelect,     "Load", 0,               select_load_tip,              m_select_load, 0 );
   MENU(mSelect,     "Save", 0,               select_save_tip,              m_select_save, 0 );

   mEdit = create_popmenu(); 
   MENU(mEdit,    "Rotate", 0,                rotate_tip,                   m_rotate, 0 );
   MENU(mEdit,    "Upright", 0,               upright_tip,                  m_upright, 0 );
   MENU(mEdit,    "Crop", 0,                  crop_tip,                     m_crop, 0 );
   MENU(mEdit,    "Retouch", 0,               retouch_tip,                  m_retouch, 0 );
   MENU(mEdit,    "Rescale", 0,               rescale_tip,                  m_rescale, 0 );
   MENU(mEdit,    "Margins", 0,               margins_tip,                  m_margins, 0 );
   MENU(mEdit,    "Markup", 0,                markup_tip,                   m_markup, 0 );
   MENU(mEdit,    "Color Mode", 0,            color_mode_tip,               m_color_mode, 0 );
   MENU(mEdit,    "Paint Image", 0,           paint_image_tip,              m_paint_image, 0 );
   MENU(mEdit,    "Paint Edits", 0,           paint_edits_tip,              m_paint_edits, 0 );
   MENU(mEdit,    "Paint Transp", 0,          paint_transp_tip,             m_paint_transp, 0 );
   MENU(mEdit,    "Copy in Image", 0,         copy_in_image_tip,            m_copy_in_image, 0 );
   MENU(mEdit,    "Copy From Image", 0,       copy_from_image_tip,          m_copy_from_image, 0 );
   MENU(mEdit,    "Copy Prior Edit", 0,       copy_prior_edit_tip,          m_copy_prior_edit, 0 );
   MENU(mEdit,    "Plugins", 0,               plugins_tip,                  m_plugins, 0);

   mRepair = create_popmenu();
   MENU(mRepair,     "Sharpen", 0,            sharpen_tip,                  m_sharpen, 0 );
   MENU(mRepair,     "Blur", 0,               blur_tip,                     m_blur, 0 );
   MENU(mRepair,     "Denoise", 0,            denoise_tip,                  m_denoise, 0 );
   MENU(mRepair,     "Defog", 0,              defog_tip,                    m_defog, 0 );
   MENU(mRepair,     "Red Eyes", 0,           redeyes_tip,                  m_redeyes, 0 );
   MENU(mRepair,     "Smart Erase", 0,        smart_erase_tip,              m_smart_erase, 0 );
   MENU(mRepair,     "Remove Halo", 0,        remove_halo_tip,              m_remove_halo, 0 );
   MENU(mRepair,     "JPEG Artifacts", 0,     jpeg_artifacts_tip,           m_jpeg_artifacts, 0 );
   MENU(mRepair,     "Anti-Alias", 0,         anti_alias_tip,               m_anti_alias, 0 );
   MENU(mRepair,     "Adjust RGB", 0,         adjust_RGB_tip,               m_adjust_RGB, 0 );
   MENU(mRepair,     "Adjust HSL", 0,         adjust_HSL_tip,               m_adjust_HSL, 0 );
   MENU(mRepair,     "Color Profile", 0,      color_profile_tip,            m_color_profile, 0 );
   MENU(mRepair,     "Remove Dust", 0,        remove_dust_tip,              m_remove_dust, 0 );
   MENU(mRepair,     "Chromatic 1", 0,        chromatic1_tip,               m_chromatic1, 0 );
   MENU(mRepair,     "Chromatic 2", 0,        chromatic2_tip,               m_chromatic2, 0 );

   mRefine = create_popmenu();
   MENU(mRefine,     "Voodoo 1", 0,              voodoo1_tip,               m_voodoo1, 0 );
   MENU(mRefine,     "Voodoo 2", 0,              voodoo2_tip,               m_voodoo2, 0);
   MENU(mRefine,     "Edit Dist", 0,             edit_dist_tip,             m_edit_dist, 0 );
   MENU(mRefine,     "Flatten Dist", 0,          flatdist_tip,              m_flatdist, 0 );
   MENU(mRefine,     "Local Contrast", 0,        localcon_tip,              m_localcon, 0 );
   MENU(mRefine,     "Gradients", 0,             gradients_tip,             m_gradients, 0 );
   MENU(mRefine,     "Global Retinex", 0,        retinex_tip,               m_gretinex, 0 );
   MENU(mRefine,     "Local Retinex", 0,         retinex_tip,               m_lretinex, 0 );
   MENU(mRefine,     "Saturation", 0,            saturation_tip,            m_saturation, 0 );
   MENU(mRefine,     "Soft Focus", 0,            soft_focus_tip,            m_soft_focus, 0 );
   MENU(mRefine,     "Match Colors", 0,          match_colors_tip,          m_match_colors, 0 );
   MENU(mRefine,     "Brite Ramp", 0,            brite_ramp_tip,            m_brite_ramp, 0 );
   MENU(mRefine,     "Vignette", 0,              vignette_tip,              m_vignette, 0 );

   mEffects = create_popmenu();
   MENU(mEffects,    "Sketch", 0,                sketch_tip,                m_sketch, 0 );
   MENU(mEffects,    "Cartoon", 0,               cartoon_tip,               m_cartoon, 0 );
   MENU(mEffects,    "Drawing", 0,               drawing_tip,               m_drawing, 0 );
   MENU(mEffects,    "Emboss", 0,                emboss_tip,                m_emboss, 0 );
   MENU(mEffects,    "Tiles1", 0,                tiles1_tip,                m_tiles1, 0 );
   MENU(mEffects,    "Tiles2", 0,                tiles2_tip,                m_tiles2, 0 );
   MENU(mEffects,    "Painting", 0,              painting_tip,              m_painting, 0 );
   MENU(mEffects,    "Texture", 0,               texture_tip,               m_texture, 0 );
   MENU(mEffects,    "Pattern", 0,               pattern_tip,               m_pattern, 0 );
   MENU(mEffects,    "Dither", 0,                dither_tip,                m_dither, 0 );
   MENU(mEffects,    "Engrave", 0,               engrave_tip,               m_engrave, 0 );
   MENU(mEffects,    "Mosaic", 0,                mosaic_tip,                m_mosaic, 0 );
   MENU(mEffects,    "Add Noise", 0,             add_noise_tip,             m_add_noise, 0 );
   MENU(mEffects,    "Color Depth", 0,           color_depth_tip,           m_color_depth, 0 );
   MENU(mEffects,    "Shift Colors", 0,          shift_colors_tip,          m_shift_colors, 0 );
   MENU(mEffects,    "Alien Colors", 0,          alien_colors_tip,          m_alien_colors, 0 ); 
   MENU(mEffects,    "Copy Colors", 0,           copy_colors_tip,           m_copy_colors, 0 ); 
   MENU(mEffects,    "Paste Colors", 0,          paste_colors_tip,          m_paste_colors, 0 ); 
   MENU(mEffects,    "Custom Kernel", 0,         anykernel_tip,             m_anykernel, 0 );

   mWarp = create_popmenu();
   MENU(mWarp,    "Unbend", 0,                   unbend_tip,                m_unbend, 0 );
   MENU(mWarp,    "Perspective", 0,              perspective_tip,           m_perspective, 0 );
   MENU(mWarp,    "Warp Area", 0,                warp_area_tip,             m_warp_area, 0 );
   MENU(mWarp,    "Warp Curved", 0,              warp_curved_tip,           m_warp_curved, 0 );
   MENU(mWarp,    "Warp Linear", 0,              warp_linear_tip,           m_warp_linear, 0 );
   MENU(mWarp,    "Warp Affine", 0,              warp_affine_tip,           m_warp_affine, 0 );
   MENU(mWarp,    "Area Rescale", 0,             area_rescale_tip,          m_area_rescale, 0);
   MENU(mWarp,    "Unwarp Closeup", 0,           unwarp_closeup_tip,        m_unwarp_closeup, 0 );
   MENU(mWarp,    "Make Waves", 0,               waves_tip,                 m_waves, 0);
   MENU(mWarp,    "Twist", 0,                    twist_tip,                 m_twist, 0);
   MENU(mWarp,    "Sphere", 0,                   sphere_tip,                m_sphere, 0);
   MENU(mWarp,    "Stretch", 0,                  stretch_tip,               m_stretch, 0);
   MENU(mWarp,    "Inside-out", 0,               inside_out_tip,            m_inside_out, 0);
   MENU(mWarp,    "Tiny Planet", 0,              tiny_planet_tip,           m_tiny_planet, 0);
   MENU(mWarp,    "Escher Spiral", 0,            escher_spiral_tip,         m_escher_spiral, 0);
   MENU(mWarp,    "Flatten Photo", 0,            flatphoto_tip,             m_flatphoto, 0 );

   mComb = create_popmenu();
   MENU(mComb,    "HDR", 0,                      HDR_tip,                   m_HDR, 0 );
   MENU(mComb,    "HDF", 0,                      HDF_tip,                   m_HDF, 0 );
   MENU(mComb,    "Stack/Paint", 0,              stack_paint_tip,           m_stack_paint, 0 );
   MENU(mComb,    "Stack/Noise", 0,              stack_noise_tip,           m_stack_noise, 0 );
   MENU(mComb,    "Stack/Versions", 0,           stack_versions_tip,        m_stack_versions, 0 );
   MENU(mComb,    "Stack/Split", 0,              stack_split_tip,           m_stack_split, 0 );
   MENU(mComb,    "Panorama", 0,                 panorama_tip,              m_panorama, 0 );
   MENU(mComb,    "Show CIM Files", 0,           show_cim_files_tip,        m_cim_show_files, 0 );
   MENU(mComb,    "Mashup", 0,                   mashup_tip,                m_mashup, 0 );
   MENU(mComb,    "Image Array", 0,              image_array_tip,           m_image_array, 0 );

   mBatch = create_popmenu();
   MENU(mBatch,    "Batch Convert", 0,           batch_convert_tip,         m_batch_convert, 0 );
   MENU(mBatch,    "Batch Copy/Move", 0,         batch_copy_move_tip,       m_batch_copy_move, 0 );
   MENU(mBatch,    "Batch Upright", 0,           batch_upright_tip,         m_batch_upright, 0 );
   MENU(mBatch,    "Batch Delete/Trash", 0,      batch_deltrash_tip,        m_batch_deltrash, 0 );
   MENU(mBatch,    "Batch RAW", 0,               batch_RAW_tip,             m_batch_RAW, 0 );
   MENU(mBatch,    "Batch Overlay", 0,           batch_overlay_tip,         m_batch_overlay, 0 );
   MENU(mBatch,    "Batch Tags", 0,              batch_tags_tip,            m_batch_tags, 0 );
   MENU(mBatch,    "Batch Rename Tags", 0,       batch_rename_tags_tip,     m_batch_rename_tags, 0 );
   MENU(mBatch,    "Batch Photo Date", 0,        batch_photo_DT_tip,        m_batch_photo_date_time, 0 );
   MENU(mBatch,    "Batch Change Meta", 0,       batch_change_meta_tip,     m_batch_change_meta, 0 );
   MENU(mBatch,    "Batch Report Meta", 0,       batch_report_meta_tip,     m_batch_report_meta, 0 );
   MENU(mBatch,    "Batch Move Meta", 0,         batch_meta_mover_tip,      m_batch_meta_mover, 0 );
   MENU(mBatch,    "Batch Geotags", 0,           batch_geotags_tip,         m_batch_geotags, 0 );
   MENU(mBatch,    "Export File List", 0,        export_filelist_tip,       m_export_filelist, 0 );
   MENU(mBatch,    "Export Files", 0,            export_files_tip,          m_export_files, 0 );
   MENU(mBatch,    "Edit Script", 0,             edit_script_tip,           m_edit_script, 0 );
   MENU(mBatch,    "Run Script", 0,              run_script_tip,            m_run_script, 0);
   MENU(mBatch,    "Batch Script", 0,            batch_script_tip,          m_batch_script, 0);

   mTools = create_popmenu();
   MENU(mTools,   "Index Files", 0,              index_tip,                 m_index, 0 );
   MENU(mTools,   "Quick Index", 0,              quick_index_tip,           m_quick_index, 0 );
   MENU(mTools,   "Settings", 0,                 settings_tip,              m_settings, 0 );
   MENU(mTools,   "KB Shortcuts", 0,             KB_shortcuts_tip,          m_KB_shortcuts, 0 );
   MENU(mTools,   "RGB Distribution", 0,         RGB_dist_tip,              m_RGB_dist, 0 );
   MENU(mTools,   "Magnify Image", 0,            magnify_tip,               m_magnify, 0 );
   MENU(mTools,   "Find Duplicates", 0,          duplicates_tip,            m_duplicates, 0 );
   MENU(mTools,   "Measure Image", 0,            measure_image_tip,         m_measure_image, 0 );
   MENU(mTools,   "Show RGB", 0,                 show_RGB_tip,              m_show_RGB, 0 );
   MENU(mTools,   "Popup Text", 0,               popup_text_tip,            m_popup_text, 0 );
   MENU(mTools,   "Calibrate Printer", 0,        calib_printer_tip,         m_calibrate_printer, 0 );
   MENU(mTools,   "Grid Settings", 0,            grid_settings_tip,         m_grid_settings, 0 );
   MENU(mTools,   "Toggle_Grid", 0,              toggle_grid_tip,           m_toggle_grid, 0 );
   MENU(mTools,   "Line Color", 0,               line_color_tip,            m_line_color, 0 );
   MENU(mTools,   "Dark/Bright Pixels", 0,       darkbrite_tip,             m_darkbrite, 0 );
   MENU(mTools,   "Monitor Color", 0,            monitor_color_tip,         m_monitor_color, 0 );
   MENU(mTools,   "Show Resources", 0,           resources_tip,             m_resources, 0 );

   mHelp = create_popmenu();
   MENU(mHelp,    "User Guide", 0,               user_guide_tip,            m_help, 0 );
   MENU(mHelp,    "Video Tutorial", 0,           video_tutorial_tip,        m_help, 0 );
   MENU(mHelp,    "All Edit Funcs", 0,           all_edit_funcs_tip,        m_help, 0 );
   MENU(mHelp,    "Outboard Programs", 0,        outboard_programs_tip,     m_help, 0 );
   MENU(mHelp,    "Log File", 0,                 logfile_tip,               m_help, 0 );
   MENU(mHelp,    "Command Params", 0,           command_params_tip,        m_help, 0 );
   MENU(mHelp,    "Changelog", 0,                changelog_tip,             m_help, 0 );
   MENU(mHelp,    "Copyright", 0,                copyright_tip,             m_help, 0 );
   MENU(mHelp,    "Licenses", 0,                 licenses_tip,              m_help, 0 );
   MENU(mHelp,    "Source Build", 0,             source_build_tip,          m_help, 0 );
   MENU(mHelp,    "Privacy", 0,                  privacy_tip,               m_help, 0 );
   MENU(mHelp,    "About Fotoxx", 0,             about_tip,                 m_help, 0 );
   MENU(mHelp,    "Home Page", 0,                homepage_tip,              m_help, 0 );
   MENU(mHelp,    "Uninstall", 0,                uninstall_tip,             m_help, 0 );

   //  main menu buttons - must be last

   MENU(0,  "File",         "file.png",          File_tip,            (cbFunc *) popup_menu, (ch *) mFile);
   MENU(0,  "Gallery",      "gallery.png",       Gallery_tip,         (cbFunc *) popup_menu, (ch *) mGallery);
   MENU(0,  "Maps",         "maps.png",          Maps_tip,            (cbFunc *) popup_menu, (ch *) mMap);
   MENU(0,  "Custom",       "custom.png",        custom_menu_tip,     m_custom_menu, 0 );
   MENU(0,  "Zoom",         "zoom.png",          zoom_menu_tip,       m_zoom_menu, 0 );
   MENU(0,  "Prev/Next",    "prev-next.png",     prev_next_tip,       m_prev_next, 0 );
   MENU(0,  "Save",         "save.png",          save_tip,            m_file_save, 0 );
   MENU(0,  "Metadata",     "metadata.png",      metadata_tip,        (cbFunc *) popup_menu, (ch *) mMeta);
   MENU(0,  "Select",       "select.png",        select_tip,          (cbFunc *) popup_menu, (ch *) mSelect);
   MENU(0,  "Undo/Redo",    "undo-redo.png",     undo_redo_tip,       m_undo_redo, 0 );
   MENU(0,  "Edit",         "edit.png",          edit_tip,            (cbFunc *) popup_menu, (ch *) mEdit);
   MENU(0,  "Repair",       "repair.png",        repair_tip,          (cbFunc *) popup_menu, (ch *) mRepair);
   MENU(0,  "Refine",       "refine.png",        refine_tip,          (cbFunc *) popup_menu, (ch *) mRefine);
   MENU(0,  "Effects",      "effects.png",       effects_tip,         (cbFunc *) popup_menu, (ch *) mEffects);
   MENU(0,  "Warp",         "warp.png",          warp_tip,            (cbFunc *) popup_menu, (ch *) mWarp);
   MENU(0,  "Combine",      "combine.png",       combine_tip,         (cbFunc *) popup_menu, (ch *) mComb);
   MENU(0,  "Batch",        "batch.png",         batch_tip,           (cbFunc *) popup_menu, (ch *) mBatch);
   MENU(0,  "Tools",        "tools.png",         tools_tip,           (cbFunc *) popup_menu, (ch *) mTools);
   MENU(0,  "Help",         "help.png",          help_tip,            (cbFunc *) popup_menu, (ch *) mHelp);

   //  show developer tools menu if required

   if (Fdevmenu) 
   {
      mDevmenu = create_popmenu();
      MENU(mDevmenu, "zappcrash test",    0, "backtrace dump", m_zappcrash_test, 0 ); 
      MENU(mDevmenu, "zmalloc report",    0, "show memory allocations by tag", m_zmalloc_report, 0);
      MENU(mDevmenu, "zmalloc growth",    0, "show memory growth by tag", m_zmalloc_growth, 0);
      MENU(mDevmenu, "mouse events",      0, "toggle: show mouse events popup text", m_mouse_events, 0);
      MENU(mDevmenu, "Audit User Guide",  0, "check all user guide links", m_audit_userguide, 0);
      MENU(mDevmenu, "Edit Template",     0, "edit function template for new code", m_template, 0);                     // 23.60
      MENU(0, "Dev-Tools", "devmenu.png", "developer tools", (cbFunc *) popup_menu, (ch *) mDevmenu);
   }

   int   Vmenus = Nmenus;                                                        //  visible menus stop here

   //  internal functions not shown in menus
   
   MENU(mDevmenu, "Copy From Image Slave", 0, "slave process",  m_copy_from_image_slave, 0 );
   MENU(mDevmenu, "Autosearch", 0, "Write search results to stdout", m_autosearch, 0);
   
   //  build the menu buttons for the main menu ---------------------------

   float    frgb[3], brgb[3];
   frgb[0] = MFrgb[0] / 256.0;                                                   //  menu font color
   frgb[1] = MFrgb[1] / 256.0;                                                   //  convert range to 0-1
   frgb[2] = MFrgb[2] / 256.0;
   brgb[0] = MBrgb[0] / 256.0;                                                   //  menu background color
   brgb[1] = MBrgb[1] / 256.0;                                                   //  convert range to 0-1
   brgb[2] = MBrgb[2] / 256.0;

   Vmenu *Xvm = Vmenu_new(MWmenu,frgb,brgb);                                     //  create main menu

   int   siz = iconsize;                                                         //  user settings parameter

   for (me = 0; me < Vmenus; me++)
   {
      if (menutab[me].topmenu)                                                   //  submenu within top menu
         add_popmenu_item(menutab[me].topmenu, menutab[me].menu,
                menutab[me].func, menutab[me].arg, menutab[me].desc);

      else                                                                       //  top menu (button)
      {
         if (strmatch(menu_style,"icons")) {                                     //  icons only
            if (menutab[me].icon)
               Vmenu_add(Xvm, 0, menutab[me].icon,siz,siz,menutab[me].desc,
                                 menutab[me].func, menutab[me].arg);
            else                                                                 //  no icon, use menu text
               Vmenu_add(Xvm, menutab[me].menu, 0, 0, 0, menutab[me].desc,
                                 menutab[me].func, menutab[me].arg);
         }
         
         else if (strmatch(menu_style,"text")) {                                 //  text only
            Vmenu_add(Xvm, menutab[me].menu, 0, 0, 0, menutab[me].desc,
                        menutab[me].func, menutab[me].arg);
         }
         
         else                                                                    //  icons + menu text
            Vmenu_add(Xvm, menutab[me].menu, menutab[me].icon, siz, siz,
                        menutab[me].desc, menutab[me].func, menutab[me].arg);
      }
   }

   Vmenu_add_setup(Xvm, 0, m_viewmode, "F");                                     //  add setup functions
   Vmenu_add_setup(Xvm, 1, m_viewmode, "G");                                     //    for top 3 menu buttons
   Vmenu_add_setup(Xvm, 2, m_viewmode, "WM");                                    //      to set corresp. view mode
   
   Vmenu_add_RMfunc(Xvm, 0, m_viewmode, "F");                                    //  add right-mouse function
   Vmenu_add_RMfunc(Xvm, 1, m_viewmode, "G");                                    //    for top 3 menu buttons
   Vmenu_add_RMfunc(Xvm, 2, m_viewmode, "WM");


   //  build table of eligible menus for KB shortcut assignment -----------------

   #define KBshort(_menu, _func, _arg)                               \
      me = Nkbsf++;                                                  \
      if (me >= maxkbsf) zappcrash("maxkbs exceeded");               \
      kbsftab[me].menu = _menu;                                      \
      kbsftab[me].func = _func;                                      \
      kbsftab[me].arg = _arg;

   Nkbsf = 0;

   //       menu                 called function           arg
   KBshort("Adjust HSL",         m_adjust_HSL,              0     );
   KBshort("Adjust RGB",         m_adjust_RGB,              0     );
   KBshort("Folder Tree",        m_folder_tree,             0     );
   KBshort("Blur",               m_blur,                    0     );
   KBshort("Bookmarks",          m_bookmarks,               0     );
   KBshort("Choose Captions",    m_meta_choose_caps,       "x"    );
   KBshort("Color Depth",        m_color_depth,             0     );
   KBshort("Color Mode",         m_color_mode,              0     );
   KBshort("Copy/Move",          m_copy_move,               0     );
   KBshort("Copy to Clipboard",  m_copyto_clip,             0     );
   KBshort("Copy to Desktop",    m_copyto_desktop,          0     );
   KBshort("Crop",               m_crop,                    0     );
   KBshort("Current Album",      m_current_album,           0     );
   KBshort("Cycle 2",            m_cycle2files,             0     );
   KBshort("Cycle 3",            m_cycle3files,             0     );
   KBshort("Delete Meta",        m_meta_delete,             0     );
   KBshort("Delete/Trash",       m_delete_trash,            0     );
   KBshort("Denoise",            m_denoise,                 0     );
   KBshort("Dither",             m_dither,                  0     );
   KBshort("Edit Any Meta",      m_meta_edit_any,           0     );
   KBshort("Edit Dist",          m_edit_dist,               0     );
   KBshort("Edit Meta",          m_meta_edit_main,          0     );
   KBshort("File View",          m_viewmode,               "F"    );
   KBshort("Flatten Dist",       m_flatdist,                0     );
   KBshort("Gallery View",       m_viewmode,               "G"    );
   KBshort("Global Retinex",     m_gretinex,                0     );
   KBshort("Gradients",          m_gradients,               0     );
   KBshort("Grid Settings",      m_grid_settings,           0     );
   KBshort("KB Shortcuts",       m_KB_shortcuts,            0     );
   KBshort("Line Color",         m_line_color,              0     );
   KBshort("Local Retinex",      m_lretinex,                0     );
   KBshort("Magnify Image",      m_magnify,                 0     );
   KBshort("Manage Albums",      m_manage_albums,           0     );
   KBshort("Map View",           m_viewmode,               "WM"   );
   KBshort("Markup",             m_markup,                  0     );
   KBshort("Meta View",          m_metaview,                0     );
   KBshort("Newest",             m_newfiles,                0     );
   KBshort("New Session",        m_new_session,             0     );
   KBshort("New Version",        m_file_save_version,       0     );
   KBshort("Open Image File",    m_open_file,               0     );
   KBshort("Permissions",        m_permissions,             0     );
   KBshort("Places/Dates",       m_meta_places_dates,       0     );
   KBshort("Popup Text",         m_popup_text,              0     );
   KBshort("Settings",           m_settings,                0     );
   KBshort("Print Calibrated",   m_print_calibrated,        0     );
   KBshort("Print",              m_print,                   0     );
   KBshort("Quit",               m_quit,                    0     );
   KBshort("Recent",             m_recentfiles,             0     );
   KBshort("Red Eyes",           m_redeyes ,                0     );
   KBshort("Redo",               m_redo,                    0     );
   KBshort("Rename",             m_rename,                  0     );
   KBshort("Replace",            m_file_save_replace,       0     );
   KBshort("Rescale",            m_rescale,                 0     );
   KBshort("Retouch",            m_retouch,                 0     );
   KBshort("RGB Distribution",   m_RGB_dist,                0     );
   KBshort("Rotate",             m_rotate,                  0     );
   KBshort("Save",               m_file_save,               0     );
   KBshort("Search",             m_search_images,           0     );
   KBshort("Select Area",        m_select_area,             0     );   
   KBshort("Select Files",       m_select_files,            0     );   
   KBshort("Sharpen",            m_sharpen,                 0     );
   KBshort("Show Hidden",        m_show_hidden,             0     );
   KBshort("Show on Map",        m_netmap_zoomin,           0     ); 
   KBshort("Show Resources",     m_resources,               0     );
   KBshort("Image Folder",       m_image_folder,            0     );
   KBshort("Timeline",           m_meta_timeline,           0     );
   KBshort("Toggle Captions",    m_meta_toggle_caps,        0     );
   KBshort("Toggle Grid",        m_toggle_grid,             0     );
   KBshort("Undo",               m_undo,                    0     );
   KBshort("Upright",            m_upright,                 0     ); 
   KBshort("View 360° Pano",     m_view360,                 0     );
   KBshort("View Meta",          m_meta_view_short,         0     );
   KBshort("Voodoo 1",           m_voodoo1,                 0     );
   KBshort("Voodoo 2",           m_voodoo2,                 0     );
   KBshort("zmalloc by tag",     m_zmalloc_report,          0     );
   KBshort("zmalloc growth",     m_zmalloc_growth,          0     );
   KBshort("Zoom-in",            m_zoom,                   "in"   );
   KBshort("Zoom-out",           m_zoom,                   "out"  );
   //  zdialog completion buttons that can have KB shortcuts assigned
   KBshort("Apply",              0,                         0     );
   KBshort("Cancel",             0,                         0     );
   KBshort("Reset",              0,                         0     );
   KBshort("OK",                 0,                         0     );
   
   //  build right-click popup menus --------------------------------------------

   ch     *menumeta1 = "View Meta";
   ch     *menumeta1A = "View All Meta";
   ch     *menumeta2 = "Edit Meta";
   ch     *menurename = "Rename";
   ch     *menudeltrash = "Delete/Trash";
   ch     *menucopymove = "Copy/Move";
   ch     *menucopytodesktop = "Copy to Desktop";
   ch     *menucopytoclip = "Copy to Clipboard";
   ch     *menurotateimage = "Rotate";
   ch     *menucropimage = "Crop";
   ch     *menuretouch = "Retouch";
   ch     *menurescale = "Rescale";
   ch     *menusharpen = "Sharpen";
   ch     *menublur = "Blur";
   ch     *menudenoise = "Denoise";
   ch     *menuvoodoo1 = "Voodoo 1";
   ch     *menuvoodoo2 = "Voodoo 2";
   ch     *menueditdist = "Edit Dist";
   ch     *menuflattendist = "Flatten Dist";
   ch     *menulocalcon = "Local Contrast";
   ch     *menugradients = "Gradients";
   ch     *menusaturation = "Saturation";
   ch     *menuthumbframe = "Thumbnail Frame";
   ch     *menupopimage = "Popup Image";
   ch     *menualbumaddselfiles = "Add Selected Files Here";
   ch     *menualbumaddcurrfile = "Add Current File Here";
   ch     *menuremovefromalbum = "Remove from Album";

   popmenu_image = create_popmenu();                                             //  popup menu for image files
   add_popmenu_item(popmenu_image,menumeta1,popup_menufunc,"view meta");
   add_popmenu_item(popmenu_image,menumeta1A,popup_menufunc,"view all meta");
   add_popmenu_item(popmenu_image,menumeta2,popup_menufunc,"edit meta");
   add_popmenu_item(popmenu_image,menurename,popup_menufunc,"rename");
   add_popmenu_item(popmenu_image,menudeltrash,popup_menufunc,"delete/trash");
   add_popmenu_item(popmenu_image,menucopymove,popup_menufunc,"copymove");
   add_popmenu_item(popmenu_image,menucopytodesktop,popup_menufunc,"copytodesktop");
   add_popmenu_item(popmenu_image,menucopytoclip,popup_menufunc,"copytoclip");
   add_popmenu_item(popmenu_image,menurotateimage,popup_menufunc,"rotate");
   add_popmenu_item(popmenu_image,menucropimage,popup_menufunc,"crop");
   add_popmenu_item(popmenu_image,menuretouch,popup_menufunc,"retouch");
   add_popmenu_item(popmenu_image,menurescale,popup_menufunc,"rescale");
   add_popmenu_item(popmenu_image,menusharpen,popup_menufunc,"sharpen");
   add_popmenu_item(popmenu_image,menublur,popup_menufunc,"blur");
   add_popmenu_item(popmenu_image,menudenoise,popup_menufunc,"denoise");
   add_popmenu_item(popmenu_image,menuvoodoo1,popup_menufunc,"voodoo1");
   add_popmenu_item(popmenu_image,menuvoodoo2,popup_menufunc,"voodoo2");
   add_popmenu_item(popmenu_image,menueditdist,popup_menufunc,"edit dist");
   add_popmenu_item(popmenu_image,menuflattendist,popup_menufunc,"flatten dist");
   add_popmenu_item(popmenu_image,menulocalcon,popup_menufunc,"local contrast");
   add_popmenu_item(popmenu_image,menugradients,popup_menufunc,"gradients");
   add_popmenu_item(popmenu_image,menusaturation,popup_menufunc,"saturation");

   popmenu_raw = create_popmenu();                                               //  popup menu for RAW files
   add_popmenu_item(popmenu_raw,menumeta1,popup_menufunc,"view meta");
   add_popmenu_item(popmenu_raw,menumeta1A,popup_menufunc,"view all meta");
   add_popmenu_item(popmenu_raw,menumeta2,popup_menufunc,"edit meta");
   add_popmenu_item(popmenu_raw,menurename,popup_menufunc,"rename");
   add_popmenu_item(popmenu_raw,menudeltrash,popup_menufunc,"delete/trash");
   add_popmenu_item(popmenu_raw,menucopymove,popup_menufunc,"copymove");

   popmenu_video = create_popmenu();                                             //  popup menu for VIDEO files
   add_popmenu_item(popmenu_video,menumeta1,popup_menufunc,"view meta");
   add_popmenu_item(popmenu_video,menumeta1A,popup_menufunc,"view all meta");
   add_popmenu_item(popmenu_video,menumeta2,popup_menufunc,"edit meta");
   add_popmenu_item(popmenu_video,menurename,popup_menufunc,"rename");
   add_popmenu_item(popmenu_video,menudeltrash,popup_menufunc,"delete/trash");
   add_popmenu_item(popmenu_video,menucopymove,popup_menufunc,"copymove");
   add_popmenu_item(popmenu_video,menucopytodesktop,popup_menufunc,"copytodesktop");
   add_popmenu_item(popmenu_video,menuthumbframe,popup_menufunc,"thumbnail_frame");

   popmenu_thumb = create_popmenu();                                             //  gallery thumbnail popup menu
   add_popmenu_item(popmenu_thumb,menupopimage,popup_menufunc,"popimage");
   add_popmenu_item(popmenu_thumb,menumeta1,popup_menufunc,"view meta");
   add_popmenu_item(popmenu_thumb,menumeta1A,popup_menufunc,"view all meta");
   add_popmenu_item(popmenu_thumb,menumeta2,popup_menufunc,"edit meta");
   add_popmenu_item(popmenu_thumb,menurename,popup_menufunc,"rename");
   add_popmenu_item(popmenu_thumb,menudeltrash,popup_menufunc,"delete/trash");
   add_popmenu_item(popmenu_thumb,menucopymove,popup_menufunc,"copymove");
   add_popmenu_item(popmenu_thumb,menucopytodesktop,popup_menufunc,"copytodesktop");
   add_popmenu_item(popmenu_thumb,menucopytoclip,popup_menufunc,"copytoclip");

   popmenu_album = create_popmenu();                                             //  album thumbnail popup menu
   add_popmenu_item(popmenu_album,menupopimage,popup_menufunc,"popimage");
   add_popmenu_item(popmenu_album,menumeta1,popup_menufunc,"view meta");
   add_popmenu_item(popmenu_album,menumeta1A,popup_menufunc,"view all meta");
   add_popmenu_item(popmenu_album,menumeta2,popup_menufunc,"edit meta");
   add_popmenu_item(popmenu_album,menucopytodesktop,popup_menufunc,"copytodesktop");
   add_popmenu_item(popmenu_album,menucopytoclip,popup_menufunc,"copytoclip");
   add_popmenu_item(popmenu_album,menualbumaddselfiles,popup_menufunc,"albumaddselfiles");
   add_popmenu_item(popmenu_album,menualbumaddcurrfile,popup_menufunc,"albumaddcurrfile");
   add_popmenu_item(popmenu_album,menuremovefromalbum,popup_menufunc,"removefromalbum");

   return;
}


//  right-click popup menu response function

void popup_menufunc(GtkWidget *, ch *menu)
{
   if (strmatch(menu,"view meta")) meta_view(1);
   if (strmatch(menu,"view all meta")) meta_view(2);
   if (strmatch(menu,"edit meta")) m_meta_edit_main(0,0);
   if (strmatch(menu,"rename")) m_rename(0,0);
   if (strmatch(menu,"delete/trash")) m_delete_trash(0,0);
   if (strmatch(menu,"copymove")) m_copy_move(0,0);
   if (strmatch(menu,"copytodesktop")) m_copyto_desktop(0,0);
   if (strmatch(menu,"copytoclip")) m_copyto_clip(0,0);
   if (strmatch(menu,"rotate")) m_rotate(0,0);
   if (strmatch(menu,"crop")) m_crop(0,0);
   if (strmatch(menu,"retouch")) m_retouch(0,0);
   if (strmatch(menu,"rescale")) m_rescale(0,0);
   if (strmatch(menu,"sharpen")) m_sharpen(0,0);
   if (strmatch(menu,"blur")) m_blur(0,0);
   if (strmatch(menu,"denoise")) m_denoise(0,0);
   if (strmatch(menu,"voodoo1")) m_voodoo1(0,0);
   if (strmatch(menu,"voodoo2")) m_voodoo2(0,0);
   if (strmatch(menu,"edit dist")) m_edit_dist(0,0);
   if (strmatch(menu,"flatten dist")) m_flatdist(0,0);
   if (strmatch(menu,"local contrast")) m_localcon(0,0);
   if (strmatch(menu,"gradients")) m_gradients(0,0);
   if (strmatch(menu,"saturation")) m_saturation(0,0);
   if (strmatch(menu,"thumbnail_frame")) m_thumbframe(0,0);
   if (strmatch(menu,"popimage")) gallery_popimage();
   if (strmatch(menu,"albumaddselfiles")) album_add_selfiles(2);
   if (strmatch(menu,"albumaddcurrfile")) album_add_currfile(2);
   if (strmatch(menu,"removefromalbum")) album_remove_file(clicked_posn);

   return;
}


//  main window mouse right-click popup menu

void image_Rclick_popup()
{
   int      ftype;

   if (! curr_file) return;

   ftype = image_file_type(curr_file);
   if (ftype == IMAGE) popup_menu(Mwin,popmenu_image);
   if (ftype == RAW) popup_menu(Mwin,popmenu_raw);
   if (ftype == VIDEO) popup_menu(Mwin,popmenu_video);
   return;
}


//  gallery thumbnail mouse left-click function
//  open the clicked file in view mode F

void gallery_Lclick_func(int Nth)
{
   ch       *file;
   int      err = 0;

   if (clicked_file) {                                                           //  free memory of clicked thumbnail
      zfree(clicked_file);
      clicked_file = 0;
   }

   file = gallery(0,"get",Nth);
   if (! file) return;

   if (curr_file && strmatch(curr_file,file)) {                                  //  clicked file = current file           23.70
      if (! Fblock(0,"blocked edits"))                                           //  if not pending, re-open file
         err = f_open(file,Nth,0,1);                                             
      if (! err) viewmode("F");                                                  //  back to F-view
      zfree(file);
      return;
   }

   if (Fblock(0,"blocked edits")) {                                              //  if pending, do nothing
      zfree(file);
      return;
   }

   err = f_open(file,Nth,0,1);                                                   //  clicked file >> current file
   if (! err) viewmode("F");
   zfree(file);
   return;
}


//  gallery thumbnail mouse right-click popup menu

void gallery_Rclick_popup(int Nth)
{
   FTYPE    ftype;

   clicked_posn = Nth;                                                           //  clicked gallery position (0 base)
   clicked_file = gallery(0,"get",Nth);                                          //  clicked_file is subject for zfree()
   if (! clicked_file) return;

   if (navi::gallerytype == ALBUM) {
      popup_menu(Mwin,popmenu_album);
      return;
   }

   ftype = image_file_type(clicked_file);
   if (ftype == IMAGE) popup_menu(Mwin,popmenu_thumb);
   if (ftype == RAW) popup_menu(Mwin,popmenu_raw);
   if (ftype == VIDEO) popup_menu(Mwin,popmenu_video);

   return;
}


/********************************************************************************/

//  set window view mode, F/G/M/W

void m_viewmode(GtkWidget *, ch *fgwm)                                           //  menu function
{
   viewmode(fgwm);
   return;
}


void viewmode(ch *fgwm)                                                          //  callable                              23.70
{
   static ch    lastWM[4] = "M";                                                 //  net map default
   ch           wintitle[100];
   
   Plog(1,"m_viewmode %c \n",*fgwm);                                             //  23.1

   if (! fgwm) fgwm = "0";                                                       //  null -> "0"

   if (strmatch(fgwm,"WM"))                                                      //  use last W or M mode
      fgwm = lastWM;

   if (*fgwm == FGWM) {                                                          //  no view mode change
      zfuncs::vmenustop = 0;                                                     //  continue menu func 
      return;
   }

   zfuncs::vmenustop = 1;                                                        //  view mode change, stop menu func 

   if (*fgwm == '0')                                                             //  set no view mode (blank window)
   {
      gtk_widget_hide(Fhbox);
      gtk_widget_hide(Ghbox);
      gtk_widget_hide(Whbox);
      gtk_widget_hide(Mhbox);
      FGWM = '0';
      PFGWM = '0';                                                               //  remember last F/G view

      Cstate = 0;                                                                //  no F/W image drawing area
      Cdrawin = 0;
      gdkwin = 0;

      if (zd_deltrash) m_delete_trash(0,0);                                      //  set target file in active dialog
      if (zd_copymove) m_copy_move(0,0);
      if (zd_metaview) meta_view(0);
      if (zd_rename) m_rename(0,0);
      if (zd_permissions) m_permissions(0,0);
   }

   if (*fgwm == 'F')                                                             //  set F view mode for image file
   {
      gtk_widget_hide(Ghbox);
      gtk_widget_hide(Whbox);
      gtk_widget_hide(Mhbox);
      gtk_widget_show_all(Fhbox);
      if (! Fpanelshow) gtk_widget_hide(Fpanel);                                 //  no panel                              23.60

      Cstate = &Fstate;                                                          //  set drawing area
      Cdrawin = Fdrawin;
      gdkwin = gtk_widget_get_window(Fdrawin);                                   //  GDK window

      gdk_window_hide(gdkwin);                                                   //  *** gdk bug workaround ***
      zmainsleep(0.001);
      gdk_window_show_unraised(gdkwin);                                          //  sometimes fails to paint  U22.04

      FGWM = 'F';
      PFGWM = 'F';                                                               //  remember last F/G view

      set_mwin_title();

      if (zd_deltrash) m_delete_trash(0,0);                                      //  set target file in active dialog
      if (zd_copymove) m_copy_move(0,0);
      if (zd_metaview) meta_view(0);
      if (zd_rename) m_rename(0,0);
      if (zd_permissions) m_permissions(0,0);
   }

   if (*fgwm == 'G')                                                             //  set G view mode for thumbnail gallery
   {
      gtk_widget_hide(Fhbox);
      gtk_widget_hide(Whbox);
      gtk_widget_hide(Mhbox);
      gtk_widget_show_all(Ghbox);
      FGWM = 'G';
      PFGWM = 'G';                                                               //  remember last F/G view

      Cstate = 0;                                                                //  no F/W image drawing area
      Cdrawin = 0;
      gdkwin = 0;

      if (curr_file) gallery(curr_file,"paint",0);                               //  set gallery posn. at curr. file
      else gallery(0,"paint",-1);                                                //  else leave unchanged
   }

   if (*fgwm == 'M')                                                             //  set M view mode for net maps
   {
      if (Findexvalid == 0) {
         zmessageACK(Mwin,"image index disabled");                               //  no image index
         return;
      }

      if (CEF) return;                                                           //  don't interrupt edit func.
      
      gtk_widget_hide(Fhbox);                                  
      gtk_widget_hide(Ghbox);
      gtk_widget_hide(Whbox);
      gtk_widget_show_all(Mhbox);
      FGWM = 'M';
      lastWM[0] = 'M';

      Cstate = 0;                                                                //  no F/W image drawing area
      Cdrawin = 0;
      gdkwin = 0;
      
      m_load_netmap(0,"init");                                                   //  load net initial map
      snprintf(wintitle,100,"%s   Image Locations",Frelease); 
      gtk_window_set_title(MWIN,wintitle);                                       //  window title
   }

   if (*fgwm == 'W')                                                             //  set W view mode for file maps
   {
      if (CEF) return;                                                           //  don't interrupt edit func.

      if (Findexvalid == 0) {
         zmessageACK(Mwin,"image index disabled");                               //  no image index
         return;
      }

      gtk_widget_hide(Fhbox);                                  
      gtk_widget_hide(Ghbox);
      gtk_widget_hide(Mhbox);
      gtk_widget_show_all(Whbox);
      FGWM = 'W';
      lastWM[0] = 'W';

      Cstate = &Wstate;                                                          //  set drawing area
      Cdrawin = Wdrawin;
      gdkwin = gtk_widget_get_window(Wdrawin);                                   //  GDK window

      snprintf(wintitle,100,"%s   Image Locations",Frelease);
      gtk_window_set_title(MWIN,wintitle);                                       //  window title
   }
   
   return;
}


/********************************************************************************/

//  generate custom user menu from custom menu file
//  menu entry options:
//     # comment
//     menuname   # comment
//     menuname -f /.../filename.ext  # comment


namespace custom_menu_names
{
   ch    **menulist = 0;
   int   Nm = 0;
}


void m_custom_menu(GtkWidget *, ch *)                                            //  23.50
{
   using namespace custom_menu_names;

   void custom_menu_process(GtkWidget *, ch *);
   void custom_menu_edit(GtkWidget *, ch *);

   GtkWidget   *popmenu;
   ch          menufile[200];
   ch          *edit_menu = "Edit Menu    # edit this menu";
   int         ii;
   
   F1_help_topic = "custom menu";
   
   Plog(1,"m_custom_menu \n");

   if (! menulist) {                                                             //  first use
      snprintf(menufile,200,"%s/custom_menu",get_zhomedir());                    //  read custom menu file
      Nm = zreadfile(menufile,menulist);                                         //  (may be empty)
   }

   popmenu = create_popmenu();                                                   //  create popup menu

   for (ii = 0; ii < Nm; ii++)                                                   //  add all lines to popup menu
      add_popmenu_item(popmenu,menulist[ii],custom_menu_process,0,0);

   add_popmenu_item(popmenu,edit_menu,custom_menu_edit,0,0);                     //  add menu edit function at end

   popup_menu(Mwin,popmenu);                                                     //  show popup menu
   
   return;
}


void custom_menu_process(GtkWidget *, ch *menu)                                  //  process selected menu item
{
   using namespace custom_menu_names;

   int      ii;
   ch       *pp, menuname[400], comment[200];
   ch       filename[400], albumname[400];

   *menuname = *comment = *filename = 0;

   strncpy0(menuname,menu,400);                                                  //  menu name and/or # comment

   pp = strchr(menuname,'#');                                                    //  get comment if any
   if (pp) {
      strncpy0(comment,pp+1,200);
      *pp = 0;
   }
   
   pp = strstr(menuname,"-f ");                                                  //  look for filename in menuname
   if (pp) {
      strncpy0(filename,pp+3,400);
      *pp = 0;
   }

   strTrim2(menuname);                                                           //  menu name only, no surrounding blanks
   strTrim2(comment);                                                            //  comment only
   strTrim2(filename);                                                           //  filename only
   
   if (strmatchcase(menuname,"album")) {
      if (*filename) {
         snprintf(albumname,400,"/home2/mico/alt-home/fotoxx/albums/%s",filename);
         album_show(albumname);
      }
      else album_show(curr_album);
      return;
   }

   if (*filename) f_open(filename,0,0,1,0);                                      //  open file

   for (ii = 0; ii < Nmenus; ii++) {                                             //  find menu name in menu table
      if (! menutab[ii].menu) continue;                                          //  separator, null menu
      if (strmatchcase(menuname,menutab[ii].menu)) break;
   }

   if (ii < Nmenus) menutab[ii].func(0,0);                                       //  found, call menu function
   else zmessageACK(Mwin,menu);                                                  //  menu not found, display it

   return;
}


void custom_menu_edit(GtkWidget *, ch *)                                         //  edit the custom menu file
{
   using namespace custom_menu_names;

   if (menulist) zreadfile_free(menulist);
   menulist = 0;
   Nm = 0;

   zshell("ack","xdg-open %s/custom_menu",get_zhomedir());
   return;
}



