
/*
 * LinuxDistro.vala
 *
 * Copyright 2012-17 Tony George <teejeetech@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 *
 */

using TeeJee.Logging;
using TeeJee.FileSystem;
using TeeJee.ProcessHelper;

public class LinuxDistro : GLib.Object{

	/* Class for getting information about Linux distribution */

	public string dist_id = ""; // Ubuntu
	public string dist_type = ""; // debian, fedora, arch

	public string description = ""; // Ubuntu 16.04
	public string release = ""; // 16.04
	public string codename = ""; // xenial
	public string dist_full_name = ""; // Ubuntu 16.04 (xenial)
	
	public string package_manager = ""; // dnf, yum, apt, pacman
	public string package_arch = ""; // x86_64, amd64, i386, i686, ...
	public string kernel_arch = ""; // x86_64, i386, i686, ...
	public int machine_arch = 0; // 32, 64, ...

	public string basepath = ""; // system root or mount path

	// ---------------------------------------
	
	public LinuxDistro(){

		basepath = "/";
		
		read_dist_release_file();
		
		check_dist_type();

		check_package_manager();

		check_package_arch();

		check_system();
	}

	public LinuxDistro.from_path(string root_path){

		basepath = root_path;

		read_dist_release_file();

		check_dist_type();

		check_package_manager();

		//check_package_arch(); // NOT_IMPLEMENTED

		//check_system(); // NOT_IMPLEMENTED
	}

	private void read_dist_release_file(){

		/* Returns info about Linux distribution installed at root path */

		string dist_file = path_combine(basepath, "/etc/lsb-release");
		
		var f = File.new_for_path(dist_file);
		
		if (f.query_exists()){

			/*
				DISTRIB_ID=Ubuntu
				DISTRIB_RELEASE=13.04
				DISTRIB_CODENAME=raring
				DISTRIB_DESCRIPTION="Ubuntu 13.04"
			*/

			foreach(string line in file_read(dist_file).split("\n")){

				if (line.split("=").length != 2){ continue; }

				string key = line.split("=")[0].strip();
				string val = line.split("=")[1].strip();

				if (val.has_prefix("\"")){
					val = val[1:val.length];
				}

				if (val.has_suffix("\"")){
					val = val[0:val.length-1];
				}

				switch (key){
					case "DISTRIB_ID":
						dist_id = val;
						break;
					case "DISTRIB_RELEASE":
						release = val;
						break;
					case "DISTRIB_CODENAME":
						codename = val;
						break;
					case "DISTRIB_DESCRIPTION":
						description = val;
						break;
				}
			}
		}
		else{

			dist_file = path_combine(basepath, "/etc/os-release");
			
			f = File.new_for_path(dist_file);
			
			if (f.query_exists()){

				/*
					NAME="Ubuntu"
					VERSION="13.04, Raring Ringtail"
					ID=ubuntu
					ID_LIKE=debian
					PRETTY_NAME="Ubuntu 13.04"
					VERSION_ID="13.04"
					HOME_URL="http://www.ubuntu.com/"
					SUPPORT_URL="http://help.ubuntu.com/"
					BUG_REPORT_URL="http://bugs.launchpad.net/ubuntu/"
				*/

				foreach(string line in file_read(dist_file).split("\n")){

					if (line.split("=").length != 2){ continue; }

					string key = line.split("=")[0].strip();
					string val = line.split("=")[1].strip();

					switch (key){
						case "ID":
							dist_id = val;
							break;
						case "VERSION_ID":
							release = val;
							break;
						//case "DISTRIB_CODENAME":
							//info.codename = val;
							//break;
						case "PRETTY_NAME":
							description = val;
							break;
					}
				}
			}
		}

		if (dist_id.length > 0){
			string val = "";
			val += dist_id;
			val += (release.length > 0) ? " " + release : "";
			val += (codename.length > 0) ? " (" + codename + ")" : "";
			dist_full_name = val;
		}
	}

	private void check_dist_type(){
			
		if (dist_id == "manjaro"){
			dist_type = "arch";
		}
		else if ((dist_id == "debian")||(dist_id == "ubuntu")){
			dist_type = "debian";
		}
		else if ((dist_id == "fedora")){
			dist_type = "fedora";
		}
	}

	private void check_package_manager(){

		if (cmd_exists_in_basepath("dpkg")){

			if (dist_type.length == 0){
				dist_type = "debian";
			}

			if (cmd_exists_in_basepath("aptitude")){
				package_manager = "aptitude";
			}
			else if (cmd_exists_in_basepath("apt")){
				package_manager = "apt";
			}
			else if (cmd_exists_in_basepath("apt-fast")){
				package_manager = "apt-fast";
			}
			else {
				package_manager = "apt-get";
			}
		}
		else if (cmd_exists_in_basepath("dnf")){

			if (dist_type.length == 0){
				dist_type = "fedora";
			}
			
			package_manager = "dnf";
		}
		else if (cmd_exists_in_basepath("yum")){

			if (dist_type.length == 0){
				dist_type = "fedora";
			}
			
			package_manager = "yum";
		}
		else if (cmd_exists_in_basepath("pacman")){

			if (dist_type.length == 0){
				dist_type = "arch";
			}
			
			package_manager = "pacman";
		}
		else{
			log_error(Messages.UNKNOWN_DISTRO);
			dist_type = "";
			package_manager = "";
		}
	}

	private bool cmd_exists_in_basepath(string cmd){
		
		return cmd_exists_in_path(basepath, cmd);
	}

	public string dist_type_name {
		owned get {
			switch(dist_type){
			case "debian":
				return "Debian / Ubuntu";
			case "fedora":
				return "Fedora / RedHat / Cent OS";
			case "arch":
				return "Arch";
			default:
				return "";
			}
		}
	}

	private void check_package_arch(){

		string cmd = "";
		
		switch(dist_type){
		case "debian":
			cmd = "dpkg --print-architecture"; // i386, amd64, ...
			break;
			
		case "fedora":
			cmd = "uname -m";
			break;
			
		case "arch":
			cmd = "uname -m";  // x86_64, i386, i686, ...
			break;
		}

		if (cmd.length == 0){ return; }
		
		string std_out, std_err;
		exec_sync(cmd, out std_out, out std_err);
		package_arch = std_out.strip();;
	}

	private void check_system(){
		
		string std_out, std_err;
		int status = exec_sync("uname -m", out std_out, out std_err);
		kernel_arch = std_out.strip();

		status = exec_sync("getconf LONG_BIT", out std_out, out std_err);
		machine_arch = int.parse(std_out);
	}

	// public -----------------------------
	
	public void print_system_info(){
		
		//log_msg(string.nfill(70,'-'));
		
		if (dist_full_name.length > 0){
			log_msg("Distribution: %s".printf(dist_full_name));
		}
		
		log_msg("Dist Type: %s".printf((dist_type_name.length > 0) ? dist_type_name : "???"));
		log_msg("Package Manager: %s".printf((package_manager.length > 0) ? package_manager : "???"));

		if (basepath == "/"){
			log_msg("Arch-Pkgs: %s".printf((package_arch.length > 0) ? package_arch : "???"));
			log_msg("Arch-Kern: %s".printf((kernel_arch.length > 0) ? kernel_arch : "???"));
			log_msg("Arch-Type: %s".printf((machine_arch > 0) ? machine_arch.to_string() + "-bit" : "???"));
		}
		//log_msg(string.nfill(70,'-'));
	}
}


