
/*

  KLayout Layout Viewer
  Copyright (C) 2006-2018 Matthias Koefferlein

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "bdReaderOptions.h"
#include "bdWriterOptions.h"
#include "gsiInterpreter.h"
#include "dbLayout.h"
#include "dbGDS2Writer.h"
#include "dbOASISWriter.h"
#include "dbReader.h"
#include "tlLog.h"
#include "tlCommandLineParser.h"
#include "rba.h"
#include "pya.h"
#include "gsi.h"
#include "gsiExpression.h"
#include "libForceLink.h"
#include "rdbForceLink.h"

#include <QFileInfo>

struct RunnerData
{
  std::string script;
  std::vector<std::pair<std::string, std::string> > vars;

  void add_var (const std::string &def)
  {
    std::string var_name, var_value;
    tl::Extractor ex (def.c_str ());
    ex.read_word (var_name);
    if (ex.test ("=")) {
      var_value = ex.get ();
    }

    vars.push_back (std::make_pair (var_name, var_value));
  }
};

BD_PUBLIC int strmrun (int argc, char *argv[])
{
  tl::CommandLineOptions cmd;
  RunnerData data;

  cmd << tl::arg ("script",                     &data.script, "The script to execute",
                  "This script will be executed by the script interpreter. "
                  "The script can be either Ruby (\".rb\") or Python (\".py\")."
                 )
      << tl::arg ("*-v|--var=\"name=value\"", &data, &RunnerData::add_var, "Defines a variable",
                  "When using this option, a global variable with name \"var\" will be defined with the string value \"value\"."
                 )
    ;

  cmd.brief ("This program runs Ruby or Python scripts with a subset of KLayout's API.");

  cmd.parse (argc, argv);

  //  initialize the GSI class system (Variant binding, Expression support)
  //  We have to do this now since plugins may register GSI classes and before the
  //  ruby interpreter, because it depends on a proper class system.
  gsi::initialize ();

  //  initialize the tl::Expression subsystem with GSI-bound classes
  gsi::initialize_expressions ();

  //  create the ruby and python interpreter instances now.
  //  Hint: we do this after load_plugin, because that way the plugins can register GSI classes and methods.
  //  TODO: do this through some auto-registration
  rba::RubyInterpreter ruby;
  pya::PythonInterpreter python;

  for (std::vector< std::pair<std::string, std::string> >::const_iterator v = data.vars.begin (); v != data.vars.end (); ++v) {
    ruby.define_variable (v->first, v->second);
    python.define_variable (v->first, v->second);
  }

  std::string script = tl::to_string (QFileInfo (tl::to_qstring (data.script)).absoluteFilePath ());

  std::string ext = tl::to_string (QFileInfo (tl::to_qstring (data.script)).suffix ());
  if (ext == "py") {
    python.load_file (script);
  } else if (ext == "rb") {
    ruby.load_file (script);
  } else {
    throw tl::Exception (tl::to_string (QObject::tr ("Unknown suffix \"%1\" - must be either .rb or .py").arg (tl::to_qstring (ext))));
  }

  return 0;
}
