// license:GPL-2.0+
// copyright-holders:Jarek Burczynski, Ernesto Corvi
/*
**
** File: ym2413.cpp - software implementation of YM2413
**                    FM sound generator type OPLL
**
** Copyright Jarek Burczynski
**
** Version 1.0
**

   Features as listed in LSI-212413A2 data sheet:
    1. FM Sound Generator for real sound creation.
    2. Two Selectable modes: 9 simultaneous sounds or 6 melody sounds plus 5 rhythm sounds
       (different tones can be used together in either case).
    3. Built-in Instruments data (15 melody tones, 5 rhythm tones, "CAPTAIN and TELETEXT applicalbe tones).
    4. Built-in DA Converter.
    5. Built-in Quartz Oscillator.
    6. Built-in Vibrato Oscillator/AM Oscillator
    7. TTL Compatible Input.
    8. Si-Gate NMOS LSI
    9. A single 5V power source.

to do:

- make sure of the sinus amplitude bits

- make sure of the EG resolution bits (looks like the biggest
  modulation index generated by the modulator is 123, 124 = no modulation)
- find proper algorithm for attack phase of EG

- tune up instruments ROM

- support sample replay in test mode (it is NOT as simple as setting bit 0
  in register 0x0f and using register 0x10 for sample data).
  Which games use this feature ?


*/

#include "emu.h"
#include "ym2413.h"

#include <algorithm>

#define FREQ_SH         16  /* 16.16 fixed point (frequency calculations) */
#define EG_SH           16  /* 16.16 fixed point (EG timing)              */
#define LFO_SH          24  /*  8.24 fixed point (LFO calculations)       */

#define FREQ_MASK       ((1<<FREQ_SH)-1)

/* envelope output entries */
#define ENV_BITS        10
#define ENV_LEN         (1<<ENV_BITS)
#define ENV_STEP        (128.0/ENV_LEN)

#define MAX_ATT_INDEX   ((1<<(ENV_BITS-2))-1) /*255*/
#define MIN_ATT_INDEX   (0)

/* register number to channel number , slot offset */
#define SLOT1 0
#define SLOT2 1

/* Envelope Generator phases */

#define EG_DMP          5
#define EG_ATT          4
#define EG_DEC          3
#define EG_SUS          2
#define EG_REL          1
#define EG_OFF          0

/* key scale level */
/* table is 3dB/octave, DV converts this into 6dB/octave */
/* 0.1875 is bit 0 weight of the envelope counter (volume) expressed in the 'decibel' scale */
#define DV (0.1875/1.0)
const double ym2413_device::ksl_tab[8*16] =
{
	/* OCT 0 */
		0.000/DV, 0.000/DV, 0.000/DV, 0.000/DV,
		0.000/DV, 0.000/DV, 0.000/DV, 0.000/DV,
		0.000/DV, 0.000/DV, 0.000/DV, 0.000/DV,
		0.000/DV, 0.000/DV, 0.000/DV, 0.000/DV,
	/* OCT 1 */
		0.000/DV, 0.000/DV, 0.000/DV, 0.000/DV,
		0.000/DV, 0.000/DV, 0.000/DV, 0.000/DV,
		0.000/DV, 0.750/DV, 1.125/DV, 1.500/DV,
		1.875/DV, 2.250/DV, 2.625/DV, 3.000/DV,
	/* OCT 2 */
		0.000/DV, 0.000/DV, 0.000/DV, 0.000/DV,
		0.000/DV, 1.125/DV, 1.875/DV, 2.625/DV,
		3.000/DV, 3.750/DV, 4.125/DV, 4.500/DV,
		4.875/DV, 5.250/DV, 5.625/DV, 6.000/DV,
	/* OCT 3 */
		0.000/DV, 0.000/DV, 0.000/DV, 1.875/DV,
		3.000/DV, 4.125/DV, 4.875/DV, 5.625/DV,
		6.000/DV, 6.750/DV, 7.125/DV, 7.500/DV,
		7.875/DV, 8.250/DV, 8.625/DV, 9.000/DV,
	/* OCT 4 */
		0.000/DV, 0.000/DV, 3.000/DV, 4.875/DV,
		6.000/DV, 7.125/DV, 7.875/DV, 8.625/DV,
		9.000/DV, 9.750/DV,10.125/DV,10.500/DV,
		10.875/DV,11.250/DV,11.625/DV,12.000/DV,
	/* OCT 5 */
		0.000/DV, 3.000/DV, 6.000/DV, 7.875/DV,
		9.000/DV,10.125/DV,10.875/DV,11.625/DV,
		12.000/DV,12.750/DV,13.125/DV,13.500/DV,
		13.875/DV,14.250/DV,14.625/DV,15.000/DV,
	/* OCT 6 */
		0.000/DV, 6.000/DV, 9.000/DV,10.875/DV,
		12.000/DV,13.125/DV,13.875/DV,14.625/DV,
		15.000/DV,15.750/DV,16.125/DV,16.500/DV,
		16.875/DV,17.250/DV,17.625/DV,18.000/DV,
	/* OCT 7 */
		0.000/DV, 9.000/DV,12.000/DV,13.875/DV,
		15.000/DV,16.125/DV,16.875/DV,17.625/DV,
		18.000/DV,18.750/DV,19.125/DV,19.500/DV,
		19.875/DV,20.250/DV,20.625/DV,21.000/DV
};
#undef DV

/* 0 / 1.5 / 3.0 / 6.0 dB/OCT, confirmed on a real YM2413 (the application manual is incorrect) */
const uint32_t ym2413_device::ksl_shift[4] = { 31, 2, 1, 0 };


/* sustain level table (3dB per step) */
/* 0 - 15: 0, 3, 6, 9,12,15,18,21,24,27,30,33,36,39,42,45 (dB)*/
#define SC(db) (uint32_t) ( db * (1.0/ENV_STEP) )
const uint32_t ym2413_device::sl_tab[16] = {
	SC( 0),SC( 1),SC( 2),SC(3 ),SC(4 ),SC(5 ),SC(6 ),SC( 7),
	SC( 8),SC( 9),SC(10),SC(11),SC(12),SC(13),SC(14),SC(15)
};
#undef SC

const uint8_t ym2413_device::eg_inc[15*RATE_STEPS] = {
	/*cycle:0 1  2 3  4 5  6 7*/

	/* 0 */ 0,1, 0,1, 0,1, 0,1, /* rates 00..12 0 (increment by 0 or 1) */
	/* 1 */ 0,1, 0,1, 1,1, 0,1, /* rates 00..12 1 */
	/* 2 */ 0,1, 1,1, 0,1, 1,1, /* rates 00..12 2 */
	/* 3 */ 0,1, 1,1, 1,1, 1,1, /* rates 00..12 3 */

	/* 4 */ 1,1, 1,1, 1,1, 1,1, /* rate 13 0 (increment by 1) */
	/* 5 */ 1,1, 1,2, 1,1, 1,2, /* rate 13 1 */
	/* 6 */ 1,2, 1,2, 1,2, 1,2, /* rate 13 2 */
	/* 7 */ 1,2, 2,2, 1,2, 2,2, /* rate 13 3 */

	/* 8 */ 2,2, 2,2, 2,2, 2,2, /* rate 14 0 (increment by 2) */
	/* 9 */ 2,2, 2,4, 2,2, 2,4, /* rate 14 1 */
	/*10 */ 2,4, 2,4, 2,4, 2,4, /* rate 14 2 */
	/*11 */ 2,4, 4,4, 2,4, 4,4, /* rate 14 3 */

	/*12 */ 4,4, 4,4, 4,4, 4,4, /* rates 15 0, 15 1, 15 2, 15 3 (increment by 4) */
	/*13 */ 8,8, 8,8, 8,8, 8,8, /* rates 15 2, 15 3 for attack */
	/*14 */ 0,0, 0,0, 0,0, 0,0, /* infinity rates for attack and decay(s) */
};


#define O(a) (a*RATE_STEPS)

/*note that there is no O(13) in this table - it's directly in the code */
const uint8_t ym2413_device::eg_rate_select[16+64+16] = {   /* Envelope Generator rates (16 + 64 rates + 16 RKS) */
	/* 16 infinite time rates */
	O(14),O(14),O(14),O(14),O(14),O(14),O(14),O(14),
	O(14),O(14),O(14),O(14),O(14),O(14),O(14),O(14),

	/* rates 00-12 */
	O( 0),O( 1),O( 2),O( 3),
	O( 0),O( 1),O( 2),O( 3),
	O( 0),O( 1),O( 2),O( 3),
	O( 0),O( 1),O( 2),O( 3),
	O( 0),O( 1),O( 2),O( 3),
	O( 0),O( 1),O( 2),O( 3),
	O( 0),O( 1),O( 2),O( 3),
	O( 0),O( 1),O( 2),O( 3),
	O( 0),O( 1),O( 2),O( 3),
	O( 0),O( 1),O( 2),O( 3),
	O( 0),O( 1),O( 2),O( 3),
	O( 0),O( 1),O( 2),O( 3),
	O( 0),O( 1),O( 2),O( 3),

	/* rate 13 */
	O( 4),O( 5),O( 6),O( 7),

	/* rate 14 */
	O( 8),O( 9),O(10),O(11),

	/* rate 15 */
	O(12),O(12),O(12),O(12),

	/* 16 dummy rates (same as 15 3) */
	O(12),O(12),O(12),O(12),O(12),O(12),O(12),O(12),
	O(12),O(12),O(12),O(12),O(12),O(12),O(12),O(12),

};
#undef O

/*rate  0,    1,    2,    3,    4,   5,   6,   7,  8,  9, 10, 11, 12, 13, 14, 15 */
/*shift 13,   12,   11,   10,   9,   8,   7,   6,  5,  4,  3,  2,  1,  0,  0,  0 */
/*mask  8191, 4095, 2047, 1023, 511, 255, 127, 63, 31, 15, 7,  3,  1,  0,  0,  0 */

#define O(a) (a*1)
const uint8_t ym2413_device::eg_rate_shift[16+64+16] = {    /* Envelope Generator counter shifts (16 + 64 rates + 16 RKS) */
	/* 16 infinite time rates */
	O(0),O(0),O(0),O(0),O(0),O(0),O(0),O(0),
	O(0),O(0),O(0),O(0),O(0),O(0),O(0),O(0),

	/* rates 00-12 */
	O(13),O(13),O(13),O(13),
	O(12),O(12),O(12),O(12),
	O(11),O(11),O(11),O(11),
	O(10),O(10),O(10),O(10),
	O( 9),O( 9),O( 9),O( 9),
	O( 8),O( 8),O( 8),O( 8),
	O( 7),O( 7),O( 7),O( 7),
	O( 6),O( 6),O( 6),O( 6),
	O( 5),O( 5),O( 5),O( 5),
	O( 4),O( 4),O( 4),O( 4),
	O( 3),O( 3),O( 3),O( 3),
	O( 2),O( 2),O( 2),O( 2),
	O( 1),O( 1),O( 1),O( 1),

	/* rate 13 */
	O( 0),O( 0),O( 0),O( 0),

	/* rate 14 */
	O( 0),O( 0),O( 0),O( 0),

	/* rate 15 */
	O( 0),O( 0),O( 0),O( 0),

	/* 16 dummy rates (same as 15 3) */
	O( 0),O( 0),O( 0),O( 0),O( 0),O( 0),O( 0),O( 0),
	O( 0),O( 0),O( 0),O( 0),O( 0),O( 0),O( 0),O( 0),
};
#undef O


/* multiple table */
#define ML 2
const uint8_t ym2413_device::mul_tab[16]= {
	/* 1/2, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,10,12,12,15,15 */
	ML/2, 1*ML, 2*ML, 3*ML, 4*ML, 5*ML, 6*ML, 7*ML,
	8*ML, 9*ML,10*ML,10*ML,12*ML,12*ML,15*ML,15*ML
};
#undef ML


#define ENV_QUIET       (TL_TAB_LEN>>5)


/* LFO Amplitude Modulation table (verified on real YM3812)
   27 output levels (triangle waveform); 1 level takes one of: 192, 256 or 448 samples

   Length: 210 elements.

    Each of the elements has to be repeated
    exactly 64 times (on 64 consecutive samples).
    The whole table takes: 64 * 210 = 13440 samples.

We use data>>1, until we find what it really is on real chip...

*/
const uint8_t ym2413_device::lfo_am_table[LFO_AM_TAB_ELEMENTS] = {
	0,0,0,0,0,0,0,
	1,1,1,1,
	2,2,2,2,
	3,3,3,3,
	4,4,4,4,
	5,5,5,5,
	6,6,6,6,
	7,7,7,7,
	8,8,8,8,
	9,9,9,9,
	10,10,10,10,
	11,11,11,11,
	12,12,12,12,
	13,13,13,13,
	14,14,14,14,
	15,15,15,15,
	16,16,16,16,
	17,17,17,17,
	18,18,18,18,
	19,19,19,19,
	20,20,20,20,
	21,21,21,21,
	22,22,22,22,
	23,23,23,23,
	24,24,24,24,
	25,25,25,25,
	26,26,26,
	25,25,25,25,
	24,24,24,24,
	23,23,23,23,
	22,22,22,22,
	21,21,21,21,
	20,20,20,20,
	19,19,19,19,
	18,18,18,18,
	17,17,17,17,
	16,16,16,16,
	15,15,15,15,
	14,14,14,14,
	13,13,13,13,
	12,12,12,12,
	11,11,11,11,
	10,10,10,10,
	9,9,9,9,
	8,8,8,8,
	7,7,7,7,
	6,6,6,6,
	5,5,5,5,
	4,4,4,4,
	3,3,3,3,
	2,2,2,2,
	1,1,1,1
};

/* LFO Phase Modulation table (verified on real YM2413) */
const int8_t ym2413_device::lfo_pm_table[8*8] = {
	/* FNUM2/FNUM = 0 00xxxxxx (0x0000) */
	0, 0, 0, 0, 0, 0, 0, 0,

	/* FNUM2/FNUM = 0 01xxxxxx (0x0040) */
	1, 0, 0, 0,-1, 0, 0, 0,

	/* FNUM2/FNUM = 0 10xxxxxx (0x0080) */
	2, 1, 0,-1,-2,-1, 0, 1,

	/* FNUM2/FNUM = 0 11xxxxxx (0x00C0) */
	3, 1, 0,-1,-3,-1, 0, 1,

	/* FNUM2/FNUM = 1 00xxxxxx (0x0100) */
	4, 2, 0,-2,-4,-2, 0, 2,

	/* FNUM2/FNUM = 1 01xxxxxx (0x0140) */
	5, 2, 0,-2,-5,-2, 0, 2,

	/* FNUM2/FNUM = 1 10xxxxxx (0x0180) */
	6, 3, 0,-3,-6,-3, 0, 3,

	/* FNUM2/FNUM = 1 11xxxxxx (0x01C0) */
	7, 3, 0,-3,-7,-3, 0, 3,
};


/* This is not 100% perfect yet but very close */
/*
 - multi parameters are 100% correct (instruments and drums)
 - LFO PM and AM enable are 100% correct
 - waveform DC and DM select are 100% correct
*/

const uint8_t ym2413_device::table[19][8] = {
/* MULT  MULT modTL DcDmFb AR/DR AR/DR SL/RR SL/RR */
/*   0     1     2     3     4     5     6    7    */
/* These YM2413(OPLL) patch dumps are done via audio analysis (and a/b testing?) from Jarek and are known to be inaccurate */
	{0x49, 0x4c, 0x4c, 0x12, 0x00, 0x00, 0x00, 0x00 },  //0

	{0x61, 0x61, 0x1e, 0x17, 0xf0, 0x78, 0x00, 0x17 },  //1
	{0x13, 0x41, 0x1e, 0x0d, 0xd7, 0xf7, 0x13, 0x13 },  //2
	{0x13, 0x01, 0x99, 0x04, 0xf2, 0xf4, 0x11, 0x23 },  //3
	{0x21, 0x61, 0x1b, 0x07, 0xaf, 0x64, 0x40, 0x27 },  //4

//{0x22, 0x21, 0x1e, 0x09, 0xf0, 0x76, 0x08, 0x28 },    //5
	{0x22, 0x21, 0x1e, 0x06, 0xf0, 0x75, 0x08, 0x18 },  //5

//{0x31, 0x22, 0x16, 0x09, 0x90, 0x7f, 0x00, 0x08 },    //6
	{0x31, 0x22, 0x16, 0x05, 0x90, 0x71, 0x00, 0x13 },  //6

	{0x21, 0x61, 0x1d, 0x07, 0x82, 0x80, 0x10, 0x17 },  //7
	{0x23, 0x21, 0x2d, 0x16, 0xc0, 0x70, 0x07, 0x07 },  //8
	{0x61, 0x61, 0x1b, 0x06, 0x64, 0x65, 0x10, 0x17 },  //9

//{0x61, 0x61, 0x0c, 0x08, 0x85, 0xa0, 0x79, 0x07 },    //A
	{0x61, 0x61, 0x0c, 0x18, 0x85, 0xf0, 0x70, 0x07 },  //A

	{0x23, 0x01, 0x07, 0x11, 0xf0, 0xa4, 0x00, 0x22 },  //B
	{0x97, 0xc1, 0x24, 0x07, 0xff, 0xf8, 0x22, 0x12 },  //C

//{0x61, 0x10, 0x0c, 0x08, 0xf2, 0xc4, 0x40, 0xc8 },    //D
	{0x61, 0x10, 0x0c, 0x05, 0xf2, 0xf4, 0x40, 0x44 },  //D

	{0x01, 0x01, 0x55, 0x03, 0xf3, 0x92, 0xf3, 0xf3 },  //E
	{0x61, 0x41, 0x89, 0x03, 0xf1, 0xf4, 0xf0, 0x13 },  //F

/* drum instruments definitions */
/* MULTI MULTI modTL  xxx  AR/DR AR/DR SL/RR SL/RR */
/*   0     1     2     3     4     5     6    7    */
/* old dumps via audio analysis (and a/b testing?) from Jarek */
//{0x01, 0x01, 0x16, 0x00, 0xfd, 0xf8, 0x2f, 0x6d },/* BD(multi verified, modTL verified, mod env - verified(close), carr. env verifed) */
//{0x01, 0x01, 0x00, 0x00, 0xd8, 0xd8, 0xf9, 0xf8 },/* HH(multi verified), SD(multi not used) */
//{0x05, 0x01, 0x00, 0x00, 0xf8, 0xba, 0x49, 0x55 },/* TOM(multi,env verified), TOP CYM(multi verified, env verified) */
/* Drums dumped from the VRC7 using debug mode, these are likely also correct for ym2413(OPLL) but need verification */
	{0x01, 0x01, 0x18, 0x0f, 0xdf, 0xf8, 0x6a, 0x6d },/* BD */
	{0x01, 0x01, 0x00, 0x00, 0xc8, 0xd8, 0xa7, 0x68 },/* HH, SD */
	{0x05, 0x01, 0x00, 0x00, 0xf8, 0xaa, 0x59, 0x55 },/* TOM, TOP CYM */
};

// VRC7 Instruments : Dumped from internal ROM
// reference : https://siliconpr0n.org/archive/doku.php?id=vendor:yamaha:opl2
const uint8_t vrc7snd_device::vrc7_table[19][8] = {
/* MULT  MULT modTL DcDmFb AR/DR AR/DR SL/RR SL/RR */
/*   0     1     2     3     4     5     6    7    */
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },  //0 (This is the user-defined instrument, should this default to anything?)

	{0x03, 0x21, 0x05, 0x06, 0xe8, 0x81, 0x42, 0x27 },  //1
	{0x13, 0x41, 0x14, 0x0d, 0xd8, 0xf6, 0x23, 0x12 },  //2
	{0x11, 0x11, 0x08, 0x08, 0xfa, 0xb2, 0x20, 0x12 },  //3
	{0x31, 0x61, 0x0c, 0x07, 0xa8, 0x64, 0x61, 0x27 },  //4
	{0x32, 0x21, 0x1e, 0x06, 0xe1, 0x76, 0x01, 0x28 },  //5
	{0x02, 0x01, 0x06, 0x00, 0xa3, 0xe2, 0xf4, 0xf4 },  //6
	{0x21, 0x61, 0x1d, 0x07, 0x82, 0x81, 0x11, 0x07 },  //7
	{0x23, 0x21, 0x22, 0x17, 0xa2, 0x72, 0x01, 0x17 },  //8
	{0x35, 0x11, 0x25, 0x00, 0x40, 0x73, 0x72, 0x01 },  //9
	{0xb5, 0x01, 0x0f, 0x0f, 0xa8, 0xa5, 0x51, 0x02 },  //A
	{0x17, 0xc1, 0x24, 0x07, 0xf8, 0xf8, 0x22, 0x12 },  //B
	{0x71, 0x23, 0x11, 0x06, 0x65, 0x74, 0x18, 0x16 },  //C
	{0x01, 0x02, 0xd3, 0x05, 0xc9, 0x95, 0x03, 0x02 },  //D
	{0x61, 0x63, 0x0c, 0x00, 0x94, 0xc0, 0x33, 0xf6 },  //E
	{0x21, 0x72, 0x0d, 0x00, 0xc1, 0xd5, 0x56, 0x06 },  //F

/* Drums (silent due to no RO output pin(?) on VRC7, but present internally; these are probably shared with YM2413) */
/* MULTI MULTI modTL  xxx  AR/DR AR/DR SL/RR SL/RR */
/*   0     1     2     3     4     5     6    7    */
	{0x01, 0x01, 0x18, 0x0f, 0xdf, 0xf8, 0x6a, 0x6d },/* BD */
	{0x01, 0x01, 0x00, 0x00, 0xc8, 0xd8, 0xa7, 0x68 },/* HH, SD */
	{0x05, 0x01, 0x00, 0x00, 0xf8, 0xaa, 0x59, 0x55 },/* TOM, TOP CYM */
};

/* work table */
#define SLOT7_1 (&P_CH[7].SLOT[SLOT1])
#define SLOT7_2 (&P_CH[7].SLOT[SLOT2])
#define SLOT8_1 (&P_CH[8].SLOT[SLOT1])
#define SLOT8_2 (&P_CH[8].SLOT[SLOT2])


int ym2413_device::limit( int val, int max, int min )
{
	if ( val > max )
		val = max;
	else if ( val < min )
		val = min;

	return val;
}


/* advance LFO to next sample */
void ym2413_device::advance_lfo()
{
	/* LFO */
	lfo_am_cnt += lfo_am_inc;
	if (lfo_am_cnt >= ((uint32_t)LFO_AM_TAB_ELEMENTS<<LFO_SH) ) /* lfo_am_table is 210 elements long */
		lfo_am_cnt -= ((uint32_t)LFO_AM_TAB_ELEMENTS<<LFO_SH);

	LFO_AM = lfo_am_table[ lfo_am_cnt >> LFO_SH ] >> 1;

	lfo_pm_cnt += lfo_pm_inc;
	LFO_PM = (lfo_pm_cnt>>LFO_SH) & 7;
}

/* advance to next sample */
void ym2413_device::advance()
{
	OPLL_CH *CH;
	OPLL_SLOT *op;
	unsigned int i;

	/* Envelope Generator */
	eg_timer += eg_timer_add;

	while (eg_timer >= eg_timer_overflow)
	{
		eg_timer -= eg_timer_overflow;

		eg_cnt++;

		for (i=0; i<9*2; i++)
		{
			CH  = &P_CH[i/2];

			op  = &CH->SLOT[i&1];

			switch(op->state)
			{
			case EG_DMP:        /* dump phase */
			/*dump phase is performed by both operators in each channel*/
			/*when CARRIER envelope gets down to zero level,
			**  phases in BOTH opearators are reset (at the same time ?)
			*/
				if ( !(eg_cnt & ((1<<op->eg_sh_dp)-1) ) )
				{
					op->volume += eg_inc[op->eg_sel_dp + ((eg_cnt>>op->eg_sh_dp)&7)];

					if ( op->volume >= MAX_ATT_INDEX )
					{
						op->volume = MAX_ATT_INDEX;
						op->state = EG_ATT;
						/* restart Phase Generator  */
						op->phase = 0;
					}
				}
			break;

			case EG_ATT:        /* attack phase */
				if ( !(eg_cnt & ((1<<op->eg_sh_ar)-1) ) )
				{
					op->volume += (~op->volume *
												(eg_inc[op->eg_sel_ar + ((eg_cnt>>op->eg_sh_ar)&7)])
												) >>2;

					if (op->volume <= MIN_ATT_INDEX)
					{
						op->volume = MIN_ATT_INDEX;
						op->state = EG_DEC;
					}
				}
			break;

			case EG_DEC:    /* decay phase */
				if ( !(eg_cnt & ((1<<op->eg_sh_dr)-1) ) )
				{
					op->volume += eg_inc[op->eg_sel_dr + ((eg_cnt>>op->eg_sh_dr)&7)];

					if ( op->volume >= op->sl )
						op->state = EG_SUS;
				}
			break;

			case EG_SUS:    /* sustain phase */
				/* this is important behaviour:
				one can change percusive/non-percussive modes on the fly and
				the chip will remain in sustain phase - verified on real YM3812 */

				if(op->eg_type)     /* non-percussive mode (sustained tone) */
				{
									/* do nothing */
				}
				else                /* percussive mode */
				{
					/* during sustain phase chip adds Release Rate (in percussive mode) */
					if ( !(eg_cnt & ((1<<op->eg_sh_rr)-1) ) )
					{
						op->volume += eg_inc[op->eg_sel_rr + ((eg_cnt>>op->eg_sh_rr)&7)];

						if ( op->volume >= MAX_ATT_INDEX )
							op->volume = MAX_ATT_INDEX;
					}
					/* else do nothing in sustain phase */
				}
			break;

			case EG_REL:    /* release phase */
			/* exclude modulators in melody channels from performing anything in this mode*/
			/* allowed are only carriers in melody mode and rhythm slots in rhythm mode */

			/*This table shows which operators and on what conditions are allowed to perform EG_REL:
			(a) - always perform EG_REL
			(n) - never perform EG_REL
			(r) - perform EG_REL in Rhythm mode ONLY
			    0: 0 (n),  1 (a)
			    1: 2 (n),  3 (a)
			    2: 4 (n),  5 (a)
			    3: 6 (n),  7 (a)
			    4: 8 (n),  9 (a)
			    5: 10(n),  11(a)
			    6: 12(r),  13(a)
			    7: 14(r),  15(a)
			    8: 16(r),  17(a)
			*/
				if ( (i&1) || ((rhythm&0x20) && (i>=12)) )/* exclude modulators */
				{
					if(op->eg_type)     /* non-percussive mode (sustained tone) */
					/*this is correct: use RR when SUS = OFF*/
					/*and use RS when SUS = ON*/
					{
						if (CH->sus)
						{
							if ( !(eg_cnt & ((1<<op->eg_sh_rs)-1) ) )
							{
								op->volume += eg_inc[op->eg_sel_rs + ((eg_cnt>>op->eg_sh_rs)&7)];
								if ( op->volume >= MAX_ATT_INDEX )
								{
									op->volume = MAX_ATT_INDEX;
									op->state = EG_OFF;
								}
							}
						}
						else
						{
							if ( !(eg_cnt & ((1<<op->eg_sh_rr)-1) ) )
							{
								op->volume += eg_inc[op->eg_sel_rr + ((eg_cnt>>op->eg_sh_rr)&7)];
								if ( op->volume >= MAX_ATT_INDEX )
								{
									op->volume = MAX_ATT_INDEX;
									op->state = EG_OFF;
								}
							}
						}
					}
					else                /* percussive mode */
					{
						if ( !(eg_cnt & ((1<<op->eg_sh_rs)-1) ) )
						{
							op->volume += eg_inc[op->eg_sel_rs + ((eg_cnt>>op->eg_sh_rs)&7)];
							if ( op->volume >= MAX_ATT_INDEX )
							{
								op->volume = MAX_ATT_INDEX;
								op->state = EG_OFF;
							}
						}
					}
				}
			break;

			default:
			break;
			}
		}
	}

	for (i=0; i<9*2; i++)
	{
		CH  = &P_CH[i/2];
		op  = &CH->SLOT[i&1];

		/* Phase Generator */
		if(op->vib)
		{
			uint8_t block;

			unsigned int fnum_lfo   = 8*((CH->block_fnum&0x01c0) >> 6);
			unsigned int block_fnum = CH->block_fnum * 2;
			signed int lfo_fn_table_index_offset = lfo_pm_table[LFO_PM + fnum_lfo ];

			if (lfo_fn_table_index_offset)  /* LFO phase modulation active */
			{
				block_fnum += lfo_fn_table_index_offset;
				block = (block_fnum&0x1c00) >> 10;
				op->phase += (fn_tab[block_fnum&0x03ff] >> (7-block)) * op->mul;
			}
			else    /* LFO phase modulation  = zero */
			{
				op->phase += op->freq;
			}
		}
		else    /* LFO phase modulation disabled for this operator */
		{
			op->phase += op->freq;
		}
	}

	/*  The Noise Generator of the YM3812 is 23-bit shift register.
	*   Period is equal to 2^23-2 samples.
	*   Register works at sampling frequency of the chip, so output
	*   can change on every sample.
	*
	*   Output of the register and input to the bit 22 is:
	*   bit0 XOR bit14 XOR bit15 XOR bit22
	*
	*   Simply use bit 22 as the noise output.
	*/

	noise_p += noise_f;
	i = noise_p >> FREQ_SH;       /* number of events (shifts of the shift register) */
	noise_p &= FREQ_MASK;
	while (i)
	{
		/*
		uint32_t j;
		j = ( (noise_rng) ^ (noise_rng>>14) ^ (noise_rng>>15) ^ (noise_rng>>22) ) & 1;
		noise_rng = (j<<22) | (noise_rng>>1);
		*/

		/*
		    Instead of doing all the logic operations above, we
		    use a trick here (and use bit 0 as the noise output).
		    The difference is only that the noise bit changes one
		    step ahead. This doesn't matter since we don't know
		    what is real state of the noise_rng after the reset.
		*/

		if (noise_rng & 1) noise_rng ^= 0x800302;
		noise_rng >>= 1;

		i--;
	}
}


int ym2413_device::op_calc(uint32_t phase, unsigned int env, signed int pm, unsigned int wave_tab)
{
	uint32_t p;

	p = (env<<5) + sin_tab[wave_tab + ((((signed int)((phase & ~FREQ_MASK) + (pm<<17))) >> FREQ_SH ) & SIN_MASK) ];

	if (p >= TL_TAB_LEN)
		return 0;
	return tl_tab[p];
}

int ym2413_device::op_calc1(uint32_t phase, unsigned int env, signed int pm, unsigned int wave_tab)
{
	uint32_t p;
	int32_t  i;

	i = (phase & ~FREQ_MASK) + pm;

/*logerror("i=%08x (i>>16)&511=%8i phase=%i [pm=%08x] ",i, (i>>16)&511, phase>>FREQ_SH, pm);*/

	p = (env<<5) + sin_tab[ wave_tab + ((i>>FREQ_SH) & SIN_MASK)];

/*logerror("(p&255=%i p>>8=%i) out= %i\n", p&255,p>>8, tl_tab[p&255]>>(p>>8) );*/

	if (p >= TL_TAB_LEN)
		return 0;
	return tl_tab[p];
}


#define volume_calc(OP) ((OP)->TLL + ((uint32_t)(OP)->volume) + (LFO_AM & (OP)->AMmask))

/* calculate output */
void ym2413_device::chan_calc( OPLL_CH *CH )
{
	OPLL_SLOT *SLOT;
	unsigned int env;
	signed int out;
	signed int phase_modulation;    /* phase modulation input (SLOT 2) */


	/* SLOT 1 */
	SLOT = &CH->SLOT[SLOT1];
	env  = volume_calc(SLOT);
	out  = SLOT->op1_out[0] + SLOT->op1_out[1];

	SLOT->op1_out[0] = SLOT->op1_out[1];
	phase_modulation = SLOT->op1_out[0];

	SLOT->op1_out[1] = 0;

	if( env < ENV_QUIET )
	{
		if (!SLOT->fb_shift)
			out = 0;
		SLOT->op1_out[1] = op_calc1(SLOT->phase, env, (out<<SLOT->fb_shift), SLOT->wavetable );
	}

	/* SLOT 2 */
	SLOT++;
	env = volume_calc(SLOT);
	if( env < ENV_QUIET )
	{
		output[0] += op_calc(SLOT->phase, env, phase_modulation, SLOT->wavetable);
	}
}

/*
    operators used in the rhythm sounds generation process:

    Envelope Generator:

channel  operator  register number   Bass  High  Snare Tom  Top
/ slot   number    TL ARDR SLRR Wave Drum  Hat   Drum  Tom  Cymbal
 6 / 0   12        50  70   90   f0  +
 6 / 1   15        53  73   93   f3  +
 7 / 0   13        51  71   91   f1        +
 7 / 1   16        54  74   94   f4              +
 8 / 0   14        52  72   92   f2                    +
 8 / 1   17        55  75   95   f5                          +

    Phase Generator:

channel  operator  register number   Bass  High  Snare Tom  Top
/ slot   number    MULTIPLE          Drum  Hat   Drum  Tom  Cymbal
 6 / 0   12        30                +
 6 / 1   15        33                +
 7 / 0   13        31                      +     +           +
 7 / 1   16        34                -----  n o t  u s e d -----
 8 / 0   14        32                                  +
 8 / 1   17        35                      +                 +

channel  operator  register number   Bass  High  Snare Tom  Top
number   number    BLK/FNUM2 FNUM    Drum  Hat   Drum  Tom  Cymbal
   6     12,15     B6        A6      +

   7     13,16     B7        A7            +     +           +

   8     14,17     B8        A8            +           +     +

*/

/* calculate rhythm */

void ym2413_device::rhythm_calc( OPLL_CH *CH, unsigned int noise )
{
	OPLL_SLOT *SLOT;
	signed int out;
	unsigned int env;
	signed int phase_modulation;    /* phase modulation input (SLOT 2) */


	/* Bass Drum (verified on real YM3812):
	  - depends on the channel 6 'connect' register:
	      when connect = 0 it works the same as in normal (non-rhythm) mode (op1->op2->out)
	      when connect = 1 _only_ operator 2 is present on output (op2->out), operator 1 is ignored
	  - output sample always is multiplied by 2
	*/


	/* SLOT 1 */
	SLOT = &CH[6].SLOT[SLOT1];
	env = volume_calc(SLOT);

	out = SLOT->op1_out[0] + SLOT->op1_out[1];
	SLOT->op1_out[0] = SLOT->op1_out[1];

	phase_modulation = SLOT->op1_out[0];

	SLOT->op1_out[1] = 0;
	if( env < ENV_QUIET )
	{
		if (!SLOT->fb_shift)
			out = 0;
		SLOT->op1_out[1] = op_calc1(SLOT->phase, env, (out<<SLOT->fb_shift), SLOT->wavetable );
	}

	/* SLOT 2 */
	SLOT++;
	env = volume_calc(SLOT);
	if( env < ENV_QUIET )
		output[1] += op_calc(SLOT->phase, env, phase_modulation, SLOT->wavetable) * 2;


	/* Phase generation is based on: */
	// HH  (13) channel 7->slot 1 combined with channel 8->slot 2 (same combination as TOP CYMBAL but different output phases)
	// SD  (16) channel 7->slot 1
	// TOM (14) channel 8->slot 1
	// TOP (17) channel 7->slot 1 combined with channel 8->slot 2 (same combination as HIGH HAT but different output phases)

	/* Envelope generation based on: */
	// HH  channel 7->slot1
	// SD  channel 7->slot2
	// TOM channel 8->slot1
	// TOP channel 8->slot2


	/* The following formulas can be well optimized.
	   I leave them in direct form for now (in case I've missed something).
	*/

	/* High Hat (verified on real YM3812) */
	env = volume_calc(SLOT7_1);
	if( env < ENV_QUIET )
	{
		/* high hat phase generation:
		    phase = d0 or 234 (based on frequency only)
		    phase = 34 or 2d0 (based on noise)
		*/

		/* base frequency derived from operator 1 in channel 7 */
		uint8_t bit7 = ((SLOT7_1->phase>>FREQ_SH)>>7)&1;
		uint8_t bit3 = ((SLOT7_1->phase>>FREQ_SH)>>3)&1;
		uint8_t bit2 = ((SLOT7_1->phase>>FREQ_SH)>>2)&1;

		uint8_t res1 = (bit2 ^ bit7) | bit3;

		/* when res1 = 0 phase = 0x000 | 0xd0; */
		/* when res1 = 1 phase = 0x200 | (0xd0>>2); */
		uint32_t phase = res1 ? (0x200|(0xd0>>2)) : 0xd0;

		/* enable gate based on frequency of operator 2 in channel 8 */
		uint8_t bit5e= ((SLOT8_2->phase>>FREQ_SH)>>5)&1;
		uint8_t bit3e= ((SLOT8_2->phase>>FREQ_SH)>>3)&1;

		uint8_t res2 = (bit3e | bit5e);

		/* when res2 = 0 pass the phase from calculation above (res1); */
		/* when res2 = 1 phase = 0x200 | (0xd0>>2); */
		if (res2)
			phase = (0x200|(0xd0>>2));


		/* when phase & 0x200 is set and noise=1 then phase = 0x200|0xd0 */
		/* when phase & 0x200 is set and noise=0 then phase = 0x200|(0xd0>>2), ie no change */
		if (phase&0x200)
		{
			if (noise)
				phase = 0x200|0xd0;
		}
		else
		/* when phase & 0x200 is clear and noise=1 then phase = 0xd0>>2 */
		/* when phase & 0x200 is clear and noise=0 then phase = 0xd0, ie no change */
		{
			if (noise)
				phase = 0xd0>>2;
		}

		output[1] += op_calc(phase<<FREQ_SH, env, 0, SLOT7_1->wavetable) * 2;
	}

	/* Snare Drum (verified on real YM3812) */
	env = volume_calc(SLOT7_2);
	if( env < ENV_QUIET )
	{
		/* base frequency derived from operator 1 in channel 7 */
		uint8_t bit8 = ((SLOT7_1->phase>>FREQ_SH)>>8)&1;

		/* when bit8 = 0 phase = 0x100; */
		/* when bit8 = 1 phase = 0x200; */
		uint32_t phase = bit8 ? 0x200 : 0x100;

		/* Noise bit XOR'es phase by 0x100 */
		/* when noisebit = 0 pass the phase from calculation above */
		/* when noisebit = 1 phase ^= 0x100; */
		/* in other words: phase ^= (noisebit<<8); */
		if (noise)
			phase ^= 0x100;

		output[1] += op_calc(phase<<FREQ_SH, env, 0, SLOT7_2->wavetable) * 2;
	}

	/* Tom Tom (verified on real YM3812) */
	env = volume_calc(SLOT8_1);
	if( env < ENV_QUIET )
		output[1] += op_calc(SLOT8_1->phase, env, 0, SLOT8_1->wavetable) * 2;

	/* Top Cymbal (verified on real YM2413) */
	env = volume_calc(SLOT8_2);
	if( env < ENV_QUIET )
	{
		/* base frequency derived from operator 1 in channel 7 */
		uint8_t bit7 = ((SLOT7_1->phase>>FREQ_SH)>>7)&1;
		uint8_t bit3 = ((SLOT7_1->phase>>FREQ_SH)>>3)&1;
		uint8_t bit2 = ((SLOT7_1->phase>>FREQ_SH)>>2)&1;

		uint8_t res1 = (bit2 ^ bit7) | bit3;

		/* when res1 = 0 phase = 0x000 | 0x100; */
		/* when res1 = 1 phase = 0x200 | 0x100; */
		uint32_t phase = res1 ? 0x300 : 0x100;

		/* enable gate based on frequency of operator 2 in channel 8 */
		uint8_t bit5e= ((SLOT8_2->phase>>FREQ_SH)>>5)&1;
		uint8_t bit3e= ((SLOT8_2->phase>>FREQ_SH)>>3)&1;

		uint8_t res2 = (bit3e | bit5e);
		/* when res2 = 0 pass the phase from calculation above (res1); */
		/* when res2 = 1 phase = 0x200 | 0x100; */
		if (res2)
			phase = 0x300;

		output[1] += op_calc(phase<<FREQ_SH, env, 0, SLOT8_2->wavetable) * 2;
	}

}

void ym2413_device::key_on(OPLL_SLOT *SLOT, uint32_t key_set)
{
	if( !SLOT->key )
	{
		/* do NOT restart Phase Generator (verified on real YM2413)*/
		/* phase -> Dump */
		SLOT->state = EG_DMP;
	}
	SLOT->key |= key_set;
}

void ym2413_device::key_off(OPLL_SLOT *SLOT, uint32_t key_clr)
{
	if( SLOT->key )
	{
		SLOT->key &= key_clr;

		if( !SLOT->key )
		{
			/* phase -> Release */
			if (SLOT->state>EG_REL)
				SLOT->state = EG_REL;
		}
	}
}

/* update phase increment counter of operator (also update the EG rates if necessary) */
void ym2413_device::calc_fcslot(OPLL_CH *CH, OPLL_SLOT *SLOT)
{
	int ksr;
	uint32_t SLOT_rs;
	uint32_t SLOT_dp;

	/* (frequency) phase increment counter */
	SLOT->freq = CH->fc * SLOT->mul;
	ksr = CH->kcode >> SLOT->KSR;

	if( SLOT->ksr != ksr )
	{
		SLOT->ksr = ksr;

		/* calculate envelope generator rates */
		if ((SLOT->ar + SLOT->ksr) < 16+62)
		{
			SLOT->eg_sh_ar  = eg_rate_shift [SLOT->ar + SLOT->ksr ];
			SLOT->eg_sel_ar = eg_rate_select[SLOT->ar + SLOT->ksr ];
		}
		else
		{
			SLOT->eg_sh_ar  = 0;
			SLOT->eg_sel_ar = 13*RATE_STEPS;
		}
		SLOT->eg_sh_dr  = eg_rate_shift [SLOT->dr + SLOT->ksr ];
		SLOT->eg_sel_dr = eg_rate_select[SLOT->dr + SLOT->ksr ];
		SLOT->eg_sh_rr  = eg_rate_shift [SLOT->rr + SLOT->ksr ];
		SLOT->eg_sel_rr = eg_rate_select[SLOT->rr + SLOT->ksr ];

	}

	if (CH->sus)
		SLOT_rs  = 16 + (5<<2);
	else
		SLOT_rs  = 16 + (7<<2);

	SLOT->eg_sh_rs  = eg_rate_shift [SLOT_rs + SLOT->ksr ];
	SLOT->eg_sel_rs = eg_rate_select[SLOT_rs + SLOT->ksr ];

	SLOT_dp  = 16 + (13<<2);
	SLOT->eg_sh_dp  = eg_rate_shift [SLOT_dp + SLOT->ksr ];
	SLOT->eg_sel_dp = eg_rate_select[SLOT_dp + SLOT->ksr ];
}

/* set multi,am,vib,EG-TYP,KSR,mul */
void ym2413_device::set_mul(int slot,int v)
{
	OPLL_CH   *CH   = &P_CH[slot/2];
	OPLL_SLOT *SLOT = &CH->SLOT[slot&1];

	SLOT->mul     = mul_tab[v&0x0f];
	SLOT->KSR     = (v&0x10) ? 0 : 2;
	SLOT->eg_type = (v&0x20);
	SLOT->vib     = (v&0x40);
	SLOT->AMmask  = (v&0x80) ? ~0 : 0;
	calc_fcslot(CH,SLOT);
}

/* set ksl, tl */
void ym2413_device::set_ksl_tl(int chan,int v)
{
	OPLL_CH   *CH   = &P_CH[chan];
/* modulator */
	OPLL_SLOT *SLOT = &CH->SLOT[SLOT1];

	SLOT->ksl = ksl_shift[v >> 6];
	SLOT->TL  = (v&0x3f)<<(ENV_BITS-2-7); /* 7 bits TL (bit 6 = always 0) */
	SLOT->TLL = SLOT->TL + (CH->ksl_base>>SLOT->ksl);
}

/* set ksl , waveforms, feedback */
void ym2413_device::set_ksl_wave_fb(int chan,int v)
{
	OPLL_CH   *CH   = &P_CH[chan];
/* modulator */
	OPLL_SLOT *SLOT = &CH->SLOT[SLOT1];
	SLOT->wavetable = ((v&0x08)>>3)*SIN_LEN;
	SLOT->fb_shift  = (v&7) ? (v&7) + 8 : 0;

/*carrier*/
	SLOT = &CH->SLOT[SLOT2];

	SLOT->ksl = ksl_shift[v >> 6];
	SLOT->TLL = SLOT->TL + (CH->ksl_base>>SLOT->ksl);

	SLOT->wavetable = ((v&0x10)>>4)*SIN_LEN;
}

/* set attack rate & decay rate  */
void ym2413_device::set_ar_dr(int slot,int v)
{
	OPLL_CH   *CH   = &P_CH[slot/2];
	OPLL_SLOT *SLOT = &CH->SLOT[slot&1];

	SLOT->ar = (v>>4)  ? 16 + ((v>>4)  <<2) : 0;

	if ((SLOT->ar + SLOT->ksr) < 16+62)
	{
		SLOT->eg_sh_ar  = eg_rate_shift [SLOT->ar + SLOT->ksr ];
		SLOT->eg_sel_ar = eg_rate_select[SLOT->ar + SLOT->ksr ];
	}
	else
	{
		SLOT->eg_sh_ar  = 0;
		SLOT->eg_sel_ar = 13*RATE_STEPS;
	}

	SLOT->dr    = (v&0x0f)? 16 + ((v&0x0f)<<2) : 0;
	SLOT->eg_sh_dr  = eg_rate_shift [SLOT->dr + SLOT->ksr ];
	SLOT->eg_sel_dr = eg_rate_select[SLOT->dr + SLOT->ksr ];
}

/* set sustain level & release rate */
void ym2413_device::set_sl_rr(int slot,int v)
{
	OPLL_CH   *CH   = &P_CH[slot/2];
	OPLL_SLOT *SLOT = &CH->SLOT[slot&1];

	SLOT->sl  = sl_tab[ v>>4 ];

	SLOT->rr  = (v&0x0f)? 16 + ((v&0x0f)<<2) : 0;
	SLOT->eg_sh_rr  = eg_rate_shift [SLOT->rr + SLOT->ksr ];
	SLOT->eg_sel_rr = eg_rate_select[SLOT->rr + SLOT->ksr ];
}

void ym2413_device::load_instrument(uint32_t chan, uint32_t slot, uint8_t* inst )
{
	set_mul         (slot,   inst[0]);
	set_mul         (slot+1, inst[1]);
	set_ksl_tl      (chan,   inst[2]);
	set_ksl_wave_fb (chan,   inst[3]);
	set_ar_dr       (slot,   inst[4]);
	set_ar_dr       (slot+1, inst[5]);
	set_sl_rr       (slot,   inst[6]);
	set_sl_rr       (slot+1, inst[7]);
}

void ym2413_device::update_instrument_zero( uint8_t r )
{
	uint8_t* inst = &inst_tab[0][0]; /* point to user instrument */
	uint32_t chan;
	uint32_t chan_max;

	chan_max = 9;
	if (rhythm & 0x20)
		chan_max=6;

	switch(r)
	{
	case 0:
		for (chan=0; chan<chan_max; chan++)
		{
			if ((instvol_r[chan]&0xf0)==0)
			{
				set_mul         (chan*2, inst[0]);
			}
		}
		break;
	case 1:
		for (chan=0; chan<chan_max; chan++)
		{
			if ((instvol_r[chan]&0xf0)==0)
			{
				set_mul         (chan*2+1,inst[1]);
			}
		}
		break;
	case 2:
		for (chan=0; chan<chan_max; chan++)
		{
			if ((instvol_r[chan]&0xf0)==0)
			{
				set_ksl_tl      (chan,   inst[2]);
			}
		}
		break;
	case 3:
		for (chan=0; chan<chan_max; chan++)
		{
			if ((instvol_r[chan]&0xf0)==0)
			{
				set_ksl_wave_fb (chan,   inst[3]);
			}
		}
		break;
	case 4:
		for (chan=0; chan<chan_max; chan++)
		{
			if ((instvol_r[chan]&0xf0)==0)
			{
				set_ar_dr       (chan*2, inst[4]);
			}
		}
		break;
	case 5:
		for (chan=0; chan<chan_max; chan++)
		{
			if ((instvol_r[chan]&0xf0)==0)
			{
				set_ar_dr       (chan*2+1,inst[5]);
			}
		}
		break;
	case 6:
		for (chan=0; chan<chan_max; chan++)
		{
			if ((instvol_r[chan]&0xf0)==0)
			{
				set_sl_rr       (chan*2, inst[6]);
			}
		}
		break;
	case 7:
		for (chan=0; chan<chan_max; chan++)
		{
			if ((instvol_r[chan]&0xf0)==0)
			{
				set_sl_rr       (chan*2+1,inst[7]);
			}
		}
		break;
	}
}

/* write a value v to register r on chip chip */
void ym2413_device::write_reg(int r, int v)
{
	OPLL_CH *CH;
	OPLL_SLOT *SLOT;
	uint8_t *inst;
	int chan;
	int slot;

	/* adjust bus to 8 bits */
	r &= 0xff;
	v &= 0xff;

	switch(r&0xf0)
	{
	case 0x00:  /* 00-0f:control */
	{
		switch(r&0x0f)
		{
		case 0x00:  /* AM/VIB/EGTYP/KSR/MULTI (modulator) */
		case 0x01:  /* AM/VIB/EGTYP/KSR/MULTI (carrier) */
		case 0x02:  /* Key Scale Level, Total Level (modulator) */
		case 0x03:  /* Key Scale Level, carrier waveform, modulator waveform, Feedback */
		case 0x04:  /* Attack, Decay (modulator) */
		case 0x05:  /* Attack, Decay (carrier) */
		case 0x06:  /* Sustain, Release (modulator) */
		case 0x07:  /* Sustain, Release (carrier) */
			inst_tab[0][r & 0x07] = v;
			update_instrument_zero(r&7);
		break;

		case 0x0e:  /* x, x, r,bd,sd,tom,tc,hh */
		{
			if(v&0x20)
			{
				if ((rhythm&0x20)==0)
				/*rhythm off to on*/
				{
					logerror("YM2413: Rhythm mode enable\n");

	/* Load instrument settings for channel seven(chan=6 since we're zero based). (Bass drum) */
					chan = 6;
					inst = &inst_tab[16][0];
					slot = chan*2;

					load_instrument(chan, slot, inst);

	/* Load instrument settings for channel eight. (High hat and snare drum) */
					chan = 7;
					inst = &inst_tab[17][0];
					slot = chan*2;

					load_instrument(chan, slot, inst);

					CH   = &P_CH[chan];
					SLOT = &CH->SLOT[SLOT1]; /* modulator envelope is HH */
					SLOT->TL  = ((instvol_r[chan]>>4)<<2)<<(ENV_BITS-2-7); /* 7 bits TL (bit 6 = always 0) */
					SLOT->TLL = SLOT->TL + (CH->ksl_base>>SLOT->ksl);

	/* Load instrument settings for channel nine. (Tom-tom and top cymbal) */
					chan = 8;
					inst = &inst_tab[18][0];
					slot = chan*2;

					load_instrument(chan, slot, inst);

					CH   = &P_CH[chan];
					SLOT = &CH->SLOT[SLOT1]; /* modulator envelope is TOM */
					SLOT->TL  = ((instvol_r[chan]>>4)<<2)<<(ENV_BITS-2-7); /* 7 bits TL (bit 6 = always 0) */
					SLOT->TLL = SLOT->TL + (CH->ksl_base>>SLOT->ksl);
				}
				/* BD key on/off */
				if(v&0x10)
				{
					key_on (&P_CH[6].SLOT[SLOT1], 2);
					key_on (&P_CH[6].SLOT[SLOT2], 2);
				}
				else
				{
					key_off(&P_CH[6].SLOT[SLOT1],~2);
					key_off(&P_CH[6].SLOT[SLOT2],~2);
				}
				/* HH key on/off */
				if(v&0x01) key_on (&P_CH[7].SLOT[SLOT1], 2);
				else       key_off(&P_CH[7].SLOT[SLOT1],~2);
				/* SD key on/off */
				if(v&0x08) key_on (&P_CH[7].SLOT[SLOT2], 2);
				else       key_off(&P_CH[7].SLOT[SLOT2],~2);
				/* TOM key on/off */
				if(v&0x04) key_on (&P_CH[8].SLOT[SLOT1], 2);
				else       key_off(&P_CH[8].SLOT[SLOT1],~2);
				/* TOP-CY key on/off */
				if(v&0x02) key_on (&P_CH[8].SLOT[SLOT2], 2);
				else       key_off(&P_CH[8].SLOT[SLOT2],~2);
			}
			else
			{
				if (rhythm&0x20)
				/*rhythm on to off*/
				{
					logerror("YM2413: Rhythm mode disable\n");
	/* Load instrument settings for channel seven(chan=6 since we're zero based).*/
					chan = 6;
					inst = &inst_tab[instvol_r[chan]>>4][0];
					slot = chan*2;

					load_instrument(chan, slot, inst);

	/* Load instrument settings for channel eight.*/
					chan = 7;
					inst = &inst_tab[instvol_r[chan]>>4][0];
					slot = chan*2;

					load_instrument(chan, slot, inst);

	/* Load instrument settings for channel nine.*/
					chan = 8;
					inst = &inst_tab[instvol_r[chan]>>4][0];
					slot = chan*2;

					load_instrument(chan, slot, inst);
				}
				/* BD key off */
				key_off(&P_CH[6].SLOT[SLOT1],~2);
				key_off(&P_CH[6].SLOT[SLOT2],~2);
				/* HH key off */
				key_off(&P_CH[7].SLOT[SLOT1],~2);
				/* SD key off */
				key_off(&P_CH[7].SLOT[SLOT2],~2);
				/* TOM key off */
				key_off(&P_CH[8].SLOT[SLOT1],~2);
				/* TOP-CY off */
				key_off(&P_CH[8].SLOT[SLOT2],~2);
			}
			rhythm  = v&0x3f;
		}
		break;
		}
	}
	break;

	case 0x10:
	case 0x20:
	{
		int block_fnum;

		chan = r&0x0f;

		if (chan >= 9)
			chan -= 9;  /* verified on real YM2413 */

		CH = &P_CH[chan];

		if(r&0x10)
		{   /* 10-18: FNUM 0-7 */
			block_fnum  = (CH->block_fnum&0x0f00) | v;
		}
		else
		{   /* 20-28: suson, keyon, block, FNUM 8 */
			block_fnum = ((v&0x0f)<<8) | (CH->block_fnum&0xff);

			if(v&0x10)
			{
				key_on (&CH->SLOT[SLOT1], 1);
				key_on (&CH->SLOT[SLOT2], 1);
			}
			else
			{
				key_off(&CH->SLOT[SLOT1],~1);
				key_off(&CH->SLOT[SLOT2],~1);
			}


			if (CH->sus!=(v&0x20))
				logerror("chan=%i sus=%2x\n",chan,v&0x20);

			CH->sus = v & 0x20;
		}
		/* update */
		if(CH->block_fnum != block_fnum)
		{
			uint8_t block;

			CH->block_fnum = block_fnum;

			/* BLK 2,1,0 bits -> bits 3,2,1 of kcode, FNUM MSB -> kcode LSB */
			CH->kcode    = (block_fnum&0x0f00)>>8;

			CH->ksl_base = static_cast<uint32_t>(ksl_tab[block_fnum>>5]);

			block_fnum   = block_fnum * 2;
			block        = (block_fnum&0x1c00) >> 10;
			CH->fc       = fn_tab[block_fnum&0x03ff] >> (7-block);

			/* refresh Total Level in both SLOTs of this channel */
			CH->SLOT[SLOT1].TLL = CH->SLOT[SLOT1].TL + (CH->ksl_base>>CH->SLOT[SLOT1].ksl);
			CH->SLOT[SLOT2].TLL = CH->SLOT[SLOT2].TL + (CH->ksl_base>>CH->SLOT[SLOT2].ksl);

			/* refresh frequency counter in both SLOTs of this channel */
			calc_fcslot(CH,&CH->SLOT[SLOT1]);
			calc_fcslot(CH,&CH->SLOT[SLOT2]);
		}
	}
	break;

	case 0x30:  /* inst 4 MSBs, VOL 4 LSBs */
	{
		uint8_t old_instvol;

		chan = r&0x0f;

		if (chan >= 9)
			chan -= 9;  /* verified on real YM2413 */

		old_instvol = instvol_r[chan];
		instvol_r[chan] = v;  /* store for later use */

		CH   = &P_CH[chan];
		SLOT = &CH->SLOT[SLOT2]; /* carrier */
		SLOT->TL  = ((v&0x0f)<<2)<<(ENV_BITS-2-7); /* 7 bits TL (bit 6 = always 0) */
		SLOT->TLL = SLOT->TL + (CH->ksl_base>>SLOT->ksl);


		/*check whether we are in rhythm mode and handle instrument/volume register accordingly*/
		if ((chan>=6) && (rhythm&0x20))
		{
			/* we're in rhythm mode*/

			if (chan>=7) /* only for channel 7 and 8 (channel 6 is handled in usual way)*/
			{
				SLOT = &CH->SLOT[SLOT1]; /* modulator envelope is HH(chan=7) or TOM(chan=8) */
				SLOT->TL  = ((instvol_r[chan]>>4)<<2)<<(ENV_BITS-2-7); /* 7 bits TL (bit 6 = always 0) */
				SLOT->TLL = SLOT->TL + (CH->ksl_base>>SLOT->ksl);
			}
		}
		else
		{
			if ( (old_instvol&0xf0) == (v&0xf0) )
				return;

			inst = &inst_tab[instvol_r[chan]>>4][0];
			slot = chan*2;

			load_instrument(chan, slot, inst);

		#if 0
			logerror("YM2413: chan#%02i inst=%02i:  (r=%2x, v=%2x)\n",chan,v>>4,r,v);
			logerror("  0:%2x  1:%2x\n",inst[0],inst[1]);   logerror("  2:%2x  3:%2x\n",inst[2],inst[3]);
			logerror("  4:%2x  5:%2x\n",inst[4],inst[5]);   logerror("  6:%2x  7:%2x\n",inst[6],inst[7]);
		#endif
		}
	}
	break;

	default:
	break;
	}
}

//-------------------------------------------------
//  sound_stream_update - handle a stream update
//-------------------------------------------------

void ym2413_device::sound_stream_update(sound_stream &stream, stream_sample_t **inputs, stream_sample_t **outputs, int samples)
{
	for(int i=0; i < samples ; i++ )
	{
		output[0] = 0;
		output[1] = 0;

		advance_lfo();

		/* FM part */
		for(int j=0; j<6; j++)
			chan_calc(&P_CH[j]);

		if(!(rhythm & 0x20))
		{
			for(int j=6; j<9; j++)
				chan_calc(&P_CH[j]);
		}
		else        /* Rhythm part */
		{
			rhythm_calc(&P_CH[0], noise_rng & 1 );
		}

		outputs[0][i] = limit( output[0] , 32767, -32768 );
		outputs[1][i] = limit( output[1] , 32767, -32768 );

		advance();
	}
}

//-------------------------------------------------
//  device_start - device-specific startup
//-------------------------------------------------

void ym2413_device::device_start()
{
	int rate = clock()/72;

	m_stream = machine().sound().stream_alloc(*this,0,2,rate);

	for (int x=0; x<TL_RES_LEN; x++)
	{
		double m = (1<<16) / pow(2, (x+1) * (ENV_STEP/4.0) / 8.0);
		m = floor(m);

		/* we never reach (1<<16) here due to the (x+1) */
		/* result fits within 16 bits at maximum */

		int n = (int)m; /* 16 bits here */
		n >>= 4;        /* 12 bits here */
		if (n&1)        /* round to nearest */
			n = (n>>1)+1;
		else
			n = n>>1;
						/* 11 bits here (rounded) */
		tl_tab[ x*2 + 0 ] = n;
		tl_tab[ x*2 + 1 ] = -tl_tab[ x*2 + 0 ];

		for (int i=1; i<11; i++)
		{
			tl_tab[ x*2+0 + i*2*TL_RES_LEN ] =  tl_tab[ x*2+0 ]>>i;
			tl_tab[ x*2+1 + i*2*TL_RES_LEN ] = -tl_tab[ x*2+0 + i*2*TL_RES_LEN ];
		}
	}

	for (int i=0; i<SIN_LEN; i++)
	{
		/* non-standard sinus */
		double m = sin( ((i*2)+1) * M_PI / SIN_LEN ); /* checked against the real chip */

		/* we never reach zero here due to ((i*2)+1) */

		double o = 8*log(1.0/fabs(m))/log(2.0);  /* convert to 'decibels' */

		o = o / (ENV_STEP/4);

		int n = (int)(2.0*o);
		if (n&1)                        /* round to nearest */
			n = (n>>1)+1;
		else
			n = n>>1;

		/* waveform 0: standard sinus  */
		sin_tab[ i ] = n*2 + (m>=0.0? 0: 1 );

		/* waveform 1:  __      __     */
		/*             /  \____/  \____*/
		/* output only first half of the sinus waveform (positive one) */
		if (i & (1<<(SIN_BITS-1)) )
			sin_tab[1*SIN_LEN+i] = TL_TAB_LEN;
		else
			sin_tab[1*SIN_LEN+i] = sin_tab[i];
	}

	/* make fnumber -> increment counter table */
	for( int i = 0 ; i < 1024; i++ )
	{
		/* OPLL (YM2413) phase increment counter = 18bit */

		fn_tab[i] = i * (64 <<(FREQ_SH-10)); /* -10 because chip works with 10.10 fixed point, while we use 16.16 */
	}

	/* Amplitude modulation: 27 output levels (triangle waveform); 1 level takes one of: 192, 256 or 448 samples */
	/* One entry from LFO_AM_TABLE lasts for 64 samples */
	lfo_am_inc = (1<<LFO_SH) / 64;

	/* Vibrato: 8 output levels (triangle waveform); 1 level takes 1024 samples */
	lfo_pm_inc = (1<<LFO_SH) / 1024;

	/* Noise generator: a step takes 1 sample */
	noise_f = 1<<FREQ_SH;

	eg_timer_add  = 1<<EG_SH;
	eg_timer_overflow = 1<<EG_SH;


	save_item(NAME(instvol_r));
	save_item(NAME(eg_cnt));
	save_item(NAME(eg_timer));
	save_item(NAME(eg_timer_add));
	save_item(NAME(eg_timer_overflow));
	save_item(NAME(rhythm));
	save_item(NAME(lfo_am_cnt));
	save_item(NAME(lfo_am_inc));
	save_item(NAME(lfo_pm_cnt));
	save_item(NAME(lfo_pm_inc));
	save_item(NAME(noise_rng));
	save_item(NAME(noise_p));
	save_item(NAME(noise_f));
	save_item(NAME(inst_tab));
	save_item(NAME(address));

	for (int chnum = 0; chnum < ARRAY_LENGTH(P_CH); chnum++)
	{
		OPLL_CH *ch = &P_CH[chnum];

		save_item(NAME(ch->block_fnum), chnum);
		save_item(NAME(ch->fc), chnum);
		save_item(NAME(ch->ksl_base), chnum);
		save_item(NAME(ch->kcode), chnum);
		save_item(NAME(ch->sus), chnum);

		for (int slotnum = 0; slotnum < ARRAY_LENGTH(ch->SLOT); slotnum++)
		{
			OPLL_SLOT *sl = &ch->SLOT[slotnum];

			save_item(NAME(sl->ar), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->dr), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->rr), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->KSR), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->ksl), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->ksr), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->mul), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->phase), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->freq), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->fb_shift), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->op1_out), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->eg_type), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->state), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->TL), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->TLL), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->volume), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->sl), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->eg_sh_dp), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->eg_sel_dp), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->eg_sh_ar), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->eg_sel_ar), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->eg_sh_dr), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->eg_sel_dr), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->eg_sh_rr), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->eg_sel_rr), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->eg_sh_rs), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->eg_sel_rs), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->key), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->AMmask), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->vib), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
			save_item(NAME(sl->wavetable), chnum * ARRAY_LENGTH(ch->SLOT) + slotnum);
		}
	}
}

//-------------------------------------------------
//  device_clock_changed
//-------------------------------------------------
void ym2413_device::device_clock_changed()
{
	m_stream->set_sample_rate(clock() / 72);
}

//-------------------------------------------------
//  device_reset - device-specific reset
//-------------------------------------------------

void ym2413_device::device_reset()
{
	eg_timer = 0;
	eg_cnt   = 0;

	noise_rng = 1;    /* noise shift register */

	/* setup instruments table */
	if (m_inst_table != nullptr)
	{
		for (int i=0; i<19; i++)
		{
			for (int c=0; c<8; c++)
			{
				inst_tab[i][c] = m_inst_table[i][c];
			}
		}
	}


	/* reset with register write */
	write_reg(0x0f,0); /*test reg*/
	for(int i = 0x3f ; i >= 0x10 ; i-- )
		write_reg(i, 0x00);

	/* reset operator parameters */
	for(int c = 0 ; c < 9 ; c++ )
	{
		OPLL_CH *CH = &P_CH[c];
		for(int s = 0 ; s < 2 ; s++ )
		{
			/* wave table */
			CH->SLOT[s].wavetable = 0;
			CH->SLOT[s].state     = EG_OFF;
			CH->SLOT[s].volume    = MAX_ATT_INDEX;
		}
	}
}


void ym2413_device::write(offs_t offset, u8 data)
{
	if (offset)
		data_port_w(data);
	else
		register_port_w(data);
}

void ym2413_device::register_port_w(u8 data)
{
	address = data;
}

void ym2413_device::data_port_w(u8 data)
{
	m_stream->update();
	write_reg(address, data);
}

DEFINE_DEVICE_TYPE(YM2413, ym2413_device, "ym2413", "Yamaha YM2413 OPLL")

ym2413_device::ym2413_device(const machine_config &mconfig, const char *tag, device_t *owner, uint32_t clock)
	: ym2413_device(mconfig, YM2413, tag, owner, clock)
{
	for (int i = 0; i < 19; i++)
	{
		for (int c = 0; c < 8; c++)
		{
			m_inst_table[i][c] = table[i][c];
		}
	}
}

ym2413_device::ym2413_device(const machine_config &mconfig, device_type type, const char *tag, device_t *owner, uint32_t clock)
	: device_t(mconfig, type, tag, owner, clock)
	, device_sound_interface(mconfig, *this)
	, eg_cnt(0)
	, eg_timer(0)
	, eg_timer_add(0)
	, eg_timer_overflow(0)
	, rhythm(0)
	, LFO_AM(0)
	, LFO_PM(0)
	, lfo_am_cnt(0)
	, lfo_am_inc(0)
	, lfo_pm_cnt(0)
	, lfo_pm_inc(0)
	, noise_rng(0)
	, noise_p(0)
	, noise_f(0)
	, address(0)

{
	for (int i = 0; i < 19; i++)
	{
		std::fill_n(&m_inst_table[i][0], 8, 0);
		std::fill_n(&inst_tab[i][0], 8, 0);
	}

	std::fill(std::begin(tl_tab), std::end(tl_tab), 0);
	std::fill(std::begin(sin_tab), std::end(sin_tab), 0);
	std::fill(std::begin(instvol_r), std::end(instvol_r), 0);
	std::fill(std::begin(fn_tab), std::end(fn_tab), 0);
	std::fill(std::begin(output), std::end(output), 0);
}

DEFINE_DEVICE_TYPE(VRC7, vrc7snd_device, "vrc7snd", "Konami VRC7 (Sound)")

vrc7snd_device::vrc7snd_device(const machine_config &mconfig, const char *tag, device_t *owner, uint32_t clock)
	: ym2413_device(mconfig, VRC7, tag, owner, clock)
{
	for (int i = 0; i < 19; i++)
	{
		for (int c = 0; c < 8; c++)
		{
			m_inst_table[i][c] = vrc7_table[i][c];
		}
	}
}
