// license:BSD-3-Clause
// copyright-holders:Aaron Giles
/***************************************************************************

    atarigen.h

    General functions for Atari games.

***************************************************************************/

#ifndef MAME_MACHINE_ATARIGEN_H
#define MAME_MACHINE_ATARIGEN_H

#include "includes/slapstic.h"
#include "screen.h"


/***************************************************************************
    TYPES & STRUCTURES
***************************************************************************/

class atarigen_state : public driver_device
{
public:
	// construction/destruction
	atarigen_state(const machine_config &mconfig, device_type type, const char *tag);

protected:
	// users must call through to these
	virtual void machine_start() override;
	virtual void machine_reset() override;
	virtual void device_post_load() override;
	virtual void device_timer(emu_timer &timer, device_timer_id id, int param, void *ptr) override;

	// slapstic helpers
	void slapstic_configure(cpu_device &device, offs_t base, offs_t mirror, u8 *mem);
	void slapstic_update_bank(int bank);
	DECLARE_WRITE16_MEMBER(slapstic_w);
	DECLARE_READ16_MEMBER(slapstic_r);

	// video helpers
	void halt_until_hblank_0(device_t &device, screen_device &screen);

	// misc helpers
	void blend_gfx(int gfx0, int gfx1, int mask0, int mask1);

	// timer IDs
	enum
	{
		TID_UNHALT_CPU,
		TID_ATARIGEN_LAST
	};

	/* internal state */
	u8               m_slapstic_num;
	u16 *            m_slapstic;
	u8               m_slapstic_bank;
	std::vector<u8>  m_slapstic_bank0;
	offs_t           m_slapstic_last_pc;
	offs_t           m_slapstic_last_address;
	offs_t           m_slapstic_base;
	offs_t           m_slapstic_mirror;


	required_device<cpu_device> m_maincpu;

	optional_device<gfxdecode_device> m_gfxdecode;
	optional_device<screen_device> m_screen;
	optional_device<atari_slapstic_device> m_slapstic_device;
};



/***************************************************************************
    GENERAL ATARI NOTES
**************************************************************************##

    Atari 68000 list:

    Driver        Pr? Up? VC? PF? P2? MO? AL? BM? PH?
    ----------    --- --- --- --- --- --- --- --- ---
    arcadecl.cpp       *               *       *
    atarig1.cpp        *       *      rle  *
    atarig42.cpp       *       *      rle  *
    atarigt.cpp                *      rle  *
    atarigx2.cpp               *      rle  *
    atarisy1.cpp   *   *       *       *   *              270->260
    atarisy2.cpp   *   *       *       *   *              150->120
    badlands.cpp       *       *       *                  250->260
    batman.cpp     *   *   *   *   *   *   *       *      200->160 ?
    blstroid.cpp       *       *       *                  240->230
    cyberbal.cpp       *       *       *   *              125->105 ?
    eprom.cpp          *       *       *   *              170->170
    gauntlet.cpp   *   *       *       *   *       *      220->250
    klax.cpp       *   *       *       *                  480->440 ?
    offtwall.cpp       *   *   *       *                  260->260
    rampart.cpp        *               *       *          280->280
    relief.cpp     *   *   *   *   *   *                  240->240
    shuuz.cpp          *   *   *       *                  410->290 fix!
    skullxbo.cpp       *       *       *   *              150->145
    thunderj.cpp       *   *   *   *   *   *       *      180->180
    toobin.cpp         *       *       *   *              140->115 fix!
    vindictr.cpp   *   *       *       *   *       *      200->210
    xybots.cpp     *   *       *       *   *              235->238
    ----------  --- --- --- --- --- --- --- --- ---

    Pr? - do we have verifiable proof on priorities?
    Up? - have we updated to use new MO's & tilemaps?
    VC? - does it use the video controller?
    PF? - does it have a playfield?
    P2? - does it have a dual playfield?
    MO? - does it have MO's?
    AL? - does it have an alpha layer?
    BM? - does it have a bitmap layer?
    PH? - does it use the palette hack?

***************************************************************************/


#endif // MAME_MACHINE_ATARIGEN_H
