// license:BSD-3-Clause
// copyright-holders:Olivier Galibert
/*

Known functional issues:
- Star Wars Arcade
  * After recent changes, ship models periodically disappear for a frame or two during attract mode
    and while in-game.
  * There are indeterminate issues with the analog controls in-game.
- Virtua Fighter
  * Gameplay mechanics - specifically, collision detection - are significantly broken due to
    imperfect TGP RAM port emulation or hookup. This is observable in both attract mode and
    in-game. It seems to break when both characters attack at the same time.

Sega Model 1 Hardware Overview
------------------------------

Note! This document is a Work-In-Progress and covers all the dumped Sega Model 1 games, including...

Star Wars Arcade (C) Sega, 1994
Virtua Fighter   (C) Sega, 1993
Virtua Racing    (C) Sega, 1992
Wing War         (C) Sega, 1994
Tecwar/Netmerc   (C) Sega and Virtuality, 1993

The Sega Model 1 system comprises several PCB's....
CPU PCB   - This is the main PCB. It holds the main CPU, some surface-mounted ROMs and the 3D co-processors
            (manufactured by Fujitsu and known as 'TGP'). There are three different PCB variants, with
            different internal 3D co-processor programs depending on game.
Video PCB - There is only one known revision of this PCB so any game can use this PCB. It contains a very large
            quantity of RAM, some surface-mounted ROMs and some custom Sega video chips. The RGB and Sync is also
            outputted from this PCB. The horizontal sync is 24kHz and requires a medium resolution (24kHz) monitor.
            The video PCB is plugged in below the CPU PCB.
ROM PCB   - This PCB is the same for all games, only the type and number of ROMs differ per game. The ROM PCB is
            plugged in above the CPU PCB.
I/O PCB   - This PCB connects the CPU PCB to the input and output controls via some multi-wire connectors.
Comm PCB  - This PCB is used for communication when linking 2 PCBs together (generally for Virtua Racing only).
            It plugs in below the Video PCB.
Sound PCB - This PCB generates the sound for the game. There is no amplification of the sound here, that is done
            via the Amp PCB.
Motor PCB - This PCB is used to control the motors and steering column clutch used for the force feedback steering.
Amp PCB   - This PCB amplifies the sound generated by the sound PCB.


ROM PCB
-------

MODEL 1 MEMORY BD 837-7893 171-6079B (C) SEGA 1992
|--------------------------------------------------------------------------------------------|
|J1                                                                                          |
|J2      IC5            IC14                    IC26                   IC27                  |
|J3                                                                                          |
|J4      IC4            IC15                                                                 |
|J5                                             IC28                   IC29                  |
|J12     IC13           IC12                                                                 |
|                                                                                            |
|        IC11           IC10                    IC30                   IC31                  |
|                                                                                            |
| CN1    IC9            IC8       CN2                                                    CN3 |
|                                               IC32                   IC33                  |
|        IC7            IC6                                                                  |
|                                               84256                  84256              J11|
|        IC41           IC42                                                              J10|
|                                               84256                  84256              J9 |
|        IC39           IC40                                                              J8 |
|                                                                                         J7 |
|                                                                     315-5478            J6 |
|--------------------------------------------------------------------------------------------|
Notes:
      J1, J2, J4    - Jumpers to configure ROM sizes, all set to 2-3
      J3, J5, J6,   \
      J7, J8, J9    \
      J10, J11,     | Jumpers to configure ROM sizes, all set to 1-2
      J12           /
      CN1, CN2, CN3 - Connectors for joining ROM PCB to CPU PCB
      84256         - Fujitsu 84256 32k x8 SRAM (x4, DIP28)
      315-5478      - Lattice GAL16V8A (DIP20)

      IC4, IC5      - Hitachi HN27C1024 128k x16bit EPROM (DIP40)

      IC14, IC15    - Intel 27C040 512k x 8bit EPROM (DIP32)

      IC6, IC7, IC8,\
      IC9, IC10,    \
      IC11, IC12,   | 834000 4M mask ROM (DIP32)
      IC13,         |
      IC39, IC40,   /
      IC41, IC42    /

      IC26, IC27,   \
      IC28, IC29,   | 8316200 16M mask ROM (DIP42)
      IC30, IC31,   /
      IC32, IC33    /

      IC#   PCB
            Label   Virtua Racing    Virtua Fighter    Wing War       Star Wars Arcade
      --------------------------------------------------------------------------------
      4     B       EPR-14878A       EPR-16080         EPR-16951      -
      5     A       EPR-14879A       EPR-16081         EPR-16950      EPR-16467
      14    LO      EPR-14882        EPR-16082         EPR-16729      EPR-16468
      15    HI      EPR-14883        EPR-16083         EPR-16730      EPR-16469
      6     LO      MPR-14880        MPR-16084         MPR-16738      -
      7     HI      MPR-14881        MPR-16085         MPR-16737      -
      8     LO      MPR-14884        MPR-16086         MPR-16736      -
      9     HI      MPR-14885        MPR-16087         MPR-16735      -
      10    LO      MPR-14886        MPR-16088         MPR-16734      -
      11    HI      MPR-14887        MPR-16089         MPR-16733      -
      12    LO      MPR-14888        MPR-16090         -              -
      13    HI      MPR-14889        MPR-16091         -              -
      39    LO      MPR-14898        -                 MPR-16741      MPR-16472
      40    ML      MPR-14899        -                 MPR-16742      MPR-16473
      41    MH      MPR-14900        -                 MPR-16739      MPR-16474
      42    HI      MPR-14901        -                 MPR-16740      MPR-16475
      26    LO      MPR-14890        MPR-16096         MPR-16743      MPR-16476
      27    HI      MPR-14891        MPR-16097         MPR-16744      MPR-16477
      28    LO      MPR-14892        MPR-16098         MPR-16745      MPR-16478
      29    HI      MPR-14893        MPR-16099         MPR-16746      MPR-16479
      30    LO      MPR-14894        MPR-16100         MPR-16747      MPR-16480
      31    HI      MPR-14895        MPR-16101         MPR-16748      MPR-16481
      32    LO      MPR-14896        MPR-16102         MPR-16749      -
      33    HI      MPR-14897        MPR-16103         MPR-16750      -

      - denotes socket not populated.


CPU PCB
-------

MODEL-1 CPU BOARD 837-8886171-6298C (C) SEGA 1992
|---------------------------------------------------------------------------------------------------------------------------------|
|                        CN2                                    CN1                                   CN4                         |
|         |---------------------------------|   |---------------------------------|            |---------------|                  |
|         |---------------------------------|   |---------------------------------|            |---------------|                  |
|                                                                                                                                 |
|                       OPR-14746.68    |-|                                    5.5V_BATT                                         |-|
|       |-----------|                   | |                                                                          PC910       | |
|       |           |   OPR-14747.69    | |                                                              *1                      | |
|       | SEGA      |                   | |                                    84256     HM658128                    PC910       | |
|       | 315-5572  |                   | |CN13   |-----------|                                                                  | |CN3
|       |           |                   | |       |NEC        |                84256     HM658128                    PC910       | |
|       |           |                   | |       |D70615GD-16|                                                                  | |
|       |-----------|                   |-|       |V60        |                                                                  |-|
|                                                 |           |                                                                   |
|                                                 |           |                                                                   |
|                      32MHz                      |-----------|            |---------|     |---------|                            |
|                                                                          |SEGA     |     |SEGA     |     MB8421                 |
|       OPR-14745.65                                                       |315-5465 |     |315-5338A|                   *2       |
|                                                                          |         |     |         |                            |
|                                                                          |---------|     |---------|                            |
|       OPR-14744.64            40MHz                                                                                             |
|                                                                                      J6             CN5                         |
|                                                                                      J5      |---------------|                  |
|                                                                                      J4      |---------------|                  |
|                         3771                                                                                                    |
|       |-----------|     3771                   |-----------|                                                                    |
|       |           |                            |           |                                                                    |
|       | SEGA      |                            | SEGA      |                                                           D71051   |
|       | 315-5572  |                            | COPRO     |                  |-----------|                                     |
|       |           |                            |(see notes)|                  |           |     M5M5178     OPR-14743.45        |
|       |           |                            |           |                  | SEGA      |     M5M5178                         |
|       |-----------|                            |-----------|                  | 315-5464  |     M5M5178     OPR-14742.44        |
|                                                                               |           |     M5M5178                         |
|                                                                               |           |                                     |
|                                                                               |-----------|                           315-5546A |
|                                                                                                                                 |
|       |-----------|                            |-----------|                                                |-----------|       |
|       |           |                            |           |                                    84256       |           |       |
|       | SEGA      |                            | SEGA      |                                    84256       | SEGA      |       |
|       | 315-5571  |                            | 315-5571  |                                    84256       | 315-5463  |       |
|       |           |                            |           |                                    84256       |           |       |
|       |           |                            |           |                                                |           |       |
|       |-----------|                            |-----------|                                                |-----------|       |
|                                                                                                                                 |
|                                                                                                     CN6                         |
|                     LED1 LED2 LED3 LED4 LED5                                                 |---------------|                  |
|       LED11         LED6 LED7 LED8 LED9 LED10                                                |---------------|                  |
|---------------------------------------------------------------------------------------------------------------------------------|
Notes:
      84256         - Fujitsu 84256A-70LL 32k x8 SRAM (x6, DIP28)
      HM658128      - Hitachi HM658128ALP-10 16k x8 SRAM (x2, DIP32)
      M5M5178       - Mitsubishi M5M5178AP-25 8k x8 SRAM (x4, DIP28)
      CN1, CN2      - Connectors to join CPU board to Video board
      CN3           - Connector for power and input/output controls (joins small adapter board to a larger filter board)
      CN4, CN5, CN6 - Connectors to join ROM board to CPU board
      CN13          - Connector for optional daughterboard (not used)
      3771          - Fujitsu MB3771 master reset IC (x2, DIP8)
      5.5_BATT      - NEC 5.5 volt battery
      PC910         - Sharp PC910 opto-isolator (x3, DIP8)
      D71051        - NEC uPD71051C-10 UART (DIP28)
      MB8421        - Fujitsu MB8421-12LP RAM? (SDIP52)
      *1            - Unpopulated position for Fujitsu MB8421
      D70615GD-16   - NEC uPD70615GD-16-S V60 CPU, running at 16.000MHz (QFP120, clock 32 / 2)
      315-5546A     - Lattice GAL16V8A (DIP20)
      315-5571      - Fujitsu MB86233 Geometrizer (IC57/IC58, QFP160)
      315-5572      - Fujitsu MB86233 Geometrizer (different code) (IC60/IC66, QFP160)
      COPRO         - Fujitsu MB86233 Coprocessor (QFP160), differs depending on game:
          315-5573      - Virtua Racing, Virtua Formula (original for above board part number)
          315-5711      - Wing War, Star Wars Arcade, Netmerc
          315-5724      - Virtua Fighter
      315-5463      - Sega Custom (QFP160)
      315-5464      - Sega Custom (QFP160) - Copro TGP glue chip
      315-5465      - Sega Custom (QFP100) - Address decode, interrupts, etc
      315-5338A     - Sega Custom (QFP100) - Inputs
      *2            - Unpopulated position for 315-5338A
      J4, J5, J6    - Jumpers, all set to 2-3

      OPR-14742.44  \
      OPR-14743.45  - 1M SOP40 mask ROMs, tied to 315-5464

      OPR-14744.64  \
      OPR-14745.65  - 1M SOP40 mask ROMs, tied to both 315-5572's

      OPR-14746.68  \
      OPR-14747.69  - 1M SOP40 mask ROMs, tied to 315-5572 @ IC66


VIDEO PCB
---------

837-7894 171-6080D (C) SEGA 1992
|---------------------------------------------------------------------------------------------------------------------------------|
|                        CN2                                    CN1                                                               |
|         |---------------------------------|   |---------------------------------|                                               |
|         |---------------------------------|   |---------------------------------|                                               |
|                                                                                                                     HM65256     |
|                                                                                     J3                                          |
|   HM658512       HM658512                                        HM658128           J2                              HM65256     |
|                                        |-----------|                                      |-----------|                         |
|                                        |           |             HM658128                 |           |                         |
|                                        | SEGA      |                                      | SEGA      |             HM658128    |
|   HM654128       HM654128              | 315-5422  |                                      | 315-5292  |                         |
|                                        |           |                                      |           |             HM658128    |
|                                        |           |             M5M5178                  |           |                         |
|                                        |-----------|                                      |-----------|             HM658128    |
|   HM654128       HM654128                                        M5M5178                                                        |
|                                      |-----------|                                                                  HM658128    |
|   CY7B185           J1               |           |                                             J4                               |
|   CY7B185                            | SEGA      |                                                                              |
|   CY7B185                            | 315-5423  |        OPR-14748.16                315-5483                                  |
|   CY7B185                            |           |                                                                              |
|                                      |           |                                    315-5484                                  |
|                                      |-----------|                                                                              |
|                                                                                                                                 |
|        |-----------|                 |-----------|                                              M5M5178                         |
|        |           |                 |           |                                                                              |
|        | SEGA      |                 | SEGA      |                                              M5M5178                         |
|        | 315-5425  |                 | 315-5424  |        OPR-14748.15                                                          |
|        |           |                 |           |                                                                              |
|        |           |                 |           |                                                                              |
|        |-----------|                 |-----------|                                                                              |
|                                                                                                                                 |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256                                     M5M5178                            |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256            315-5485                 M5M5178                            |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256                                     M5M5178                            |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256            36MHz        315-5486                                       |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256                         315-5486                                       |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256                         315-5486                                       |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256                                                                        |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256                                        CN4          CN3      LED1      |
|---------------------------------------------------------------------------------------------------------------------------------|
Notes:
      HM65256  - Hitachi HM65256BLSP-10 32k x8 SRAM (x34, DIP28)
      HM658128 - Hitachi HM658128ALP-10 16k x8 SRAM (x10, DIP32)
      HM658512 - Hitachi HM658512LP-8 64k x8 SRAM (x2, DIP32)
      M5M5178  - Mitsubishi M5M5178AP-25 8k x8 SRAM (x7, DIP28)
      CY7B185  - Cypress CY7B185 8k x8 SRAM (x4, DIP28)
      J1       - Jumper, set to 2-3
      J2       - Jumper, set to 1-2
      J3       - Jumper, set to 2-3
      J4       - Jumper, set to 1-2
      CN1, CN2 - Connectors to join Video board to CPU board
      CN3      - Connector for R/G/B/Sync output (@ 24kHz)
      CN4      - 20 pin IDC flat-cable connector (purpose unknown)
      315-5483 - PAL CK2605 (DIP20)           -- Priorities
      315-5484 - Lattice GAL16V8A (DIP20)     -- Palette bank selection
      315-5485 - Lattice GAL16V8A (DIP20)     -- Palette access address decode
      315-5486 - Lattice GAL16V8A (x3, DIP20) -- Color clear on blank
      315-5422 - Sega Custom (QFP160) -- Z sorting (8K*16bits + 128K*16bits storage)
      315-5423 - Sega Custom (QFP160) -- Renderer 1
      315-5424 - Sega Custom (QFP160) -- Renderer 2, provides opr-14748 addresses
      315-5425 - Sega Custom (QFP160) -- Renderer 3, framebuffer management (32K*8bits + 2x256K*16bits storage)
      315-5292 - Sega Custom (QFP160) -- Tilemaps (128k*32bits + 32K*16bits storage)

      OPR-14748.15  \
      OPR-14748.16  - 1M SOP40 mask ROMs, tied to 315-5423 & 315-5424. Note both ROMs are identical.


Motor PCB
---------

SJ25-0155-01 838-9081
|------------------------------------------------------|
|          CN6                      CN7        CN8     |
|CN1                                    6A             |
|                         DSW2                         |
|        |---------|                 2A    C4149       |
|        |SEGA     |      DSW1                      CN9|
|        |315-5296 |                                   |
|        |         |                               CN10|
|        |---------|                MB3759             |
|                                                   C11|
|CN2                   |---------|                     |
|                      |SEGA     |                     |
|          8MHz        |315-5296 |                     |
|                      |         |                     |
|      315-5625        |---------|                     |
|           3771                                       |
|CN3            3773                                   |
|                                    2A                |
|              Z80                          C4149      |
|                                                      |
|              15111.12                            CN12|
|                                   MB3759             |
|           M6253   6264                               |
|                                                      |
|CN4        CN5      7-SEG                             |
|------------------------------------------------------|
Notes:
      There are also many capacitors/transistors and other power-related components which are not shown.

      315-5296 - Sega Custom (QFP100)
      7-SEG    - 7-Segment Display (x4), used for PCB status/diagnostics
      15111.12 - ST Microelectronics M27C512 64k x8 EPROM (DIP28, labelled 'EPR-15111')
      Z80      - Zilog Z0840004PSC Z80 CPU, running at 4.000MHz (DIP40, clock 8 / 2)
      MB8464   - Fujitsu MB8464 8k x8 SRAM (DIP28)
      3771     - Fujitsu MB3771 Master Reset IC (DIP8)
      3773     - Fujitsu MB3771 Master Reset IC with Watchdog Timer (DIP8)
      2A       - Fuse, 2 Amp
      6A       - Fuse, 6 Amp
      MB3759   - Fujitsu MB3759 Switching Regulator Controller (x2, DIP16)
      CN1, CN2 - Connects to Left Side I/O PCB
      CN3, CN4 - Connects to Right Side I/O PCB
      CN5      - Connects to Steering Voltage Regulator (Left and Right Sides)
      CN6      - Connects to Power Supply (+5V)
      CN7, CN8 - Connects to 28VAC transformer
      CN9      - 2 pin connector (not used?)
      CN10     - 2 pin connector (not used?)
      CN11     - 7 pin connector (joins to Motor and Clutch Left Side)
      CN12     - 7 pin connector (joins to Motor and Clutch Right Side)
      DSW1/2   - 8-position Dip Switch (x2)
                 +-------------------------------------------------+
                 |DSW1               1   2   3   4   5   6   7   8 |
                 +---------+-------+---+---+---+---+---+---+---+---+
                 |Feedback | Light |OFF|OFF|OFF|   |   |   |   |   |
                 |Weight   |       +---+---+---+   |   |   |   |   |
                 |PL1      |  /\   |ON |OFF|OFF|   |   |   |   |   |
                 |         |  ||   +---+---+---+   |   |   |   |   |
                 |         |       |OFF|ON |OFF|   |   |   |   |   |
                 |         |       +---+---+---+   |   |   |   |   |
                 |         |       |ON |ON |OFF|   |   |   |   |   |
                 |         |       +---+---+---+   |   |   |   |   |
                 |         |       |OFF|OFF|ON |   |   |   |   |   |
                 |         |  ||   +---+---+---+   |   |   |   |   |
                 |         |  \/   |ON |OFF|ON |   |   |   |   |   |
                 |         |       +---+---+---+   |   |   |   |   |
                 |         | Heavy |ON |ON |ON |   |   |   |   |   |
                 +---------+-------+---+---+---+---+   |   |   |   |
                 |Power On | Yes   |           |OFF|   |   |   |   |
                 |Check    +-------+-----------+---+   |   |   |   |
                 |         | No    |           |ON |   |   |   |   |
                 +---------+-------+-----------+---+---+---+   |   |
                 |Not Used |                       |OFF|OFF|   |   |
                 +---------+-----------------------+---+---+---+---+
                 |Not Used |                               |OFF|OFF|
                 +---------+-------------------------------+---+---+

                 +-------------------------------------------------+
                 |DSW2               1   2   3   4   5   6   7   8 |
                 +---------+-------+---+---+---+---+---+---+---+---+
                 |Feedback | Light |OFF|OFF|OFF|   |   |   |   |   |
                 |Weight   |       +---+---+---+   |   |   |   |   |
                 |PL2      |  /\   |ON |OFF|OFF|   |   |   |   |   |
                 |         |  ||   +---+---+---+   |   |   |   |   |
                 |         |       |OFF|ON |OFF|   |   |   |   |   |
                 |         |       +---+---+---+   |   |   |   |   |
                 |         |       |ON |ON |OFF|   |   |   |   |   |
                 |         |       +---+---+---+   |   |   |   |   |
                 |         |       |OFF|OFF|ON |   |   |   |   |   |
                 |         |  ||   +---+---+---+   |   |   |   |   |
                 |         |  \/   |ON |OFF|ON |   |   |   |   |   |
                 |         |       +---+---+---+   |   |   |   |   |
                 |         | Heavy |ON |ON |ON |   |   |   |   |   |
                 +---------+-------+---+---+---+---+   |   |   |   |
                 |Power On | Yes   |           |OFF|   |   |   |   |
                 |Check    +-------+-----------+---+   |   |   |   |
                 |         | No    |           |ON |   |   |   |   |
                 +---------+-------+-----------+---+---+---+---+---+
                 |7-SEG LED|Display V.R. Value 00-80-FF    |OFF|OFF|
                 |         +-------------------------------+---+---+
                 |         |Display Clutch Value           |ON |OFF|
                 |         +-------------------------------+---+---+
                 |         |Data from the I/O Board (00-FF)|OFF|ON |
                 |         +-------------------------------+---+---+
                 |         |Data to the I/O Board (00-FF)  |ON |ON |
                 +---------+-------------------------------+---+---+


Comm PCB
--------

MODEL-1 COMMUNICATION BD 837-8842 171-6293B (C) SEGA 1992
|--------------------------------------------------------------------------------|
|                                                                                |
|    MB89237A            MB89374                                                 |
|       JP4                                                                 LED1 |
|    15112.17            Z80                                                     |
|    JP2  JP3                                                       75179        |
|    MB8464              315-5624                                     JP6        |
|                                                       315-5547                 |
|        315-5611                                            SW1    PC910     CN4|
|                                                                                |
|                                                                   PC910     CN5|
|     MB8421             MB8431                                JP7               |
|                                                                   JP5          |
|        JP8                                                                  CN7|
|                CN1                                    CN2                      |
| |---------------------------------|   |---------------------------------|   CN6|
| |---------------------------------|   |---------------------------------|      |
|--------------------------------------------------------------------------------|
Notes:
      15112.17 - AMD AM27C100 128k x8 EPROM (DIP32, labelled 'EPR-15112')
      Z80      - Zilog Z0840004PSC Z80 CPU, running at 4.000MHz (DIP40)
      MB8464   - Fujitsu MB8464 8k x8 SRAM (DIP28)
      MB8421   - Fujitsu MB8421-12LP RAM? (SDIP52)
      MB8431   - Fujitsu MB8431-90LP RAM? (SDIP52)
      MB89237A - Fujitsu MB89237A ? (DIP20)
      MB89374  - Fujitsu MB89374 ? (SDIP42)
      75179    - Texas Instruments SN75179 Differential Driver and Receiver Pair (DIP8)
      315-5547 - AMI 18CV8PC-25 PAL (DIP20)
      315-5624 - MMI PAL16L8BCN PAL (DIP20)
      315-5611 - Lattice GAL16V8A PAL (DIP20)
      PC910    - Sharp PC910 opto-isolator (x2, DIP8)
      SW1      - Push Button Switch (purpose unknown)
      CN1, CN2 - Connectors to join Comm board to Video board
      CN4      - 8 pin connector (purpose unknown)
      CN5      - 6 pin connector (purpose unknown)
      CN6, CN7 - Connectors for network optical cable link
      JP2      - Jumper, set to 2-3
      JP3      - Jumper, set to 1-2
      JP4      - Jumper, set to 1-2
      JP5      - Jumper, shorted
      JP6      - Jumper, not shorted
      JP7      - Jumper, not shorted
      JP8      - Jumper, set to 1-2


Sound PCB
---------

MODEL-1 SOUND BD 837-8679 (C) SEGA 1992
|-----------------------------------------------------------|   SOUND BD OPTION 837-8680 (C) SEGA 1992
| CN2                    CN1                 CN5  CN6  CN7  |   |--------------------------|
|                   LED5               CN9                  |   |  CN2    CN3  CN4 CN5     |
| PC910                                               TL062 |   |              uPC844C     |
|                JP5                           TL062        |   |     LC78820        TL062 |
| JP3     IC8    JP6                                        |   |                          |
| JP4                MB8464    82C51                 uPC844C|   |             IC5          |
|                    MB8464                YM3438           |   |                          |
| JP1     IC7                                               |   |    IC3                   |
| JP2                                                       |   |             IC4          |
|                                                  LC78820  |   | LED                      |
|                  315-5578                16MHz            |   |JP6           |---------| |
|                                                           |   |JP5           |SEGA     | |
|                  315-5577                                 |   |JP4           |315-5560 | |
|                                                           |   |JP3  315-5579 |         | |
|                           |---------|                     |   |JP2           |---------| |
| |-----------------|       |SEGA     |    CN8              |   |JP1    CN1                |
| |  TMP68000N-10   |       |315-5560 |          315-5579   |   |--------------------------|
| |                 |       |         |                     |   Notes:
| |-----------------|  JP7  |---------|          IC33       |         JP1, JP2 - Jumpers, both set to 2-3
|                                         IC31              |         JP3, JP4,- Jumpers, all set to 1-2
|   LED1 LED2 LED3 LED4                          IC32       |         JP5, JP6 /
|DSW1                3771      20MHz                        |         CN1, CN2 - Connectors to join Sound Option Board to Main Sound Board
|-----------------------------------------------------------|         CN3      - 5 pin connector, unamplified sound out to Amp PCB
                                                                      CN4, CN5 - Unamplified audio output, left/right RCA audio connectors (not used)
Notes:
      TMP68000N-10- Toshiba TMP68000N-10 CPU, running at 10.000MHz (SDIP64, clock 20 / 2)
      82C51       - Toshiba 82C51AM-10 Programmable 8-bit I/O Serial Interface (SOP28)
      DSW1        - 4 position Dip Switch
      MB8464      - Fujitsu MB8464 8k x8 SRAM (DIP28)
      3771        - Fujitsu MB3771 Master Reset IC (DIP8)
      TL062       - ST Microelectronics Dual Low Power Operational Amplifier (DIP8)
      PC910       - Sharp PC910 opto-isolator (DIP8)
      LC78820     - Sanyo LC78820 2-channel 18-bit D/A Converter (DIP20)
      uPC844C     - NEC uPC844C Quad High Speed Wide Band Operational Amplifier (DIP14)
      315-5560    - Sega Custom (QFP100)
      315-5577    - Lattice GAL16V8A (DIP20)
      315-5578    - Lattice GAL16V8A (DIP20)
      315-5579    - Lattice GAL16V8A (DIP20)
      YM3438      - Yamaha YM3438 sound chip, running at 8.000MHz (DIP24, clock 16 / 2)
      JP1, JP2    - Jumpers to configure ROM size of IC7, both set to 2-3
      JP3, JP4    - Jumpers to configure ROM size of IC8, both set to 2-3
      JP5, JP6    - Jumpers to configure size of RAM, both set to 2-3
      JP7         - Jumper, set to 2-3
      JP13, JP14, \
      JP15, JP16, | Jumpers to configure size of ROMs at IC31, IC32 & IC33
      JP17, JP18  / JP13 & JP14 set to 2-3, others set to 1-2
      CN1         - 10 pin connector for power input (+5V, GND, +12V)
      CN2         - 6 pin connector (joins to main PCB, used for sound communication from Main PCB to Sound PCB)
      CN5         - 5 pin connector, unamplified sound out to Amp PCB
      CN6, CN7    - Unamplified audio output, left/right RCA audio connectors (not used)
      CN8, CN9    - Connectors to join Sound Option Board to Main Sound Board

      Sound Main PCB
      --------------

                    IC#   PCB
                          Label        Virtua Racing    Virtua Fighter    Wing War     Star Wars Arcade
                    -----------------------------------------------------------------------------------
                    7     1024/4096    EPR-14870A       EPR-16120         EPR-16751    !
                    8     1024/4096    -                EPR-16121         EPR-16752    ?
                    31    27C4001      -                -                 -            ?
                    32    23C16000     MPR-14873        MPR-16122         MPR-16753    !
                    33    23C16000     -                MPR-16123         MPR-16754    ?

      Sound Sub PCB
      -------------

                    IC#   PCB
                          Label        Virtua Racing    Virtua Fighter    Wing War     Star Wars Arcade
                    -----------------------------------------------------------------------------------
                    3     27C4001      -                -                 -            ?
                    4     23C16000     MPR-14876        MPR-16124         MPR-16755    ?
                    5     23C16000     -                MPR-16125         MPR-16756    ?

                    - denotes socket known to be not populated.
                    ! denotes known to be not dumped.
                    ? denotes unknown if socket is populated or not, but not dumped either way.


Audio Mix PCB
-------------

839-0542 (C) SEGA 1992
|-------------|
|             |
|             |
|CN1  CN3  CN2|
|             |
|             |
|-------------|
Notes:
      This is a very small PCB with only 3 connectors on it and 4 resistors
      which simply mixes the left/right audio into one overall output which
      is fed into the Amp PCB.
      CN1 - 5 pin connector, right channel audio input
      CN2 - 5 pin connectos, mixed audio output to Amp PCB
      CN3 - 5 pin connector, left channel audio input


Amp PCB
-------

838-10018 SJ25-0168-03
|----------------------------------|
|             CN1                  |
|                        7812   CN2|
|CN3                               |
|                                  |
|                        TA8225H   |
|                                  |
|                               CN4|
|                                  |
|                                  |
|                        TA8225H   |
|                                  |
|LCH                               |
|RCH                               |
|----------------------------------|
Notes:
      There are also many capacitors/transistors and other power-related
      components which are not shown.

      7812     - 12V voltage regulator
      TA8225H  - Toshiba TA8225H 45W power amplifier (x2)
      CN1      - 3 pin connector, 12VAC power input
      CN2      - 2 pin 12VDC power output
      CN3      - 7 pin connector, joins to credit board and coin mech (power/credit related outputs)
      CN4      - 4 pin connector, amplified dual (left/right) speaker audio outputs
      RCH, LCH - RCA connectors, left/right sound input from CN2 of Audio Mix PCB


*/

#include "emu.h"
#include "includes/model1.h"

#include "cpu/i386/i386.h"
#include "machine/clock.h"
#include "machine/model1io.h"
#include "machine/model1io2.h"
#include "speaker.h"

#include "vr.lh"
#include "model1io2.lh"

// On the real system, another 315-5338A is acting as slave
// and writes the data to the dual port RAM. This isn't
// emulated yet, data just gets written to RAM.

u8 model1_state::dpram_r(offs_t offset)
{
	// insert waitstate
	m_maincpu->adjust_icount(-1);
	return m_dpram->right_r(offset);
}

void model1_state::gen_outputs_w(uint8_t data)
{
	// generic output lines, output to outx where x = bit
	// eg. out0 = coin counter 1, see below for per-game descriptions
	for (int i = 0; i < 8; i++)
		m_outs[i] = BIT(data, i);

	m_digits[1] = data;
}

void model1_state::vf_outputs_w(uint8_t data)
{
	// 7654----  unknown (not used?)
	// ----3---  start 2 lamp
	// -----2--  start 1 lamp
	// ------1-  coin counter 2
	// -------0  coin counter 1

	machine().bookkeeping().coin_counter_w(1, BIT(data, 1));
	machine().bookkeeping().coin_counter_w(0, BIT(data, 0));
}

void model1_state::vr_outputs_w(uint8_t data)
{
	// 7-------  race leader lamp
	// -6------  vr4 (green) lamp
	// --5-----  vr3 (yellow) lamp
	// ---4----  vr2 (blue) lamp
	// ----3---  vr1 (red) lamp
	// -----2--  start lamp
	// ------1-  coin counter 2
	// -------0  coin counter 1

	machine().bookkeeping().coin_counter_w(1, BIT(data, 1));
	machine().bookkeeping().coin_counter_w(0, BIT(data, 0));
}

void model1_state::swa_outputs_w(uint8_t data)
{
	// 7-------  unknown (not used?)
	// -6------  unknown (1 while in-game)
	// --5-----  unknown (not used?)
	// ---4----  vr lamp
	// ----3---  unknown (not used?)
	// -----2--  start lamp
	// ------1-  coin counter 2
	// -------0  coin counter 1

	machine().bookkeeping().coin_counter_w(1, BIT(data, 1));
	machine().bookkeeping().coin_counter_w(0, BIT(data, 0));
}

void model1_state::wingwar_outputs_w(uint8_t data)
{
	// 7-------  unknown (not used?)
	// -6------  view selector 4 lamp
	// --5-----  view selector 3 lamp
	// ---4----  view selector 2 lamp
	// ----3---  view selector 1 lamp
	// -----2--  start lamp
	// ------1-  unknown (not used?)
	// -------0  coin counter 1

	machine().bookkeeping().coin_counter_w(0, BIT(data, 0));
}

void model1_state::wingwar360_outputs_w(uint8_t data)
{
	// 7654----  unknown (not used?)
	// ----3---  danger lamp
	// -----2--  start lamp
	// ------1-  coin counter 2
	// -------0  coin counter 1

	machine().bookkeeping().coin_counter_w(1, BIT(data, 1));
	machine().bookkeeping().coin_counter_w(0, BIT(data, 0));
}

void model1_state::netmerc_outputs_w(uint8_t data)
{
	// 76------  unknown (not used?)
	// --54----  mvd backlights
	// ----3---  mvd holder light
	// -----2--  trigger/thumb motor
	// ------1-  unknown (not used?)
	// -------0  coin counter 1

	machine().bookkeeping().coin_counter_w(0, BIT(data, 0));
}

void model1_state::drive_board_w(uint8_t data)
{
	m_digits[0] = data;
}

uint8_t model1_state::r360_r()
{
	return m_r360_state;
}

void model1_state::r360_w(uint8_t data)
{
	/*
	    this uses the feedback board protocol
	    command group B - these seem to be gamestates

	    bf = init
	    be = attract
	    bd = setup #1 (lower safety bar etc.)
	    bc = setup #2 (push emergency button)
	    bb = ready to go
	    ba = ingame
	    b9 = game over

	    results:
	    40 = default status
	    41 = * (setup #1 ack)
	    42 = lowered safety bar
	    43 = closed belt
	    44 = lever up
	    45 = pushed button
	    46 = game start
	    47 = game over
	    48 = lever down
	    49 = released belt
	*/
	switch (data & 0xff)
	{
		case 0xbf:
		case 0xbe:
			m_r360_state = ~0x40;
			break;

		case 0xbd:
			m_r360_state = ~0x44;
			break;

		case 0xbc:
			m_r360_state = ~0x45;
			break;

		case 0xbb:
			m_r360_state = ~0x46;
			break;

		case 0xba:
		case 0xb9:
			m_r360_state = ~0x40;
			break;
	}
}

u16 model1_state::fifoin_status_r()
{
	return 0xffff;
}

void model1_state::bank_w(offs_t offset, u16 data, u16 mem_mask)
{

	if(ACCESSING_BITS_0_7) {
		switch(data & 0xf) {
		case 0x1: // 100000-1fffff data roms banking
			membank("bank1")->set_base(memregion("maincpu")->base() + 0x1000000 + 0x100000*((data >> 4) & 0x7)); // netmerc has bit 0x80 set when banking, probably not a bank bit.
			logerror("BANK %x\n", 0x1000000 + 0x100000*((data >> 4) & 0xf));
			break;
		case 0x2: // 200000-2fffff data roms banking (unused, all known games have only one bank)
			break;
		case 0xf: // f00000-ffffff program roms banking (unused, all known games have only one bank)
			break;
		}
	}
}



void model1_state::irq_raise(int level)
{
	//  logerror("irq: raising %d\n", level);
	m_irq_status |= (1 << level);
	m_maincpu->set_input_line(0, ASSERT_LINE);
}

IRQ_CALLBACK_MEMBER(model1_state::irq_callback)
{
	for (int i = 0; i < 8; i++)
		if (BIT(m_irq_status, i))
		{
			m_last_irq = i;
			break;
		}

	return m_last_irq;
}

void model1_state::irq_control_w(u8 data)
{
	switch (data)
	{
	case 0x10:
		m_irq_status = 0;
		m_maincpu->set_input_line(0, CLEAR_LINE);
		break;

	case 0x20:
		m_irq_status &= ~(1 << m_last_irq);
		if (m_irq_status == 0)
			m_maincpu->set_input_line(0, CLEAR_LINE);
		break;
	}
}

// vf
// 1 = fe3ed4
// 3 = fe3f5c
// other = fe3ec8 / fe3ebc

// vr
// 1 = fe02bc
// other = f302a4 / fe02b0

// swa
// 1 = ff504
// 3 = ff54c
// other = ff568/ff574

void model1_state::irq_init()
{
	m_maincpu->set_input_line(0, CLEAR_LINE);
	m_m1uart->write_cts(0);
}

TIMER_DEVICE_CALLBACK_MEMBER(model1_state::model1_interrupt)
{
	int scanline = param;

	if (scanline == 384)
	{
		if (m_m1comm != nullptr)
			m_m1comm->check_vint_irq();
		irq_raise(1);
	}
	else if(scanline == 384/2)
	{
		irq_raise(m_sound_irq);
	}
}

void model1_state::machine_reset()
{
	membank("bank1")->set_base(memregion("maincpu")->base() + 0x1000000);
	irq_init();
	copro_reset();

	if (!strcmp(machine().system().name, "swa") ||
		!strcmp(machine().system().name, "swaj"))
	{
		m_sound_irq = 0;
	}
	else
	{
		m_sound_irq = 3;
	}
}


void model1_state::model1_mem(address_map &map)
{
	/* ROMA */ map(0x000000, 0x0fffff).rom();
	/* ROMO */ map(0x100000, 0x1fffff).bankr("bank1");
	/* ROMX */ map(0x200000, 0x2fffff).rom();
	/* ROMY */
	/* RAMA */ map(0x400000, 0x40ffff).ram();
	/* RAMB */ map(0x500000, 0x53ffff).ram();

	/* TGP  */ map(0x600000, 0x60ffff).ram().share("display_list0");
	/*      */ map(0x610000, 0x61ffff).ram().share("display_list1");
	/*      */ map(0x680000, 0x680003).rw(FUNC(model1_state::model1_listctl_r), FUNC(model1_state::model1_listctl_w));

	/* SCR  */ map(0x700000, 0x70ffff).rw(m_tiles, FUNC(segas24_tile_device::tile_r), FUNC(segas24_tile_device::tile_w));
	/*      */ map(0x720000, 0x720001).nopw();        // Unknown, always 0
	/*      */ map(0x740000, 0x740001).nopw();        // Horizontal synchronization register
	/*      */ map(0x760000, 0x760001).nopw();        // Vertical synchronization register
	/*      */ map(0x770000, 0x770001).nopw();        // Video synchronization switch
	/*      */ map(0x780000, 0x7fffff).rw(m_tiles, FUNC(segas24_tile_device::char_r), FUNC(segas24_tile_device::char_w));

	/* OBJ  */

	/* COL  */ map(0x900000, 0x903fff).ram().w(m_palette, FUNC(palette_device::write16)).share("palette");
	/*      */ map(0x910000, 0x91bfff).ram().share("color_xlat");

	/* EX0  */
	/* EX1  */

	/* I/O  */ map(0xc00000, 0xc00fff).r(FUNC(model1_state::dpram_r)).w(m_dpram, FUNC(mb8421_device::right_w)).umask16(0x00ff); // 2k*8-bit dual port ram
	/*      */ map(0xc40000, 0xc40003).rw(m_m1uart, FUNC(i8251_device::read), FUNC(i8251_device::write)).umask16(0x00ff);

	/* CPR  */ map(0xd00000, 0xd00001).rw(FUNC(model1_state::v60_copro_ram_adr_r), FUNC(model1_state::v60_copro_ram_adr_w)).mirror(0x1fffe);
	/*      */ map(0xd20000, 0xd20003).rw(FUNC(model1_state::v60_copro_ram_r), FUNC(model1_state::v60_copro_ram_w)).mirror(0x1fffc);
	/*      */ map(0xd80000, 0xd80003).rw(FUNC(model1_state::v60_copro_fifo_r), FUNC(model1_state::v60_copro_fifo_w)).mirror(0x1fffc);
	/*      */ map(0xdc0000, 0xdc0003).r(FUNC(model1_state::fifoin_status_r)).mirror(0x1fffc);

	/* GLUE */ map(0xe00000, 0xe00000).w(FUNC(model1_state::irq_control_w));
	/*      */ map(0xe00004, 0xe00005).w(FUNC(model1_state::bank_w));
	/*      */ map(0xe0000c, 0xe0000f).nopw();

	/* ROM0 */ map(0xf80000, 0xffffff).rom();
}

void model1_state::model1_io(address_map &map)
{
	map(0xd00000, 0xd00001).rw(FUNC(model1_state::v60_copro_ram_adr_r), FUNC(model1_state::v60_copro_ram_adr_w)).mirror(0x1fffe);
	map(0xd20000, 0xd20003).rw(FUNC(model1_state::v60_copro_ram_r), FUNC(model1_state::v60_copro_ram_w)).mirror(0x1fffc);
	map(0xd80000, 0xd80003).rw(FUNC(model1_state::v60_copro_fifo_r), FUNC(model1_state::v60_copro_fifo_w)).mirror(0x1fffc);
	map(0xdc0000, 0xdc0003).r(FUNC(model1_state::fifoin_status_r)).mirror(0x1fffc);
}

void model1_state::model1_comm_mem(address_map &map)
{
	model1_mem(map);

	map(0xb00000, 0xb00fff).rw(m_m1comm, FUNC(m1comm_device::share_r), FUNC(m1comm_device::share_w));
	map(0xb01000, 0xb01000).rw(m_m1comm, FUNC(m1comm_device::cn_r), FUNC(m1comm_device::cn_w));
	map(0xb01002, 0xb01002).rw(m_m1comm, FUNC(m1comm_device::fg_r), FUNC(m1comm_device::fg_w));
}

static INPUT_PORTS_START( ioboard_dipswitches )
	PORT_START("ioboard:dsw1")
	PORT_DIPUNUSED_DIPLOC(0x01, 0x01, "DSW1:1")
	PORT_DIPUNUSED_DIPLOC(0x02, 0x02, "DSW1:2")
	PORT_DIPUNUSED_DIPLOC(0x04, 0x04, "DSW1:3")
	PORT_DIPUNUSED_DIPLOC(0x08, 0x08, "DSW1:4")
	PORT_DIPUNUSED_DIPLOC(0x10, 0x10, "DSW1:5")
	PORT_DIPUNUSED_DIPLOC(0x20, 0x20, "DSW1:6")
	PORT_DIPUNUSED_DIPLOC(0x40, 0x40, "DSW1:7")
	PORT_DIPUNUSED_DIPLOC(0x80, 0x80, "DSW1:8")

	PORT_START("ioboard:dsw2")
	PORT_DIPUNUSED_DIPLOC(0x01, 0x01, "DSW2:1")
	PORT_DIPUNUSED_DIPLOC(0x02, 0x02, "DSW2:2")
	PORT_DIPUNUSED_DIPLOC(0x04, 0x04, "DSW2:3")
	PORT_DIPUNUSED_DIPLOC(0x08, 0x08, "DSW2:4")
	PORT_DIPUNUSED_DIPLOC(0x10, 0x10, "DSW2:5")
	PORT_DIPUNUSED_DIPLOC(0x20, 0x20, "DSW2:6")
	PORT_DIPUNUSED_DIPLOC(0x40, 0x40, "DSW2:7")
	PORT_DIPUNUSED_DIPLOC(0x80, 0x80, "DSW2:8")

	PORT_START("ioboard:dsw3")
	PORT_DIPUNUSED_DIPLOC(0x01, 0x01, "DSW3:1")
	PORT_DIPUNUSED_DIPLOC(0x02, 0x02, "DSW3:2")
	PORT_DIPUNUSED_DIPLOC(0x04, 0x04, "DSW3:3")
	PORT_DIPUNUSED_DIPLOC(0x08, 0x08, "DSW3:4")
	PORT_DIPUNUSED_DIPLOC(0x10, 0x10, "DSW3:5")
	PORT_DIPUNUSED_DIPLOC(0x20, 0x20, "DSW3:6")
	PORT_DIPUNUSED_DIPLOC(0x40, 0x40, "DSW3:7")
	PORT_DIPUNUSED_DIPLOC(0x80, 0x80, "DSW3:8")
INPUT_PORTS_END

static INPUT_PORTS_START( vf )
	PORT_START("IN.0")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_SERVICE_NO_TOGGLE(0x04, IP_ACTIVE_LOW)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("IN.1")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 )        PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 )        PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 )        PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN )  PORT_PLAYER(1) PORT_8WAY
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP )    PORT_PLAYER(1) PORT_8WAY
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1) PORT_8WAY
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT )  PORT_PLAYER(1) PORT_8WAY

	PORT_START("IN.2")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 )        PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 )        PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 )        PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN )  PORT_PLAYER(2) PORT_8WAY
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP )    PORT_PLAYER(2) PORT_8WAY
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(2) PORT_8WAY
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT )  PORT_PLAYER(2) PORT_8WAY

	PORT_INCLUDE(ioboard_dipswitches)
INPUT_PORTS_END

static INPUT_PORTS_START( vr )
	PORT_START("IN.0")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_SERVICE_NO_TOGGLE(0x04, IP_ACTIVE_LOW)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("VR1 (Red)")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("VR2 (Blue)")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_NAME("VR3 (Yellow)")

	PORT_START("IN.1")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_NAME("VR4 (Green)")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON7 ) PORT_NAME("Shift Down")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON8 ) PORT_NAME("Shift Up")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("IN.2")   /* 8Bit RX-line from drive board */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("WHEEL")
	PORT_BIT( 0xff, 0x80, IPT_PADDLE ) PORT_SENSITIVITY(100) PORT_KEYDELTA(3)

	PORT_START("ACCEL")
	PORT_BIT( 0xff, 0x30, IPT_PEDAL ) PORT_MINMAX(1,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(16)

	PORT_START("BRAKE")
	PORT_BIT( 0xff, 0x30, IPT_PEDAL2 ) PORT_MINMAX(1,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(16)

	PORT_INCLUDE(ioboard_dipswitches)
INPUT_PORTS_END

static INPUT_PORTS_START( swa )
	PORT_START("IN.0")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_SERVICE_NO_TOGGLE(0x0004, IP_ACTIVE_LOW)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("IN.1")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0xe0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("IN.2")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("STICK1X")
	PORT_BIT( 0xff, 0x7f, IPT_AD_STICK_X ) PORT_MINMAX(27,227) PORT_SENSITIVITY(100) PORT_KEYDELTA(4)  PORT_REVERSE

	PORT_START("STICK1Y")
	PORT_BIT( 0xff, 0x7f, IPT_AD_STICK_Y ) PORT_MINMAX(27,227) PORT_SENSITIVITY(100) PORT_KEYDELTA(4)  PORT_REVERSE

	PORT_START("THROTTLE")
	PORT_BIT( 0xff, 0x7f, IPT_PEDAL )      PORT_MINMAX(28,200) PORT_SENSITIVITY(100) PORT_KEYDELTA(16) PORT_REVERSE

	PORT_START("STICK2X")
	PORT_BIT( 0xff, 0x7f, IPT_AD_STICK_X ) PORT_MINMAX(27,227) PORT_SENSITIVITY(100) PORT_KEYDELTA(4)  PORT_PLAYER(2) PORT_REVERSE

	PORT_START("STICK2Y")
	PORT_BIT( 0xff, 0x7f, IPT_AD_STICK_Y ) PORT_MINMAX(27,227) PORT_SENSITIVITY(100) PORT_KEYDELTA(4)  PORT_PLAYER(2)

	PORT_INCLUDE(ioboard_dipswitches)
INPUT_PORTS_END

static INPUT_PORTS_START( wingwar )
	PORT_START("IN.0")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE(0x0004, IP_ACTIVE_LOW)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_PLAYER(1) PORT_NAME("View 1")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_PLAYER(1) PORT_NAME("View 2")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_PLAYER(1) PORT_NAME("View 3")

	PORT_START("IN.1")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON7 ) PORT_PLAYER(1) PORT_NAME("View 4")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1) PORT_NAME("Machine Gun")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1) PORT_NAME("Missile")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1) PORT_NAME("Smoke")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("STICKX")
	PORT_BIT( 0xff, 0x80, IPT_AD_STICK_X ) PORT_SENSITIVITY(100) PORT_KEYDELTA(4) PORT_REVERSE

	PORT_START("STICKY")
	PORT_BIT( 0xff, 0x80, IPT_AD_STICK_Y ) PORT_SENSITIVITY(100) PORT_KEYDELTA(4)

	PORT_START("THROTTLE")
	PORT_BIT( 0xff, 0x01, IPT_PEDAL ) PORT_MINMAX(1,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(16)

	PORT_INCLUDE(ioboard_dipswitches)
INPUT_PORTS_END

static INPUT_PORTS_START( wingwar360 )
	PORT_INCLUDE(wingwar)

	PORT_MODIFY("IN.0")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0xe0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_MODIFY("IN.1")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_MODIFY("STICKX")
	PORT_BIT( 0xff, 0x80, IPT_AD_STICK_X ) PORT_SENSITIVITY(100) PORT_KEYDELTA(4)

	PORT_MODIFY("STICKY")
	PORT_BIT( 0xff, 0x80, IPT_AD_STICK_Y ) PORT_SENSITIVITY(100) PORT_KEYDELTA(4) PORT_REVERSE
INPUT_PORTS_END

static INPUT_PORTS_START( netmerc )
	PORT_START("IN.0")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE(0x04, IP_ACTIVE_LOW)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("IN.1")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Trigger")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Thumb")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("MVD Holder")
	PORT_BIT( 0xf8, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("STICKX")
	PORT_BIT( 0xff, 0x7f, IPT_AD_STICK_X ) PORT_SENSITIVITY(100) PORT_KEYDELTA(4) PORT_REVERSE

	PORT_START("STICKY")
	PORT_BIT( 0xff, 0x7f, IPT_AD_STICK_Y ) PORT_SENSITIVITY(100) PORT_KEYDELTA(4) PORT_REVERSE

	PORT_INCLUDE(ioboard_dipswitches)
INPUT_PORTS_END



#define MODEL1_CPU_BOARD \
	ROM_REGION32_LE( 0x40000, "copro_tables", 0 ) \
	ROM_LOAD32_WORD("opr14742.bin",  0x000000,  0x20000, CRC(446a1085) SHA1(51b3f4d3a35a36087ea0ba4e26d6e7d17b6418e2) ) \
	ROM_LOAD32_WORD("opr14743.bin",  0x000002,  0x20000, CRC(e8953554) SHA1(1499f8e30ac15affc66e6f04ae031bb8680d9260) ) \
	\
	ROM_REGION32_LE( 0x80000, "other_data", 0 ) \
	/* 1/x table */ \
	ROM_LOAD32_WORD("opr-14744.58",   0x000000,  0x20000, CRC(730ea9e0) SHA1(651f1db4089a400d073b19ada299b4b08b08f372) ) \
	ROM_LOAD32_WORD("opr-14745.59",   0x000002,  0x20000, CRC(4c934d96) SHA1(e3349ece0e47f684d61ad11bfea4a90602287350) ) \
	/* 1/sqrt(x) table */ \
	ROM_LOAD32_WORD("opr-14746.62",   0x040000,  0x20000, CRC(2a266cbd) SHA1(34e047a93459406c22acf4c25089d1a4955f94ca) ) \
	ROM_LOAD32_WORD("opr-14747.63",   0x040002,  0x20000, CRC(a4ad5e19) SHA1(7d7ec300eeb9a8de1590011e37108688c092f329) ) \
	\
	ROM_REGION16_LE( 0x20000, "other_other_data", 0 ) \
	/* 1/(1+x) again, but bottom 16 bits only */ \
	ROM_LOAD("opr14748.bin",   0x000000,  0x20000, CRC(4a532cb8) SHA1(23280ebbcd6b2bc8a8e643a2d07a58d6598301b8) ) \
	\
	ROM_REGION32_LE( 0x2000, "315_5571", 0) \
	ROM_LOAD("315-5571.bin", 0, 0x2000, CRC(1233db2a) SHA1(06760409d40f3d9117fd3e7c7ab62dfd70aa2a4d) ) \
	\
	ROM_REGION32_LE( 0x2000, "315_5572", 0) \
	ROM_LOAD("315-5572.bin", 0, 0x2000, CRC(66785906) SHA1(d94a51eced65b0073fb39625927782bf264d271a) )

ROM_START( vf )
	MODEL1_CPU_BOARD

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16082.14", 0x200000, 0x80000, CRC(b23f22ee) SHA1(9fd5b5a5974703a60a54de3d2bce4301bfc0e533) ) /* Rom board with Sega ID# 834-10170 */
	ROM_LOAD16_BYTE( "epr-16083.15", 0x200001, 0x80000, CRC(d12c77f8) SHA1(b4aeba8d5f1ab4aec024391407a2cb58ce2e94b0) )

	ROM_LOAD( "epr-16080.4", 0xfc0000, 0x20000, CRC(3662e1a5) SHA1(6bfceb1a7c1c7912679c907f2b7516ae9c7dda67) )
	ROM_LOAD( "epr-16081.5", 0xfe0000, 0x20000, CRC(6dec06ce) SHA1(7891544456bccd2fc647bccd058945ad50466636) )

	ROM_LOAD16_BYTE( "mpr-16084.6", 0x1000000, 0x80000, CRC(483f453b) SHA1(41a5527be73f5dd1c87b2a8113235bdd247ec049) )
	ROM_LOAD16_BYTE( "mpr-16085.7", 0x1000001, 0x80000, CRC(5fa01277) SHA1(dfa7ddff0a7daf29071431f26b93dd8e8e5793b6) )
	ROM_LOAD16_BYTE( "mpr-16086.8", 0x1100000, 0x80000, CRC(deac47a1) SHA1(3a8016124e4dc579d4aae745d4af1905ad0e4fbd) )
	ROM_LOAD16_BYTE( "mpr-16087.9", 0x1100001, 0x80000, CRC(7a64daac) SHA1(da6a9cad4b0cb2af4299e664c0889f3fbdc25530) )
	ROM_LOAD16_BYTE( "mpr-16088.10", 0x1200000, 0x80000, CRC(fcda2d1e) SHA1(0f7d0f604d429a1da0d1c3f31694520bada49680) )
	ROM_LOAD16_BYTE( "mpr-16089.11", 0x1200001, 0x80000, CRC(39befbe0) SHA1(362c493092cd0536fadee7326ecc7f973e23fb58) )
	ROM_LOAD16_BYTE( "mpr-16090.12", 0x1300000, 0x80000, CRC(90c76831) SHA1(5a3c25f2a131cfbb2ad067bef1ab7b1c95645d41) )
	ROM_LOAD16_BYTE( "mpr-16091.13", 0x1300001, 0x80000, CRC(53115448) SHA1(af798d5b1fcb720d7288a5ac48839d9ace16a2f2) )

	ROM_REGION32_LE( 0x2000, "tgp_copro", 0)
	ROM_LOAD("315-5724.bin", 0, 0x2000, CRC(4b4f330e) SHA1(8809d93d47593f808faca55161999677ac7a3eb0) BAD_DUMP )

	ROM_REGION( 0xc0000, M1AUDIO_CPU_REGION, ROMREGION_BE|ROMREGION_16BIT )  /* 68K code */
	ROM_LOAD16_WORD_SWAP( "epr-16120.7", 0x00000, 0x20000, CRC(2bff8378) SHA1(854b08ab983e4e98cb666f2f44de9a6829b1eb52) )
	ROM_LOAD16_WORD_SWAP( "epr-16121.8", 0x20000, 0x20000, CRC(ff6723f9) SHA1(53498b8c103745883657dfd6efe27edfd48b356f) )
	ROM_RELOAD( 0x80000, 0x20000)

	ROM_REGION( 0x400000, M1AUDIO_MPCM1_REGION, 0 ) /* Samples */
	ROM_LOAD( "mpr-16122.32", 0x000000, 0x200000, CRC(568bc64e) SHA1(31fd0ef8319efe258011b4621adebb790b620770) )
	ROM_LOAD( "mpr-16123.33", 0x200000, 0x200000, CRC(15d78844) SHA1(37c17e38604cf7004a951408024941cd06b1d93e) )

	ROM_REGION( 0x400000, M1AUDIO_MPCM2_REGION, 0 ) /* Samples */
	ROM_LOAD( "mpr-16124.4", 0x000000, 0x200000, CRC(45520ba1) SHA1(c33e3c12639961016e5fa6b5025d0a67dff28907) )
	ROM_LOAD( "mpr-16125.5", 0x200000, 0x200000, CRC(9b4998b6) SHA1(0418d9b0acf79f35d0f7575c21f1be9a0ea343da) )

	ROM_REGION32_LE( 0x1000000, "polygons", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16096.26", 0x000000, 0x200000, CRC(a92b0bf3) SHA1(fd3adff5f41f0b0be98df548c848eda04fc0da48) )
	ROM_LOAD32_WORD( "mpr-16097.27", 0x000002, 0x200000, CRC(0232955a) SHA1(df934fb6d022032620932571ff5ed176d5dcb017) )
	ROM_LOAD32_WORD( "mpr-16098.28", 0x400000, 0x200000, CRC(cf2e1b84) SHA1(f3d16c72344f7f218a792ce7f1dd7cad910a8c97) )
	ROM_LOAD32_WORD( "mpr-16099.29", 0x400002, 0x200000, CRC(20e46854) SHA1(423d3642bd2f14e68d29029c027b791de2c1ec53) )
	ROM_LOAD32_WORD( "mpr-16100.30", 0x800000, 0x200000, CRC(e13e983d) SHA1(120637caa2404ad4124b676fd6fcd721f30948df) )
	ROM_LOAD32_WORD( "mpr-16101.31", 0x800002, 0x200000, CRC(0dbed94d) SHA1(df1cddcc1d3976816bd786c2d6211a8563f6f690) )
	ROM_LOAD32_WORD( "mpr-16102.32", 0xc00000, 0x200000, CRC(4cb41fb6) SHA1(4a07bfad4f221508de8c931861424dcc5be3f46a) )
	ROM_LOAD32_WORD( "mpr-16103.33", 0xc00002, 0x200000, CRC(526d1c76) SHA1(edc8dafc9261cd0e970c3b50e3c1ca51a32a4cdf) )
ROM_END

/*
Virtua Racing

        Sega game ID# 833-8942 VIRTUA RACING TWIN
   Sega ROM board ID# 834-8941
*/
ROM_START( vr )
	MODEL1_CPU_BOARD

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-14882.14", 0x200000, 0x80000, CRC(547d75ad) SHA1(a57c11966886c37de1d7df131ad60457669231dd) )
	ROM_LOAD16_BYTE( "epr-14883.15", 0x200001, 0x80000, CRC(6bfad8b1) SHA1(c1f780e456b405abd42d92f4e03e40aad88f8c22) )

	ROM_LOAD( "epr-14878a.4", 0xfc0000, 0x20000, CRC(6d69e695) SHA1(12d3612d3dfd474b8020cdfb8ffc5dcc64e2e1a3) )
	ROM_LOAD( "epr-14879a.5", 0xfe0000, 0x20000, CRC(d45af9dd) SHA1(48f2bf940c78e3ae4273a56e9f32371d870e41e0) )

	ROM_LOAD16_BYTE( "mpr-14880.6",  0x1000000, 0x80000, CRC(adc7c208) SHA1(967b6f522011f17fd2821ccbe20bcb6d4680a4a0) )
	ROM_LOAD16_BYTE( "mpr-14881.7",  0x1000001, 0x80000, CRC(e5ab89df) SHA1(873dea86628457e69f732158e3efb05d133eaa44) )
	ROM_LOAD16_BYTE( "mpr-14884.8",  0x1100000, 0x80000, CRC(6cf9c026) SHA1(f4d717958dba6b6402f5ffe7638fe0bf353b61ed) )
	ROM_LOAD16_BYTE( "mpr-14885.9",  0x1100001, 0x80000, CRC(f65c9262) SHA1(511a22bcfaf199737bfc5a809fd66cb4439dd386) )
	ROM_LOAD16_BYTE( "mpr-14886.10", 0x1200000, 0x80000, CRC(92868734) SHA1(e1dfb134dc3ba7e0b1d40681621e09ac5a222518) )
	ROM_LOAD16_BYTE( "mpr-14887.11", 0x1200001, 0x80000, CRC(10c7c636) SHA1(c77d55460bba4354349e473e129f21afeed05e91) )
	ROM_LOAD16_BYTE( "mpr-14888.12", 0x1300000, 0x80000, CRC(04bfdc5b) SHA1(bb8788a761620d0440a62ae51c3b41f70a04b5e4) )
	ROM_LOAD16_BYTE( "mpr-14889.13", 0x1300001, 0x80000, CRC(c49f0486) SHA1(cc2bb9059c016ba2c4f6e7508bd1687df07b8b48) )

	ROM_REGION( 0x2000, "tgp_copro", 0 ) /* TGP program rom */
	ROM_LOAD("315-5573.bin", 0, 0x2000, CRC(3335a19b) SHA1(72eedfcc799ec4c7534fd7415de6631087ff6731) )

	ROM_REGION( 0xc0000, M1AUDIO_CPU_REGION, ROMREGION_BE|ROMREGION_16BIT )  /* 68K code */
	ROM_LOAD16_WORD_SWAP( "epr-14870a.7", 0x00000, 0x20000, CRC(919d9b75) SHA1(27be79881cc9a2b5cf37e18f1e2d87251426b428) )

	ROM_REGION( 0x400000, M1AUDIO_MPCM1_REGION, 0 ) /* Samples */
	ROM_LOAD( "mpr-14873.32", 0x000000, 0x200000, CRC(b1965190) SHA1(fc47e9ed4a44d48477bd9a35e42c26508c0f4a0c) )

	ROM_REGION( 0x400000, M1AUDIO_MPCM2_REGION, 0 ) /* Samples */
	ROM_LOAD( "mpr-14876.4", 0x000000, 0x200000, CRC(ba6b2327) SHA1(02285520624a4e612cb4b65510e3458b13b1c6ba) )

	ROM_REGION32_LE( 0x1000000, "polygons", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-14890.26", 0x000000, 0x200000, CRC(dcbe006b) SHA1(195be7fabec405ca1b4e1338d3b8d7bb4a06dd73) )
	ROM_LOAD32_WORD( "mpr-14891.27", 0x000002, 0x200000, CRC(25832b38) SHA1(a8d74538149c92bae661334e327b031eaca2a8f2) )
	ROM_LOAD32_WORD( "mpr-14892.28", 0x400000, 0x200000, CRC(5136f3ba) SHA1(ce8312975764db09bbfa2068b99559a5c1798a36) )
	ROM_LOAD32_WORD( "mpr-14893.29", 0x400002, 0x200000, CRC(1c531ada) SHA1(8b373ac97a3649c64f48eb3d1dd95c5833f330b6) )
	ROM_LOAD32_WORD( "mpr-14894.30", 0x800000, 0x200000, CRC(830a71bc) SHA1(884378e8a5afeb819daf5285d0d205986d566340) )
	ROM_LOAD32_WORD( "mpr-14895.31", 0x800002, 0x200000, CRC(af027ac5) SHA1(523f03d90358ddb7d0e96fd06b9a65cebfc09f24) )
	ROM_LOAD32_WORD( "mpr-14896.32", 0xc00000, 0x200000, CRC(382091dc) SHA1(efa266f0f6bfe36ad1c365e588fff33b01e166dd) )
	ROM_LOAD32_WORD( "mpr-14897.33", 0xc00002, 0x200000, CRC(74873195) SHA1(80705ec577d14570f9bba77cc26766f831c41f42) )

	ROM_REGION32_LE( 0x200000, "copro_data", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-14898.39", 0x000000, 0x80000, CRC(61da2bb6) SHA1(7a12ba522d64a1aeec1ca6f5a87ee063692131f9) )
	ROM_LOAD32_BYTE( "mpr-14899.40", 0x000001, 0x80000, CRC(2cd58bee) SHA1(73defec823de4244a387af55fea7210edc1b314f) )
	ROM_LOAD32_BYTE( "mpr-14900.41", 0x000002, 0x80000, CRC(aa7c017d) SHA1(0fa2b59a8bb5f5907b2b2567e69d11c73b398dc1) )
	ROM_LOAD32_BYTE( "mpr-14901.42", 0x000003, 0x80000, CRC(175b7a9a) SHA1(c86602e771cd49bab425b4ba7926d2f44858bd39) )

	ROM_REGION16_LE(0x80, "ioboard:eeprom", 0)
	ROM_LOAD("93c45.bin", 0x00, 0x80, CRC(65aac303) SHA1(17687fedf1578e977cae4e7c3f5c00cad4aa490d) )
ROM_END

ROM_START( vformula )
	MODEL1_CPU_BOARD

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-15638.14", 0x200000, 0x80000, CRC(b9db21a2) SHA1(db58c047977f5fc37f278afe7159a78e3fa6c015) )
	ROM_LOAD16_BYTE( "epr-15639.15", 0x200001, 0x80000, CRC(4c3796f5) SHA1(1bf312a4999a15fbc5d194627f9c0ad9dbc1f2c0) )

	ROM_LOAD( "epr-15623.4", 0xfc0000, 0x20000, CRC(155fa5be) SHA1(a0a3fd8980c52279adbc1c64aa22e42a0b196dd9) )
	ROM_LOAD( "epr-15622.5", 0xfe0000, 0x20000, CRC(18007f6f) SHA1(61573742627ec027abd2cc700e79f04da5215bfd) )

	ROM_LOAD16_BYTE( "epr-15641.6",  0x1000000, 0x80000, CRC(bf01e4d5) SHA1(53ad9ecd2de2ea1d7b446f9db61352e10a55ea05) )
	ROM_LOAD16_BYTE( "epr-15640.7",  0x1000001, 0x80000, CRC(3e6d83dc) SHA1(62aa552a38ee193e0dfab5d1261756fe237db42c) )
	ROM_LOAD16_BYTE( "mpr-14884.8",  0x1100000, 0x80000, CRC(6cf9c026) SHA1(f4d717958dba6b6402f5ffe7638fe0bf353b61ed) )
	ROM_LOAD16_BYTE( "mpr-14885.9",  0x1100001, 0x80000, CRC(f65c9262) SHA1(511a22bcfaf199737bfc5a809fd66cb4439dd386) )
	ROM_LOAD16_BYTE( "mpr-14886.10", 0x1200000, 0x80000, CRC(92868734) SHA1(e1dfb134dc3ba7e0b1d40681621e09ac5a222518) )
	ROM_LOAD16_BYTE( "mpr-14887.11", 0x1200001, 0x80000, CRC(10c7c636) SHA1(c77d55460bba4354349e473e129f21afeed05e91) )
	ROM_LOAD16_BYTE( "mpr-14888.12", 0x1300000, 0x80000, CRC(04bfdc5b) SHA1(bb8788a761620d0440a62ae51c3b41f70a04b5e4) )
	ROM_LOAD16_BYTE( "mpr-14889.13", 0x1300001, 0x80000, CRC(c49f0486) SHA1(cc2bb9059c016ba2c4f6e7508bd1687df07b8b48) )

	ROM_REGION( 0x2000, "tgp_copro", 0 ) /* TGP program rom */
	ROM_LOAD("315-5573.bin", 0, 0x2000, CRC(3335a19b) SHA1(72eedfcc799ec4c7534fd7415de6631087ff6731) )

	ROM_REGION( 0xc0000, M1AUDIO_CPU_REGION, ROMREGION_BE|ROMREGION_16BIT )  /* 68K code */
	ROM_LOAD16_WORD_SWAP( "epr-14870a.7", 0x00000, 0x20000, CRC(919d9b75) SHA1(27be79881cc9a2b5cf37e18f1e2d87251426b428) )

	ROM_REGION( 0x400000, M1AUDIO_MPCM1_REGION, 0 ) /* Samples */
	ROM_LOAD( "mpr-14873.32", 0x000000, 0x200000, CRC(b1965190) SHA1(fc47e9ed4a44d48477bd9a35e42c26508c0f4a0c) )

	ROM_REGION( 0x400000, M1AUDIO_MPCM2_REGION, 0 ) /* Samples */
	ROM_LOAD( "mpr-14876.4", 0x000000, 0x200000, CRC(ba6b2327) SHA1(02285520624a4e612cb4b65510e3458b13b1c6ba) )

	ROM_REGION32_LE( 0x1000000, "polygons", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-14890.26", 0x000000, 0x200000, CRC(dcbe006b) SHA1(195be7fabec405ca1b4e1338d3b8d7bb4a06dd73) )
	ROM_LOAD32_WORD( "mpr-14891.27", 0x000002, 0x200000, CRC(25832b38) SHA1(a8d74538149c92bae661334e327b031eaca2a8f2) )
	ROM_LOAD32_WORD( "mpr-14892.28", 0x400000, 0x200000, CRC(5136f3ba) SHA1(ce8312975764db09bbfa2068b99559a5c1798a36) )
	ROM_LOAD32_WORD( "mpr-14893.29", 0x400002, 0x200000, CRC(1c531ada) SHA1(8b373ac97a3649c64f48eb3d1dd95c5833f330b6) )
	ROM_LOAD32_WORD( "mpr-14894.30", 0x800000, 0x200000, CRC(830a71bc) SHA1(884378e8a5afeb819daf5285d0d205986d566340) )
	ROM_LOAD32_WORD( "mpr-14895.31", 0x800002, 0x200000, CRC(af027ac5) SHA1(523f03d90358ddb7d0e96fd06b9a65cebfc09f24) )
	ROM_LOAD32_WORD( "mpr-14896.32", 0xc00000, 0x200000, CRC(382091dc) SHA1(efa266f0f6bfe36ad1c365e588fff33b01e166dd) )
	ROM_LOAD32_WORD( "mpr-14897.33", 0xc00002, 0x200000, CRC(74873195) SHA1(80705ec577d14570f9bba77cc26766f831c41f42) )

	ROM_REGION32_LE( 0x200000, "copro_data", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-14898.39", 0x000000, 0x80000, CRC(61da2bb6) SHA1(7a12ba522d64a1aeec1ca6f5a87ee063692131f9) )
	ROM_LOAD32_BYTE( "mpr-14899.40", 0x000001, 0x80000, CRC(2cd58bee) SHA1(73defec823de4244a387af55fea7210edc1b314f) )
	ROM_LOAD32_BYTE( "mpr-14900.41", 0x000002, 0x80000, CRC(aa7c017d) SHA1(0fa2b59a8bb5f5907b2b2567e69d11c73b398dc1) )
	ROM_LOAD32_BYTE( "mpr-14901.42", 0x000003, 0x80000, CRC(175b7a9a) SHA1(c86602e771cd49bab425b4ba7926d2f44858bd39) )

	ROM_REGION( 0x20000, "commboard", 0 ) /* Comms Board */
	ROM_LOAD( "epr-15624.17", 0x00000, 0x20000, CRC(9b3ba315) SHA1(0cd0983cc8b2f2d6b41617d0d0a24cc6c188e62a) )
ROM_END


ROM_START( swa )
	MODEL1_CPU_BOARD

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16669.14", 0x200000, 0x80000, CRC(52e5e7e4) SHA1(0eb584299d821a61ed2c27c615356eaeb18b41c6) )
	ROM_LOAD16_BYTE( "epr-16670.15", 0x200001, 0x80000, CRC(1e7ecabd) SHA1(62e8dd5d3a053426a1f3d94d15b04621d36d6a2c) )

	ROM_LOAD( "epr-16668.5", 0xf80000, 0x80000, CRC(9e112425) SHA1(fc7be83b33b586780444e3426164e1cb5125f794) )
	ROM_RELOAD(          0x000000, 0x80000 )
	ROM_RELOAD(          0x080000, 0x80000 )

	ROM_REGION32_LE( 0x2000, "tgp_copro", 0)
	ROM_LOAD("315-5711.bin", 0, 0x2000, CRC(c5ddb8fc) SHA1(9e21d3a07ffa315e0139483b664e3fa283ef4e06) BAD_DUMP)

	ROM_REGION( 0xc0000, M1AUDIO_CPU_REGION, ROMREGION_BE|ROMREGION_16BIT )  /* 68K code */
	ROM_LOAD16_WORD_SWAP( "epr-16470.7", 0x000000, 0x020000, CRC(7da18cf7) SHA1(bd432d882d217277faee120e2577357a32eb4a6e) )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, M1AUDIO_MPCM1_REGION, 0 ) /* Samples */
	ROM_LOAD( "mpr-16486.32", 0x000000, 0x200000, CRC(7df50533) SHA1(f2fb876738e37d70eb9005e5629a9ae89aa413a8) )
	ROM_LOAD( "mpr-16487.33", 0x200000, 0x200000, CRC(31b28dfa) SHA1(bd1ac11bf2f9161f61f8af3b9ff4c2709b7ee700) )

	ROM_REGION( 0x400000, M1AUDIO_MPCM2_REGION, 0 ) /* Samples */
	ROM_LOAD( "mpr-16484.4", 0x000000, 0x200000, CRC(9d4c334d) SHA1(8b4d903f14559fed425d225bb23ccfe8da23cbd3) )
	ROM_LOAD( "mpr-16485.5", 0x200000, 0x200000, CRC(95aadcad) SHA1(4276db655db9834692c3843eb96a3e3a89cb7252) )

	ROM_REGION( 0x20000, "mpegcpu", 0 ) /* Z80 DSB code */
	ROM_LOAD( "epr-16471.2", 0x000000, 0x020000, CRC(f4ee84a4) SHA1(f12b214e6f195b0e5f49ba9f41d8e54bfcea9acc) )

	ROM_REGION( 0x800000, "mpeg", 0 ) /* DSB MPEG data */
	ROM_LOAD( "mpr-16514.57", 0x000000, 0x200000, CRC(3175b0be) SHA1(63649d053c8c17ce1746d16d0cc8202be20c302f) )
	ROM_LOAD( "mpr-16515.58", 0x200000, 0x200000, CRC(3114d748) SHA1(9ef090623cdd2a1d06b5d1bc4b9a07ab4eff5b76) )

	ROM_REGION32_LE( 0xc00000, "polygons", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16476.26", 0x000000, 0x200000, CRC(d48609ae) SHA1(8c8686a5c9ca4837447a7f70ed194e2f1882b66d) )
	ROM_LOAD32_WORD( "mpr-16477.27", 0x000002, 0x200000, CRC(b979b082) SHA1(0c60d259093e987f3856730b57b43bde7e9562e3) )
	ROM_LOAD32_WORD( "mpr-16478.28", 0x400000, 0x200000, CRC(80c780f7) SHA1(2f57c5373b02765d302bcd81e24f7b7bc4181387) )
	ROM_LOAD32_WORD( "mpr-16479.29", 0x400002, 0x200000, CRC(e43183b3) SHA1(4e62c67cdf7a6fdac0ded86d5f9e81044b9dea8d) )
	ROM_LOAD32_WORD( "mpr-16480.30", 0x800000, 0x200000, CRC(3185547a) SHA1(9871937372c2c755717802117a3ad39e1a11410e) )
	ROM_LOAD32_WORD( "mpr-16481.31", 0x800002, 0x200000, CRC(ce8d76fe) SHA1(0406f0500d19d6707515627b4143f92a9a5db769) )

	ROM_REGION32_LE( 0x200000, "copro_data", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16472.39", 0x000000, 0x80000, CRC(5a0d7553) SHA1(ba8e08e5a0c6b7fbc10084ad7ad3edf61efb0d70) )
	ROM_LOAD32_BYTE( "mpr-16473.40", 0x000001, 0x80000, CRC(876c5399) SHA1(be7e40c77a385600941f11c24852cd73c71696f0) )
	ROM_LOAD32_BYTE( "mpr-16474.41", 0x000002, 0x80000, CRC(5864a26f) SHA1(be0c22dfff37408f6b401b1970f7fcc6fc7fbcd2) )
	ROM_LOAD32_BYTE( "mpr-16475.42", 0x000003, 0x80000, CRC(b9266be9) SHA1(cf195cd89c9d191b9eb8c5299f8cc154c2b4bd82) )
ROM_END

ROM_START( swaj )
	MODEL1_CPU_BOARD

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16468.14", 0x200000, 0x80000, CRC(681d03c0) SHA1(4d21e26ce211466d429b84bca69a8147ff31ec6c) )
	ROM_LOAD16_BYTE( "epr-16469.15", 0x200001, 0x80000, CRC(6f281f7c) SHA1(6a9179e48d14838bb2a1a3f63fdd3a68ed009e03) )

	ROM_LOAD( "epr-16467.5", 0xf80000, 0x80000, CRC(605068f5) SHA1(99d7e171ce3353477c282d7567dedb9947206f14) )
	ROM_RELOAD(          0x000000, 0x80000 )
	ROM_RELOAD(          0x080000, 0x80000 )

	ROM_REGION32_LE( 0x2000, "tgp_copro", 0)
	ROM_LOAD("315-5711.bin", 0, 0x2000, CRC(c5ddb8fc) SHA1(9e21d3a07ffa315e0139483b664e3fa283ef4e06) BAD_DUMP )

	ROM_REGION( 0xc0000, M1AUDIO_CPU_REGION, ROMREGION_BE|ROMREGION_16BIT )  /* 68K code */
	ROM_LOAD16_WORD_SWAP( "epr-16470.7", 0x000000, 0x020000, CRC(7da18cf7) SHA1(bd432d882d217277faee120e2577357a32eb4a6e) )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, M1AUDIO_MPCM1_REGION, 0 ) /* Samples */
	ROM_LOAD( "mpr-16486.32", 0x000000, 0x200000, CRC(7df50533) SHA1(f2fb876738e37d70eb9005e5629a9ae89aa413a8) )
	ROM_LOAD( "mpr-16487.33", 0x200000, 0x200000, CRC(31b28dfa) SHA1(bd1ac11bf2f9161f61f8af3b9ff4c2709b7ee700) )

	ROM_REGION( 0x400000, M1AUDIO_MPCM2_REGION, 0 ) /* Samples */
	ROM_LOAD( "mpr-16484.4", 0x000000, 0x200000, CRC(9d4c334d) SHA1(8b4d903f14559fed425d225bb23ccfe8da23cbd3) )
	ROM_LOAD( "mpr-16485.5", 0x200000, 0x200000, CRC(95aadcad) SHA1(4276db655db9834692c3843eb96a3e3a89cb7252) )

	ROM_REGION( 0x20000, "mpegcpu", 0 ) /* Z80 DSB code */
	ROM_LOAD( "epr-16471.2", 0x000000, 0x020000, CRC(f4ee84a4) SHA1(f12b214e6f195b0e5f49ba9f41d8e54bfcea9acc) )

	ROM_REGION( 0x400000, "mpeg", 0 ) /* DSB MPEG data */
	ROM_LOAD( "mpr-16514.57", 0x000000, 0x200000, CRC(3175b0be) SHA1(63649d053c8c17ce1746d16d0cc8202be20c302f) )
	ROM_LOAD( "mpr-16515.58", 0x200000, 0x200000, CRC(3114d748) SHA1(9ef090623cdd2a1d06b5d1bc4b9a07ab4eff5b76) )

	ROM_REGION32_LE( 0xc00000, "polygons", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16476.26", 0x000000, 0x200000, CRC(d48609ae) SHA1(8c8686a5c9ca4837447a7f70ed194e2f1882b66d) )
	ROM_LOAD32_WORD( "mpr-16477.27", 0x000002, 0x200000, CRC(b979b082) SHA1(0c60d259093e987f3856730b57b43bde7e9562e3) )
	ROM_LOAD32_WORD( "mpr-16478.28", 0x400000, 0x200000, CRC(80c780f7) SHA1(2f57c5373b02765d302bcd81e24f7b7bc4181387) )
	ROM_LOAD32_WORD( "mpr-16479.29", 0x400002, 0x200000, CRC(e43183b3) SHA1(4e62c67cdf7a6fdac0ded86d5f9e81044b9dea8d) )
	ROM_LOAD32_WORD( "mpr-16480.30", 0x800000, 0x200000, CRC(3185547a) SHA1(9871937372c2c755717802117a3ad39e1a11410e) )
	ROM_LOAD32_WORD( "mpr-16481.31", 0x800002, 0x200000, CRC(ce8d76fe) SHA1(0406f0500d19d6707515627b4143f92a9a5db769) )

	ROM_REGION32_LE( 0x200000, "copro_data", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16472.39", 0x000000, 0x80000, CRC(5a0d7553) SHA1(ba8e08e5a0c6b7fbc10084ad7ad3edf61efb0d70) )
	ROM_LOAD32_BYTE( "mpr-16473.40", 0x000001, 0x80000, CRC(876c5399) SHA1(be7e40c77a385600941f11c24852cd73c71696f0) )
	ROM_LOAD32_BYTE( "mpr-16474.41", 0x000002, 0x80000, CRC(5864a26f) SHA1(be0c22dfff37408f6b401b1970f7fcc6fc7fbcd2) )
	ROM_LOAD32_BYTE( "mpr-16475.42", 0x000003, 0x80000, CRC(b9266be9) SHA1(cf195cd89c9d191b9eb8c5299f8cc154c2b4bd82) )
ROM_END


ROM_START( wingwar )
	MODEL1_CPU_BOARD

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16729.14", 0x200000, 0x80000, CRC(7edec2cc) SHA1(3e423a868ca7c8475fbb5bc1a10526e69d94d865) )
	ROM_LOAD16_BYTE( "epr-16730.15", 0x200001, 0x80000, CRC(bab24dee) SHA1(26c95139c1aa7f34b6a5cce39e5bd1dd2ef0dd49) )

	ROM_LOAD( "epr16953.4", 0xfc0000, 0x20000, CRC(c821a920) SHA1(7fc9ea5d828aac664514fa6d38f708f1ffd26220) )
	ROM_RELOAD(          0x000000, 0x20000 )
	ROM_LOAD( "epr16952.5", 0xfe0000, 0x20000, CRC(03a3ecc5) SHA1(5c4aa221302b0a0800e1af99a41ab46fe4325184) )
	ROM_RELOAD(          0x020000, 0x20000 )

	ROM_LOAD16_BYTE( "mpr-16738.6",  0x1000000, 0x80000, CRC(51518ffa) SHA1(e4674ddfed4205957b14e133c6fdf6454872f324) )
	ROM_LOAD16_BYTE( "mpr-16737.7",  0x1000001, 0x80000, CRC(37b1379c) SHA1(98620c324268e1dd906c077ac8a8cd903b9de1f7) )
	ROM_LOAD16_BYTE( "mpr-16736.8",  0x1100000, 0x80000, CRC(10b6a025) SHA1(7a4f624ceb7c0b92044a5db8ff55440562ef836b) )
	ROM_LOAD16_BYTE( "mpr-16735.9",  0x1100001, 0x80000, CRC(c82fd198) SHA1(d9e53ae1e14dfc8e84a14c0026ef0b904863bb1b) )
	ROM_LOAD16_BYTE( "mpr-16734.10", 0x1200000, 0x80000, CRC(f76371c1) SHA1(0ff082db3877383d0dd977dc60c932b725e3d164) )
	ROM_LOAD16_BYTE( "mpr-16733.11", 0x1200001, 0x80000, CRC(e105847b) SHA1(8489a6c91fd6d1e9ba81e8eaf36c514da30dccbe) )

	ROM_REGION32_LE( 0x2000, "tgp_copro", 0)
	ROM_LOAD("315-5711.bin", 0, 0x2000, CRC(c5ddb8fc) SHA1(9e21d3a07ffa315e0139483b664e3fa283ef4e06) BAD_DUMP )

	ROM_REGION( 0xc0000, M1AUDIO_CPU_REGION, ROMREGION_BE|ROMREGION_16BIT )  /* 68K code */
	ROM_LOAD16_WORD_SWAP("epr-17126.7", 0x000000, 0x20000, CRC(50178e40) SHA1(fb01aecfbe4e90adc997de0d45a63c16ef353b37) )
	ROM_LOAD16_WORD_SWAP("epr-16752.8", 0x020000, 0x20000, CRC(6541c48f) SHA1(9341eff160e31a8574b9545fafc1c4059323fa0c) )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, M1AUDIO_MPCM1_REGION, 0 ) /* Samples */
	ROM_LOAD("mpr-16753.32", 0x000000, 0x200000, CRC(324a8333) SHA1(960342e08db637c6f72615d49cffd9fb0889620b) )
	ROM_LOAD("mpr-16754.33", 0x200000, 0x200000, CRC(144f3cf5) SHA1(d2f8cc9086affbbc5ef2195272200230f724e5d1) )

	ROM_REGION( 0x400000, M1AUDIO_MPCM2_REGION, 0 ) /* Samples */
	ROM_LOAD("mpr-16755.4", 0x000000, 0x200000, CRC(4baaf878) SHA1(661d4ea9be6a4952852d0ef94becee7ed42bf4a1) )
	ROM_LOAD("mpr-16756.5", 0x200000, 0x200000, CRC(d9c40672) SHA1(83e6f1156b30888d3a00103f079dc74f4fca8446) )

	ROM_REGION32_LE( 0x1000000, "polygons", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16743.26", 0x000000, 0x200000, CRC(a710d33c) SHA1(1d0184545b34789ed511caaa25d57db3cd9a8e2f) )
	ROM_LOAD32_WORD( "mpr-16744.27", 0x000002, 0x200000, CRC(de796e1f) SHA1(397efb86a21b178770f29d2464bacf5f893619a0) )
	ROM_LOAD32_WORD( "mpr-16745.28", 0x400000, 0x200000, CRC(905b689c) SHA1(685dec2a65d5b3a386bda0af1bb5ae7e2b73a01a) )
	ROM_LOAD32_WORD( "mpr-16746.29", 0x400002, 0x200000, CRC(163b312e) SHA1(6b45007d6a9d17c8a0b46d81ec84ce9bfefb1695) )
	ROM_LOAD32_WORD( "mpr-16747.30", 0x800000, 0x200000, CRC(7353bb12) SHA1(608c5d561e909b8ba31d53db18e6e199855eaaec) )
	ROM_LOAD32_WORD( "mpr-16748.31", 0x800002, 0x200000, CRC(8ce98d3a) SHA1(1978776a0e2ea817508e30ba232d5f8d9c158f70) )
	ROM_LOAD32_WORD( "mpr-16749.32", 0xc00000, 0x200000, CRC(0e36dc1a) SHA1(4939177a6ac51ca57d0acd118ff14af4f4e438bb) )
	ROM_LOAD32_WORD( "mpr-16750.33", 0xc00002, 0x200000, CRC(e4f0b98d) SHA1(e69de2e5ccea2834fb8326bdd61fc6b517bc60b7) )

	ROM_REGION32_LE( 0x200000, "copro_data", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16741.39", 0x000000, 0x80000, CRC(84b2ffd8) SHA1(0eba3855d20b88567c6fa08046e12429664d87cb) )
	ROM_LOAD32_BYTE( "mpr-16742.40", 0x000001, 0x80000, CRC(e9cc12bb) SHA1(40c83c968be3b11fad193a00e7b760f074450683) )
	ROM_LOAD32_BYTE( "mpr-16739.41", 0x000002, 0x80000, CRC(6c73e98f) SHA1(7b31e62922ab6d0df97c3ecc52b78e6d086c8635) )
	ROM_LOAD32_BYTE( "mpr-16740.42", 0x000003, 0x80000, CRC(44b31007) SHA1(4bb265fea25a7bbcbb8ab080fdcf09849b18f1de) )
ROM_END

/*
wing War (US)

        Sega game ID# 833-10844-91-01
   Sega ROM board ID# 834-10845-91-01
Driver/Control BD ID# 837-10859 with EPR-16891
837-8679 MODEL-1 SOUND BD + 837-8680 SOUND BD OPTION daughterboard with ID# 837-10858
838-10141-04 Sound AMP board
*/
ROM_START( wingwaru )
	MODEL1_CPU_BOARD

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16729.14", 0x200000, 0x80000, CRC(7edec2cc) SHA1(3e423a868ca7c8475fbb5bc1a10526e69d94d865) )
	ROM_LOAD16_BYTE( "epr-16730.15", 0x200001, 0x80000, CRC(bab24dee) SHA1(26c95139c1aa7f34b6a5cce39e5bd1dd2ef0dd49) )

	ROM_LOAD( "epr-16951.4", 0xfc0000, 0x20000, BAD_DUMP CRC(8df5a798) SHA1(ef2756f237933ecf429dab0f362e572eb1965f4d) )
	ROM_RELOAD(          0x000000, 0x20000 )
	ROM_LOAD( "epr-16950.5", 0xfe0000, 0x20000, CRC(841e2195) SHA1(66f465aaf71955496e6f83335f7b836ad1d8c724) )
	ROM_RELOAD(          0x020000, 0x20000 )

	ROM_LOAD16_BYTE( "mpr-16738.6",  0x1000000, 0x80000, CRC(51518ffa) SHA1(e4674ddfed4205957b14e133c6fdf6454872f324) )
	ROM_LOAD16_BYTE( "mpr-16737.7",  0x1000001, 0x80000, CRC(37b1379c) SHA1(98620c324268e1dd906c077ac8a8cd903b9de1f7) )
	ROM_LOAD16_BYTE( "mpr-16736.8",  0x1100000, 0x80000, CRC(10b6a025) SHA1(7a4f624ceb7c0b92044a5db8ff55440562ef836b) )
	ROM_LOAD16_BYTE( "mpr-16735.9",  0x1100001, 0x80000, CRC(c82fd198) SHA1(d9e53ae1e14dfc8e84a14c0026ef0b904863bb1b) )
	ROM_LOAD16_BYTE( "mpr-16734.10", 0x1200000, 0x80000, CRC(f76371c1) SHA1(0ff082db3877383d0dd977dc60c932b725e3d164) )
	ROM_LOAD16_BYTE( "mpr-16733.11", 0x1200001, 0x80000, CRC(e105847b) SHA1(8489a6c91fd6d1e9ba81e8eaf36c514da30dccbe) )

	ROM_REGION32_LE( 0x2000, "tgp_copro", 0)
	ROM_LOAD("315-5711.bin", 0, 0x2000, CRC(c5ddb8fc) SHA1(9e21d3a07ffa315e0139483b664e3fa283ef4e06) BAD_DUMP )

	ROM_REGION( 0xc0000, M1AUDIO_CPU_REGION, ROMREGION_BE|ROMREGION_16BIT )  /* 68K code */
	ROM_LOAD16_WORD_SWAP("epr-16751.7", 0x000000, 0x20000, CRC(23ba5ebc) SHA1(b98aab546c5e980baeedbada4e7472eb4c588260) )
	ROM_LOAD16_WORD_SWAP("epr-16752.8", 0x020000, 0x20000, CRC(6541c48f) SHA1(9341eff160e31a8574b9545fafc1c4059323fa0c) )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, M1AUDIO_MPCM1_REGION, 0 ) /* Samples */
	ROM_LOAD("mpr-16753.32", 0x000000, 0x200000, CRC(324a8333) SHA1(960342e08db637c6f72615d49cffd9fb0889620b) )
	ROM_LOAD("mpr-16754.33", 0x200000, 0x200000, CRC(144f3cf5) SHA1(d2f8cc9086affbbc5ef2195272200230f724e5d1) )

	ROM_REGION( 0x400000, M1AUDIO_MPCM2_REGION, 0 ) /* Samples */
	ROM_LOAD("mpr-16755.4", 0x000000, 0x200000, CRC(4baaf878) SHA1(661d4ea9be6a4952852d0ef94becee7ed42bf4a1) )
	ROM_LOAD("mpr-16756.5", 0x200000, 0x200000, CRC(d9c40672) SHA1(83e6f1156b30888d3a00103f079dc74f4fca8446) )

	ROM_REGION32_LE( 0x1000000, "polygons", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16743.26", 0x000000, 0x200000, CRC(a710d33c) SHA1(1d0184545b34789ed511caaa25d57db3cd9a8e2f) )
	ROM_LOAD32_WORD( "mpr-16744.27", 0x000002, 0x200000, CRC(de796e1f) SHA1(397efb86a21b178770f29d2464bacf5f893619a0) )
	ROM_LOAD32_WORD( "mpr-16745.28", 0x400000, 0x200000, CRC(905b689c) SHA1(685dec2a65d5b3a386bda0af1bb5ae7e2b73a01a) )
	ROM_LOAD32_WORD( "mpr-16746.29", 0x400002, 0x200000, CRC(163b312e) SHA1(6b45007d6a9d17c8a0b46d81ec84ce9bfefb1695) )
	ROM_LOAD32_WORD( "mpr-16747.30", 0x800000, 0x200000, CRC(7353bb12) SHA1(608c5d561e909b8ba31d53db18e6e199855eaaec) )
	ROM_LOAD32_WORD( "mpr-16748.31", 0x800002, 0x200000, CRC(8ce98d3a) SHA1(1978776a0e2ea817508e30ba232d5f8d9c158f70) )
	ROM_LOAD32_WORD( "mpr-16749.32", 0xc00000, 0x200000, CRC(0e36dc1a) SHA1(4939177a6ac51ca57d0acd118ff14af4f4e438bb) )
	ROM_LOAD32_WORD( "mpr-16750.33", 0xc00002, 0x200000, CRC(e4f0b98d) SHA1(e69de2e5ccea2834fb8326bdd61fc6b517bc60b7) )

	ROM_REGION32_LE( 0x200000, "copro_data", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16741.39", 0x000000, 0x80000, CRC(84b2ffd8) SHA1(0eba3855d20b88567c6fa08046e12429664d87cb) )
	ROM_LOAD32_BYTE( "mpr-16742.40", 0x000001, 0x80000, CRC(e9cc12bb) SHA1(40c83c968be3b11fad193a00e7b760f074450683) )
	ROM_LOAD32_BYTE( "mpr-16739.41", 0x000002, 0x80000, CRC(6c73e98f) SHA1(7b31e62922ab6d0df97c3ecc52b78e6d086c8635) )
	ROM_LOAD32_BYTE( "mpr-16740.42", 0x000003, 0x80000, CRC(44b31007) SHA1(4bb265fea25a7bbcbb8ab080fdcf09849b18f1de) )
ROM_END

ROM_START( wingwarj )
	MODEL1_CPU_BOARD

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16729.14", 0x200000, 0x80000, CRC(7edec2cc) SHA1(3e423a868ca7c8475fbb5bc1a10526e69d94d865) )
	ROM_LOAD16_BYTE( "epr-16730.15", 0x200001, 0x80000, CRC(bab24dee) SHA1(26c95139c1aa7f34b6a5cce39e5bd1dd2ef0dd49) )

	ROM_LOAD( "epr-16728.4", 0xfc0000, 0x20000, CRC(194d58ff) SHA1(281201cbb9bb148a06fe88edeccf680f837dcc10) )
	ROM_RELOAD(          0x000000, 0x20000 )
	ROM_LOAD( "epr-16727.5", 0xfe0000, 0x20000, CRC(430380eb) SHA1(25bd81a69a9b4cdc644f912283b65b3eab988327) )
	ROM_RELOAD(          0x020000, 0x20000 )

	ROM_LOAD16_BYTE( "mpr-16738.6",  0x1000000, 0x80000, CRC(51518ffa) SHA1(e4674ddfed4205957b14e133c6fdf6454872f324) )
	ROM_LOAD16_BYTE( "mpr-16737.7",  0x1000001, 0x80000, CRC(37b1379c) SHA1(98620c324268e1dd906c077ac8a8cd903b9de1f7) )
	ROM_LOAD16_BYTE( "mpr-16736.8",  0x1100000, 0x80000, CRC(10b6a025) SHA1(7a4f624ceb7c0b92044a5db8ff55440562ef836b) )
	ROM_LOAD16_BYTE( "mpr-16735.9",  0x1100001, 0x80000, CRC(c82fd198) SHA1(d9e53ae1e14dfc8e84a14c0026ef0b904863bb1b) )
	ROM_LOAD16_BYTE( "mpr-16734.10", 0x1200000, 0x80000, CRC(f76371c1) SHA1(0ff082db3877383d0dd977dc60c932b725e3d164) )
	ROM_LOAD16_BYTE( "mpr-16733.11", 0x1200001, 0x80000, CRC(e105847b) SHA1(8489a6c91fd6d1e9ba81e8eaf36c514da30dccbe) )

	ROM_REGION32_LE( 0x2000, "tgp_copro", 0)
	ROM_LOAD("315-5711.bin", 0, 0x2000, CRC(c5ddb8fc) SHA1(9e21d3a07ffa315e0139483b664e3fa283ef4e06) BAD_DUMP )

	ROM_REGION( 0xc0000, M1AUDIO_CPU_REGION, ROMREGION_BE|ROMREGION_16BIT )  /* 68K code */
	ROM_LOAD16_WORD_SWAP("epr-16751.7", 0x000000, 0x20000, CRC(23ba5ebc) SHA1(b98aab546c5e980baeedbada4e7472eb4c588260) )
	ROM_LOAD16_WORD_SWAP("epr-16752.8", 0x020000, 0x20000, CRC(6541c48f) SHA1(9341eff160e31a8574b9545fafc1c4059323fa0c) )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, M1AUDIO_MPCM1_REGION, 0 ) /* Samples */
	ROM_LOAD("mpr-16753.32", 0x000000, 0x200000, CRC(324a8333) SHA1(960342e08db637c6f72615d49cffd9fb0889620b) )
	ROM_LOAD("mpr-16754.33", 0x200000, 0x200000, CRC(144f3cf5) SHA1(d2f8cc9086affbbc5ef2195272200230f724e5d1) )

	ROM_REGION( 0x400000, M1AUDIO_MPCM2_REGION, 0 ) /* Samples */
	ROM_LOAD("mpr-16755.4", 0x000000, 0x200000, CRC(4baaf878) SHA1(661d4ea9be6a4952852d0ef94becee7ed42bf4a1) )
	ROM_LOAD("mpr-16756.5", 0x200000, 0x200000, CRC(d9c40672) SHA1(83e6f1156b30888d3a00103f079dc74f4fca8446) )

	ROM_REGION32_LE( 0x1000000, "polygons", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16743.26", 0x000000, 0x200000, CRC(a710d33c) SHA1(1d0184545b34789ed511caaa25d57db3cd9a8e2f) )
	ROM_LOAD32_WORD( "mpr-16744.27", 0x000002, 0x200000, CRC(de796e1f) SHA1(397efb86a21b178770f29d2464bacf5f893619a0) )
	ROM_LOAD32_WORD( "mpr-16745.28", 0x400000, 0x200000, CRC(905b689c) SHA1(685dec2a65d5b3a386bda0af1bb5ae7e2b73a01a) )
	ROM_LOAD32_WORD( "mpr-16746.29", 0x400002, 0x200000, CRC(163b312e) SHA1(6b45007d6a9d17c8a0b46d81ec84ce9bfefb1695) )
	ROM_LOAD32_WORD( "mpr-16747.30", 0x800000, 0x200000, CRC(7353bb12) SHA1(608c5d561e909b8ba31d53db18e6e199855eaaec) )
	ROM_LOAD32_WORD( "mpr-16748.31", 0x800002, 0x200000, CRC(8ce98d3a) SHA1(1978776a0e2ea817508e30ba232d5f8d9c158f70) )
	ROM_LOAD32_WORD( "mpr-16749.32", 0xc00000, 0x200000, CRC(0e36dc1a) SHA1(4939177a6ac51ca57d0acd118ff14af4f4e438bb) )
	ROM_LOAD32_WORD( "mpr-16750.33", 0xc00002, 0x200000, CRC(e4f0b98d) SHA1(e69de2e5ccea2834fb8326bdd61fc6b517bc60b7) )

	ROM_REGION32_LE( 0x200000, "copro_data", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16741.39", 0x000000, 0x80000, CRC(84b2ffd8) SHA1(0eba3855d20b88567c6fa08046e12429664d87cb) )
	ROM_LOAD32_BYTE( "mpr-16742.40", 0x000001, 0x80000, CRC(e9cc12bb) SHA1(40c83c968be3b11fad193a00e7b760f074450683) )
	ROM_LOAD32_BYTE( "mpr-16739.41", 0x000002, 0x80000, CRC(6c73e98f) SHA1(7b31e62922ab6d0df97c3ecc52b78e6d086c8635) )
	ROM_LOAD32_BYTE( "mpr-16740.42", 0x000003, 0x80000, CRC(44b31007) SHA1(4bb265fea25a7bbcbb8ab080fdcf09849b18f1de) )
ROM_END

// Dumper's note: As per other R360 games wingwar is just a slightly modified version with 4 roms different on the rom board.

ROM_START( wingwar360 )
	MODEL1_CPU_BOARD

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16729.14", 0x200000, 0x80000, CRC(7edec2cc) SHA1(3e423a868ca7c8475fbb5bc1a10526e69d94d865) )
	ROM_LOAD16_BYTE( "epr-16730.15", 0x200001, 0x80000, CRC(bab24dee) SHA1(26c95139c1aa7f34b6a5cce39e5bd1dd2ef0dd49) )

	ROM_LOAD( "epr-17052.4", 0xfc0000, 0x20000, CRC(0f4743e7) SHA1(cc47fd1d25808728ed05d95d510733b8bd011b41) )
	ROM_RELOAD(          0x000000, 0x20000 )
	ROM_LOAD( "epr-17053.5", 0xfe0000, 0x20000, CRC(83af2415) SHA1(46dfee9db95171a3942cd32c851ec75c3d9e03da) )
	ROM_RELOAD(          0x020000, 0x20000 )

	ROM_LOAD16_BYTE( "ic6_17056.bin",  0x1000000, 0x80000, CRC(5216de4d) SHA1(1463311d3f96ca7c46b8f676ee3963caddeec9e2) )
	ROM_LOAD16_BYTE( "ic7_17057.bin",  0x1000001, 0x80000, CRC(b21aa2db) SHA1(861e637b5ae9b2d7fe6d09a7e788a6a3ff09af6e) )
	ROM_LOAD16_BYTE( "mpr-16736.8",  0x1100000, 0x80000, CRC(10b6a025) SHA1(7a4f624ceb7c0b92044a5db8ff55440562ef836b) )
	ROM_LOAD16_BYTE( "mpr-16735.9",  0x1100001, 0x80000, CRC(c82fd198) SHA1(d9e53ae1e14dfc8e84a14c0026ef0b904863bb1b) )
	ROM_LOAD16_BYTE( "mpr-16734.10", 0x1200000, 0x80000, CRC(f76371c1) SHA1(0ff082db3877383d0dd977dc60c932b725e3d164) )
	ROM_LOAD16_BYTE( "mpr-16733.11", 0x1200001, 0x80000, CRC(e105847b) SHA1(8489a6c91fd6d1e9ba81e8eaf36c514da30dccbe) )

	ROM_REGION32_LE( 0x2000, "tgp_copro", 0)
	ROM_LOAD("315-5711.bin", 0, 0x2000, CRC(c5ddb8fc) SHA1(9e21d3a07ffa315e0139483b664e3fa283ef4e06) BAD_DUMP )

	ROM_REGION( 0xc0000, M1AUDIO_CPU_REGION, ROMREGION_BE|ROMREGION_16BIT )  /* 68K code */
	ROM_LOAD16_WORD_SWAP("epr-17126.7", 0x000000, 0x20000, CRC(50178e40) SHA1(fb01aecfbe4e90adc997de0d45a63c16ef353b37) )
	ROM_LOAD16_WORD_SWAP("epr-16752.8", 0x020000, 0x20000, CRC(6541c48f) SHA1(9341eff160e31a8574b9545fafc1c4059323fa0c) )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, M1AUDIO_MPCM1_REGION, 0 ) /* Samples */
	ROM_LOAD("mpr-16753.32", 0x000000, 0x200000, CRC(324a8333) SHA1(960342e08db637c6f72615d49cffd9fb0889620b) )
	ROM_LOAD("mpr-16754.33", 0x200000, 0x200000, CRC(144f3cf5) SHA1(d2f8cc9086affbbc5ef2195272200230f724e5d1) )

	ROM_REGION( 0x400000, M1AUDIO_MPCM2_REGION, 0 ) /* Samples */
	ROM_LOAD("mpr-16755.4", 0x000000, 0x200000, CRC(4baaf878) SHA1(661d4ea9be6a4952852d0ef94becee7ed42bf4a1) )
	ROM_LOAD("mpr-16756.5", 0x200000, 0x200000, CRC(d9c40672) SHA1(83e6f1156b30888d3a00103f079dc74f4fca8446) )

	ROM_REGION32_LE( 0x1000000, "polygons", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16743.26", 0x000000, 0x200000, CRC(a710d33c) SHA1(1d0184545b34789ed511caaa25d57db3cd9a8e2f) )
	ROM_LOAD32_WORD( "mpr-16744.27", 0x000002, 0x200000, CRC(de796e1f) SHA1(397efb86a21b178770f29d2464bacf5f893619a0) )
	ROM_LOAD32_WORD( "mpr-16745.28", 0x400000, 0x200000, CRC(905b689c) SHA1(685dec2a65d5b3a386bda0af1bb5ae7e2b73a01a) )
	ROM_LOAD32_WORD( "mpr-16746.29", 0x400002, 0x200000, CRC(163b312e) SHA1(6b45007d6a9d17c8a0b46d81ec84ce9bfefb1695) )
	ROM_LOAD32_WORD( "mpr-16747.30", 0x800000, 0x200000, CRC(7353bb12) SHA1(608c5d561e909b8ba31d53db18e6e199855eaaec) )
	ROM_LOAD32_WORD( "mpr-16748.31", 0x800002, 0x200000, CRC(8ce98d3a) SHA1(1978776a0e2ea817508e30ba232d5f8d9c158f70) )
	ROM_LOAD32_WORD( "mpr-16749.32", 0xc00000, 0x200000, CRC(0e36dc1a) SHA1(4939177a6ac51ca57d0acd118ff14af4f4e438bb) )
	ROM_LOAD32_WORD( "mpr-16750.33", 0xc00002, 0x200000, CRC(e4f0b98d) SHA1(e69de2e5ccea2834fb8326bdd61fc6b517bc60b7) )

	ROM_REGION32_LE( 0x200000, "copro_data", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16741.39", 0x000000, 0x80000, CRC(84b2ffd8) SHA1(0eba3855d20b88567c6fa08046e12429664d87cb) )
	ROM_LOAD32_BYTE( "mpr-16742.40", 0x000001, 0x80000, CRC(e9cc12bb) SHA1(40c83c968be3b11fad193a00e7b760f074450683) )
	ROM_LOAD32_BYTE( "mpr-16739.41", 0x000002, 0x80000, CRC(6c73e98f) SHA1(7b31e62922ab6d0df97c3ecc52b78e6d086c8635) )
	ROM_LOAD32_BYTE( "mpr-16740.42", 0x000003, 0x80000, CRC(44b31007) SHA1(4bb265fea25a7bbcbb8ab080fdcf09849b18f1de) )

	// Dumper's note: Video & Drive is the control board in the attendants' tower, same hardware as G-Loc R360 with the two program roms being the only difference.
	ROM_REGION( 0x400000, "controlboard", 0 )
	ROM_LOAD("ic22_18851.bin", 0x00000, 0x20000, CRC(85f75bd7) SHA1(43cc8f8c81631d71b661e55e15f3fe8803a8a7e9) )
	ROM_LOAD("ic67_18850.bin", 0x20000, 0x08000, CRC(45a3e9fc) SHA1(bf2d1a7acc35b72a62551bfbb55e5058d6fd6d5c) )
ROM_END

ROM_START( netmerc )
	MODEL1_CPU_BOARD

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD( "epr-18120.ic5", 0xf80000, 0x80000, CRC(de489762) SHA1(4e935f11e844489c5b5dc06439dc7902c4fddc9f) ) /* Rom board with Sega ID# ??? */
	ROM_RELOAD(          0x000000, 0x80000 )

	ROM_LOAD16_BYTE( "epr-18122.ic6",  0x1000000, 0x80000, CRC(10581ab3) SHA1(e3f5ec4ce14a1e1a1134c8b516c371b7655f3544) )
	ROM_LOAD16_BYTE( "epr-18123.ic7",  0x1000001, 0x80000, CRC(27e0d848) SHA1(117f99efd0643ba92b3340171d8952b6a1a568d4) )
	ROM_LOAD16_BYTE( "epr-18124.ic8",  0x1100000, 0x80000, CRC(5a0e5618) SHA1(a092d05a414532aa27d4bae908b053da4255462a) )
	ROM_LOAD16_BYTE( "epr-18125.ic9",  0x1100001, 0x80000, CRC(bd47284f) SHA1(ae9325e6789bb956ab664c7ad9c48b80056f3916) )
	ROM_LOAD16_BYTE( "epr-18126.ic10", 0x1200000, 0x80000, CRC(43dc5ca8) SHA1(249ed83b90b0237ceabbed814cd42dc60dc7a174) )
	ROM_LOAD16_BYTE( "epr-18127.ic11", 0x1200001, 0x80000, CRC(d307a4ca) SHA1(5555235f740c1b09f6e1587d0fceb35b23d4a8a8) )

	ROM_REGION32_LE( 0x2000, "tgp_copro", 0)
	ROM_LOAD("315-5711.bin", 0, 0x2000, CRC(c5ddb8fc) SHA1(9e21d3a07ffa315e0139483b664e3fa283ef4e06) BAD_DUMP )

	ROM_REGION( 0xc0000, M1AUDIO_CPU_REGION, ROMREGION_BE|ROMREGION_16BIT )  /* 68K code */
	ROM_LOAD16_WORD_SWAP( "epr-18121.ic7", 0x00000, 0x80000, CRC(113285b5) SHA1(5d060cee41e8d6a4a918f890c2d169d87dbcad79) )
	ROM_RELOAD( 0x80000, 0x20000) // ?

	ROM_REGION( 0x400000, M1AUDIO_MPCM1_REGION, 0 ) /* Samples */
	ROM_LOAD( "mpr-18134.ic32", 0x000000, 0x200000, BAD_DUMP CRC(9a4109e5) SHA1(ba59caac5f5a80fc52c507d8a47f322a380aa9a1) )
	ROM_LOAD( "mpr-18135.ic33", 0x200000, 0x200000, CRC(291d17e4) SHA1(7465d5bde9ece5112932efa58b3c1c07f2b11543) )

	ROM_REGION( 0x400000, M1AUDIO_MPCM2_REGION, 0 ) /* Samples */
	ROM_LOAD( "mpr-18136.ic4", 0x000000, 0x200000, CRC(2a9c0aaf) SHA1(e8c52a6032ae968afcc41b3d8bfd15065057ea4a) )

	ROM_REGION32_LE( 0x1000000, "polygons", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-18128.ic26", 0x000000, 0x200000, CRC(70d8028c) SHA1(6e55028be778cd246f67e9d8cd69b9f5366bc429) )
	ROM_LOAD32_WORD( "mpr-18129.ic27", 0x000002, 0x200000, CRC(134feea3) SHA1(afaf9fa08c5b0ab8c9d5fdef78d829e8582cc61a) )
	ROM_LOAD32_WORD( "mpr-18130.ic28", 0x400000, 0x200000, CRC(c7aeae3e) SHA1(4f636178f903cd03e1b9de73a11d1105b7cdb084) )
	ROM_LOAD32_WORD( "mpr-18131.ic29", 0x400002, 0x200000, CRC(32683b33) SHA1(daf808853d9d85dff87eba5c081f47b4af4ae0a2) )
	ROM_LOAD32_WORD( "mpr-18132.ic30", 0x800000, 0x200000, CRC(a17e3ac2) SHA1(19827c06ebc3e9de63668ef07675224e169d853e) )
	ROM_LOAD32_WORD( "mpr-18133.ic31", 0x800002, 0x200000, CRC(f56354dd) SHA1(2ef1fe8b4995a67b70b565adf8f0ea0ad6e10094) )

	ROM_REGION32_LE( 0x200000, "copro_data", ROMREGION_ERASE00 ) // IC39-IC42 unpopulated

	ROM_REGION( 0x8000, "polhemus", 0 ) /* POLHEMUS board */
	ROM_LOAD16_BYTE( "u1", 0x0000, 0x4000, CRC(7073a312) SHA1(d2582f9520b8c8c051708dd372633112af59206e) )
	ROM_LOAD16_BYTE( "u2", 0x0001, 0x4000, CRC(c589f428) SHA1(98dc0114a5f89636b4e237ed954e19f1cfd186ab) )
ROM_END

void model1_state::model1(machine_config &config)
{
	V60(config, m_maincpu, 16000000);
	m_maincpu->set_addrmap(AS_PROGRAM, &model1_state::model1_mem);
	m_maincpu->set_addrmap(AS_IO, &model1_state::model1_io);
	m_maincpu->set_irq_acknowledge_callback(FUNC(model1_state::irq_callback));

	GENERIC_FIFO_U32(config, "copro_fifo_in", 0);
	GENERIC_FIFO_U32(config, "copro_fifo_out", 0);

	TIMER(config, "scantimer").configure_scanline(FUNC(model1_state::model1_interrupt), "screen", 0, 1);

	MB86233(config, m_tgp_copro, 16000000);
	m_tgp_copro->set_addrmap(AS_PROGRAM, &model1_state::copro_prog_map);
	m_tgp_copro->set_addrmap(AS_DATA, &model1_state::copro_data_map);
	m_tgp_copro->set_addrmap(AS_IO, &model1_state::copro_io_map);
	m_tgp_copro->set_addrmap(mb86233_device::AS_RF, &model1_state::copro_rf_map);

	S24TILE(config, m_tiles, 0, 0x3fff).set_palette(m_palette);

	SCREEN(config, m_screen, SCREEN_TYPE_RASTER);
	m_screen->set_video_attributes(VIDEO_UPDATE_AFTER_VBLANK);
	m_screen->set_raw(XTAL(16'000'000), 656, 0/*+69*/, 496/*+69*/, 424, 0/*+25*/, 384/*+25*/);
	m_screen->set_screen_update(FUNC(model1_state::screen_update_model1));
	m_screen->screen_vblank().set(FUNC(model1_state::screen_vblank_model1));

	PALETTE(config, m_palette).set_format(palette_device::xBGR_555, 8192);

	// create SEGA_MODEL1IO device *after* SCREEN device
	model1io_device &ioboard(SEGA_MODEL1IO(config, "ioboard", 0));
	ioboard.read_callback().set(m_dpram, FUNC(mb8421_device::left_r));
	ioboard.write_callback().set(m_dpram, FUNC(mb8421_device::left_w));
	ioboard.in_callback<0>().set_ioport("IN.0");
	ioboard.in_callback<1>().set_ioport("IN.1");

	MB8421(config, m_dpram, 0);

	SEGAM1AUDIO(config, m_m1audio, 0);
	m_m1audio->rxd_handler().set(m_m1uart, FUNC(i8251_device::write_rxd));

	I8251(config, m_m1uart, 8000000); // uPD71051C, clock unknown
	m_m1uart->txd_handler().set(m_m1audio, FUNC(segam1audio_device::write_txd));

	clock_device &m1uart_clock(CLOCK(config, "m1uart_clock", 500000)); // 16 times 31.25MHz (standard Sega/MIDI sound data rate)
	m1uart_clock.signal_handler().set(m_m1uart, FUNC(i8251_device::write_txc));
	m1uart_clock.signal_handler().append(m_m1uart, FUNC(i8251_device::write_rxc));
}

void model1_state::vf(machine_config &config)
{
	model1(config);

	model1io_device &ioboard(*subdevice<model1io_device>("ioboard"));
	ioboard.set_default_bios_tag("epr14869b");
	ioboard.in_callback<2>().set_ioport("IN.2");
	ioboard.output_callback().set(FUNC(model1_state::vf_outputs_w));
	ioboard.output_callback().append(FUNC(model1_state::gen_outputs_w));
}

void model1_state::vr(machine_config &config)
{
	model1(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &model1_state::model1_comm_mem);

	model1io_device &ioboard(*subdevice<model1io_device>("ioboard"));
	ioboard.drive_write_callback().set(FUNC(model1_state::drive_board_w));
	ioboard.an_callback<0>().set_ioport("WHEEL");
	ioboard.an_callback<1>().set_ioport("ACCEL");
	ioboard.an_callback<2>().set_ioport("BRAKE");
	ioboard.output_callback().set(FUNC(model1_state::vr_outputs_w));
	ioboard.output_callback().append(FUNC(model1_state::gen_outputs_w));

	M1COMM(config, "m1comm", 0).set_default_bios_tag("epr15112");
}

void model1_state::vformula(machine_config &config)
{
	model1(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &model1_state::model1_comm_mem);

	model1io_device &ioboard(*subdevice<model1io_device>("ioboard"));
	ioboard.drive_write_callback().set(FUNC(model1_state::drive_board_w));
	ioboard.an_callback<0>().set_ioport("WHEEL");
	ioboard.an_callback<1>().set_ioport("ACCEL");
	ioboard.an_callback<2>().set_ioport("BRAKE");
	ioboard.output_callback().set(FUNC(model1_state::vr_outputs_w));
	ioboard.output_callback().append(FUNC(model1_state::gen_outputs_w));

	M1COMM(config, "m1comm", 0).set_default_bios_tag("epr15624");
}

void model1_state::swa(machine_config &config)
{
	model1(config);

	model1io_device &ioboard(*subdevice<model1io_device>("ioboard"));
	ioboard.set_default_bios_tag("epr14869b");
	ioboard.an_callback<0>().set_ioport("STICK1X");
	ioboard.an_callback<1>().set_ioport("STICK1Y");
	ioboard.an_callback<2>().set_ioport("THROTTLE");
	ioboard.an_callback<4>().set_ioport("STICK2X");
	ioboard.an_callback<5>().set_ioport("STICK2Y");
	ioboard.output_callback().set(FUNC(model1_state::swa_outputs_w));
	ioboard.output_callback().append(FUNC(model1_state::gen_outputs_w));

	SPEAKER(config, "dleft").front_left();
	SPEAKER(config, "dright").front_right();
	DSBZ80(config, m_dsbz80, 0);
	m_dsbz80->add_route(0, "dleft", 1.0);
	m_dsbz80->add_route(1, "dright", 1.0);

	// Apparently m1audio has to filter out commands the DSB shouldn't see
	m_m1audio->rxd_handler().append(m_dsbz80, FUNC(dsbz80_device::write_txd));
}

void model1_state::wingwar(machine_config &config)
{
	model1(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &model1_state::model1_comm_mem);

	model1io2_device &ioboard(SEGA_MODEL1IO2(config.replace(), "ioboard", 0));
	ioboard.set_default_bios_tag("epr16891");
	ioboard.read_callback().set(m_dpram, FUNC(mb8421_device::left_r));
	ioboard.write_callback().set(m_dpram, FUNC(mb8421_device::left_w));
	ioboard.in_callback<0>().set_ioport("IN.0");
	ioboard.in_callback<1>().set_ioport("IN.1");
	ioboard.an_callback<0>().set_ioport("STICKX");
	ioboard.an_callback<1>().set_ioport("STICKY");
	ioboard.an_callback<2>().set_ioport("THROTTLE");
	ioboard.output_callback().set(FUNC(model1_state::wingwar_outputs_w));
	ioboard.output_callback().append(FUNC(model1_state::gen_outputs_w));

	config.set_default_layout(layout_model1io2);

	M1COMM(config, "m1comm", 0).set_default_bios_tag("epr15112");
}

void model1_state::wingwar360(machine_config &config)
{
	wingwar(config);

	model1io2_device &ioboard(*subdevice<model1io2_device>("ioboard"));
	ioboard.in_callback<2>().set(FUNC(model1_state::r360_r));
	ioboard.drive_write_callback().set(FUNC(model1_state::r360_w));
	ioboard.an_callback<2>().set_constant(0);
	ioboard.output_callback().set(FUNC(model1_state::wingwar360_outputs_w));
	ioboard.output_callback().append(FUNC(model1_state::gen_outputs_w));

	config.set_default_layout(layout_model1io2);
}

void model1_state::polhemus_map(address_map &map)
{
	map.global_mask(0xfffff);
	map(0x00000, 0x03fff).ram();
	map(0xf8000, 0xfffff).rom().region("polhemus", 0);
}

void model1_state::netmerc(machine_config &config)
{
	model1(config);
	i386sx_device &polhemus(I386SX(config, "polhemus", 16000000));
	polhemus.set_addrmap(AS_PROGRAM, &model1_state::polhemus_map);

	model1io2_device &ioboard(SEGA_MODEL1IO2(config.replace(), "ioboard", 0));
	ioboard.set_default_bios_tag("epr18021");
	ioboard.read_callback().set(m_dpram, FUNC(mb8421_device::left_r));
	ioboard.write_callback().set(m_dpram, FUNC(mb8421_device::left_w));
	ioboard.in_callback<0>().set_ioport("IN.0");
	ioboard.in_callback<1>().set_ioport("IN.1");
	ioboard.an_callback<0>().set_ioport("STICKX");
	ioboard.an_callback<2>().set_ioport("STICKY");
	ioboard.output_callback().set(FUNC(model1_state::netmerc_outputs_w));
	ioboard.output_callback().append(FUNC(model1_state::gen_outputs_w));

	config.set_default_layout(layout_model1io2);
}


//**************************************************************************
//  SYSTEM DRIVERS
//**************************************************************************

//    YEAR  NAME        PARENT   MACHINE     INPUT       CLASS         INIT        ROTATION  COMPANY  FULLNAME              FLAGS
GAME( 1993, vf,         0,       vf,         vf,         model1_state, empty_init, ROT0,     "Sega",  "Virtua Fighter",           MACHINE_NOT_WORKING )
GAMEL(1992, vr,         0,       vr,         vr,         model1_state, empty_init, ROT0,     "Sega",  "Virtua Racing",            0, layout_vr )
GAME( 1993, vformula,   vr,      vformula,   vr,         model1_state, empty_init, ROT0,     "Sega",  "Virtua Formula",           MACHINE_IMPERFECT_GRAPHICS )
GAME( 1993, swa,        0,       swa,        swa,        model1_state, empty_init, ROT0,     "Sega",  "Star Wars Arcade (US)",    MACHINE_IMPERFECT_GRAPHICS | MACHINE_IMPERFECT_CONTROLS )
GAME( 1993, swaj,       swa,     swa,        swa,        model1_state, empty_init, ROT0,     "Sega",  "Star Wars Arcade (Japan)", MACHINE_IMPERFECT_GRAPHICS | MACHINE_IMPERFECT_CONTROLS )
GAME( 1994, wingwar,    0,       wingwar,    wingwar,    model1_state, empty_init, ROT0,     "Sega",  "Wing War (World)",         0 )
GAME( 1994, wingwaru,   wingwar, wingwar,    wingwar,    model1_state, empty_init, ROT0,     "Sega",  "Wing War (US)",            0 )
GAME( 1994, wingwarj,   wingwar, wingwar,    wingwar,    model1_state, empty_init, ROT0,     "Sega",  "Wing War (Japan)",         0 )
GAME( 1994, wingwar360, wingwar, wingwar360, wingwar360, model1_state, empty_init, ROT0,     "Sega",  "Wing War R360 (US)",       0 )
GAME( 1993, netmerc,    0,       netmerc,    netmerc,    model1_state, empty_init, ROT0,     "Sega",  "Sega NetMerc",             MACHINE_NOT_WORKING )
