/*
 * Potamus: an audio player
 * Copyright (C) 2004, 2005, 2006, 2007, 2013 Adam Sampson <ats@offog.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <FLAC/stream_decoder.h>
#include <stdlib.h>
#include "buffer.h"
#include "format.h"
#include "input.h"
#include "input-flac.h"

typedef struct {
	FLAC__StreamDecoder *dec;
	buffer *buf;
	int seen_audio;
	int seen_format;
	int seen_total;
	int seen_position;
	int had_error;
	FLAC__uint64 position;
	FLAC__uint64 total_samples;
	FLAC__uint64 seek_dest;
	int seeking;
} aoflac;

static FLAC__StreamDecoderWriteStatus write_cb
    (const FLAC__StreamDecoder *decoder, const FLAC__Frame *frame,
     const FLAC__int32 *const buffer[], void *data) {
	input *p = (input *) data;
	aoflac *a = (aoflac *) p->data;

	if (a->buf == NULL)
		g_error("write_cb called with NULL buffer");

	const FLAC__FrameHeader *fh = &frame->header;
	p->fmt.bits = fh->bits_per_sample;
	p->fmt.rate = fh->sample_rate;
	p->fmt.channels = fh->channels;
	p->fmt.byte_format = END_LITTLE;
	a->seen_format = 1;

	if (fh->number_type == FLAC__FRAME_NUMBER_TYPE_SAMPLE_NUMBER) {
		a->position = fh->number.sample_number;
	} else {
		// This is only approximate.
		a->position = fh->number.frame_number * fh->blocksize;
	}
	a->seen_position = 1;

	const FLAC__int32 *in[fh->channels];
	for (int j = 0; j < fh->channels; j++)
		in[j] = buffer[j];

	const int nbytes = bytes_per_sample(fh->bits_per_sample);

	size_t out_size = nbytes * fh->blocksize * fh->channels;
	unsigned char *out = buffer_reserve(a->buf, out_size);
	if (out == NULL)
		g_error("out of memory");

	for (int i = 0; i < fh->blocksize; i++) {
		for (int j = 0; j < fh->channels; j++) {
			FLAC__int32 sample = *in[j]++;
			int n = nbytes;
			if (n == 4)
				sample <<= 8;
			while (n-- > 0) {
				*out++ = sample & 0xFF;
				sample >>= 8;
			}
		}
	}
	a->seen_audio = 1;
	a->buf->used += out_size;

	return FLAC__STREAM_DECODER_WRITE_STATUS_CONTINUE;
}

void metadata_cb(const FLAC__StreamDecoder *decoder,
                 const FLAC__StreamMetadata *metadata, void *data) {
	input *p = (input *) data;
	aoflac *a = (aoflac *) p->data;

	if (metadata->type != FLAC__METADATA_TYPE_STREAMINFO)
		return;

	const FLAC__StreamMetadata_StreamInfo *si = &metadata->data.stream_info;
	p->fmt.bits = si->bits_per_sample;
	p->fmt.rate = si->sample_rate;
	p->fmt.channels = si->channels;
	p->fmt.byte_format = END_LITTLE;
	a->seen_format = 1;
	a->total_samples = si->total_samples;
	a->seen_total = 1;
}

void error_cb(const FLAC__StreamDecoder *decoder,
              FLAC__StreamDecoderErrorStatus status, void *data) {
	input *p = (input *) data;
	aoflac *a = (aoflac *) p->data;

	a->had_error = 1;
}

static int flac_open(input *p, const char *fn) {
	aoflac *a = malloc(sizeof *a);
	if (a == NULL)
		g_error("out of memory");
	p->data = a;

	a->dec = FLAC__stream_decoder_new();
	if (a->dec == NULL)
		g_error("out of memory");

	a->buf = NULL;
	a->seen_format = a->seen_position = a->seen_total = a->seen_audio = 0;
	a->seeking = 0;

	if (FLAC__stream_decoder_init_file(a->dec, fn,
	                                   write_cb, metadata_cb,
	                                   error_cb, p)
	    != FLAC__STREAM_DECODER_INIT_STATUS_OK)
		return -1;

	return 0;
}

static int flac_get_audio(input *p, buffer *buf) {
	aoflac *a = (aoflac *) p->data;
	a->buf = buf;
	size_t old_used = buf->used;
	a->seen_audio = 0;
	a->had_error = 0;

	if (a->seeking) {
		// We have to do this here because it might call the write
		// callback.
		if (!FLAC__stream_decoder_seek_absolute(a->dec, a->seek_dest))
			goto out;
		a->seeking = 0;
	}

	int playing = 1;
	do {
		if (!FLAC__stream_decoder_process_single(a->dec))
			goto out;

		switch (FLAC__stream_decoder_get_state(a->dec)) {
		case FLAC__STREAM_DECODER_SEARCH_FOR_METADATA:
		case FLAC__STREAM_DECODER_READ_METADATA:
		case FLAC__STREAM_DECODER_SEARCH_FOR_FRAME_SYNC:
		case FLAC__STREAM_DECODER_READ_FRAME:
			// No problems.
			break;
		case FLAC__STREAM_DECODER_END_OF_STREAM:
			playing = 0;
			break;
		case FLAC__STREAM_DECODER_OGG_ERROR:
		case FLAC__STREAM_DECODER_SEEK_ERROR:
		case FLAC__STREAM_DECODER_ABORTED:
		case FLAC__STREAM_DECODER_MEMORY_ALLOCATION_ERROR:
		case FLAC__STREAM_DECODER_UNINITIALIZED:
			// Some error.
			goto out;
		}
	} while (playing && !(a->seen_audio && a->seen_format));

	a->buf = NULL;
	return buf->used - old_used;

out:
	a->buf = NULL;
	return -1;
}

static int flac_get_pos(input *p, double *pos) {
	aoflac *a = (aoflac *) p->data;

	if (!(a->seen_format && a->seen_position))
		return -1;

	*pos = (1.0L * a->position) / p->fmt.rate;

	return 0;
}

static int flac_get_len(input *p, double *len) {
	aoflac *a = (aoflac *) p->data;

	if (!(a->seen_format && a->seen_total))
		return -1;

	*len = (1.0L * a->total_samples) / p->fmt.rate;

	return 0;
}

static int flac_get_seekable(input *p) {
	aoflac *a = (aoflac *) p->data;

	return a->seen_format;
}

static int flac_set_pos(input *p, double pos) {
	aoflac *a = (aoflac *) p->data;

	a->seek_dest = pos * p->fmt.rate;
	// Don't try to seek past the end of the stream.
	if (a->seek_dest >= a->total_samples)
		a->seek_dest = a->total_samples - 1;

	a->seeking = 1;

	return 0;
}

static int flac_close(input *p) {
	aoflac *a = (aoflac *) p->data;

	FLAC__stream_decoder_finish(a->dec);
	FLAC__stream_decoder_delete(a->dec);

	free(a);
	free(p);

	return 0;
}

input *input_new_flac(void) {
	input *p = input_alloc();

	p->open = flac_open;
	p->get_audio = flac_get_audio;
	p->get_pos = flac_get_pos;
	p->get_len = flac_get_len;
	p->get_seekable = flac_get_seekable;
	p->set_pos = flac_set_pos;
	p->close = flac_close;

	return p;
}
