/*
 * Potamus: an audio player
 * Copyright (C) 2004, 2005, 2006, 2007, 2013 Adam Sampson <ats@offog.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <libmodplug/modplug.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "buffer.h"
#include "format.h"
#include "input.h"
#include "input-modplug.h"

typedef struct {
	ModPlugFile *mp;
	void *mod_data;
	long pos;
} aomodplug;

static int modplug_open(input *p, const char *fn) {
	ModPlug_Settings settings;
	ModPlug_GetSettings(&settings);
	settings.mFlags |= MODPLUG_ENABLE_OVERSAMPLING;
	settings.mChannels = 2;
	// FIXME Figure out how to get 24-bit output working
	// (I'm not entirely convinced it's worth it...)
	settings.mBits = 16;
	settings.mFrequency = 44100;
	settings.mResamplingMode = MODPLUG_RESAMPLE_FIR;
	settings.mLoopCount = 0;
	ModPlug_SetSettings(&settings);

	p->fmt.bits = 16;
	p->fmt.rate = 44100;
	p->fmt.channels = 2;
	p->fmt.byte_format = END_NATIVE; // I assume...

	aomodplug *a = malloc(sizeof *a);
	if (a == NULL)
		g_error("out of memory");
	a->mp = NULL;
	a->mod_data = NULL;
	a->pos = 0;
	p->data = a;

	FILE *f = fopen(fn, "r");
	if (f == NULL)
		return -1;

	struct stat st;
	if (fstat(fileno(f), &st) < 0) {
		fclose(f);
		return -1;
	}

	a->mod_data = malloc(st.st_size);
	if (a->mod_data == NULL)
		g_error("out of memory");

	if (fread(a->mod_data, 1, st.st_size, f) != st.st_size) {
		fclose(f);
		return -1;
	}

	fclose(f);

	a->mp = ModPlug_Load(a->mod_data, st.st_size);
	if (a->mp == NULL)
		return -1;

	return 0;
}

static int modplug_get_audio(input *p, buffer *buf) {
	aomodplug *a = (aomodplug *) p->data;

	const int chunk_size = 65536;
	unsigned char *out = buffer_reserve(buf, chunk_size);
	if (out == NULL)
		g_error("out of memory");

	int n = ModPlug_Read(a->mp, out, chunk_size);
	buf->used += n;
	a->pos += n / (p->fmt.channels * bytes_per_sample(p->fmt.bits));

	return n;
}

static int modplug_get_pos(input *p, double *pos) {
	aomodplug *a = (aomodplug *) p->data;

	*pos = (a->pos * 1.0L) / p->fmt.rate;

	return 0;
}

static int modplug_get_len(input *p, double *len) {
	aomodplug *a = (aomodplug *) p->data;

	*len = ModPlug_GetLength(a->mp) / 1000.0L;

	return 0;
}

static int modplug_get_seekable(input *p) {
	return 1;
}

static int modplug_set_pos(input *p, double pos) {
	aomodplug *a = (aomodplug *) p->data;

	ModPlug_Seek(a->mp, pos * 1000);
	a->pos = pos * p->fmt.rate;

	return 0;
}

static int modplug_close(input *p) {
	aomodplug *a = (aomodplug *) p->data;

	if (a->mp != NULL)
		ModPlug_Unload(a->mp);
	free(a->mod_data);
	free(a);
	free(p);

	return 0;
}

input *input_new_modplug(void) {
	input *p = input_alloc();

	p->open = modplug_open;
	p->get_audio = modplug_get_audio;
	p->get_pos = modplug_get_pos;
	p->get_len = modplug_get_len;
	p->get_seekable = modplug_get_seekable;
	p->set_pos = modplug_set_pos;
	p->close = modplug_close;

	return p;
}
