/*
 * Potamus: an audio player
 * Copyright (C) 2013 Adam Sampson <ats@offog.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <opusfile.h>
#include <stdlib.h>
#include "buffer.h"
#include "format.h"
#include "input.h"
#include "input-opus.h"

// Opus always runs at 48kHz.
const int SAMPLE_RATE = 48000;

typedef struct {
	OggOpusFile *of;
} aoopus;

static int opus_open(input *p, const char *fn) {
	aoopus *a = malloc(sizeof *a);
	if (a == NULL)
		g_error("out of memory");
	p->data = a;

	a->of = op_open_file(fn, NULL);
	if (a->of == NULL)
		return -1;

	return 0;
}

static int opus_get_audio(input *p, buffer *buf) {
	aoopus *a = (aoopus *) p->data;

	const int chunk_size = 65536;
	opus_int16 *out = (opus_int16 *) buffer_reserve(buf, chunk_size * sizeof(opus_int16));
	if (out == NULL)
		g_error("out of memory");

	int link;
	int samples = op_read(a->of, out, chunk_size, &link);
	if (samples < 0)
		return -1;
	if (samples == 0)
		return 0;

	const OpusHead *head = op_head(a->of, link);
	int bytes = samples * head->channel_count * sizeof(opus_int16);
	buf->used += bytes;

	p->bitrate = op_bitrate_instant(a->of) / 1000.0;

	p->fmt.bits = 16;
	p->fmt.rate = SAMPLE_RATE;
	p->fmt.channels = head->channel_count;
	p->fmt.byte_format = END_LITTLE;

	return bytes;
}

static int opus_get_pos(input *p, double *pos) {
	aoopus *a = (aoopus *) p->data;

	*pos = (1.0 * op_pcm_tell(a->of)) / SAMPLE_RATE;

	return 0;
}

static int opus_get_len(input *p, double *len) {
	aoopus *a = (aoopus *) p->data;

	*len = (1.0 * op_pcm_total(a->of, -1)) / SAMPLE_RATE;

	return 0;
}

static int opus_get_seekable(input *p) {
	aoopus *a = (aoopus *) p->data;

	return op_seekable(a->of) != 0;
}

static int opus_set_pos(input *p, double pos) {
	aoopus *a = (aoopus *) p->data;

	if (op_pcm_seek(a->of, pos * SAMPLE_RATE) != 0)
		return -1;

	return 0;
}

static int opus_close(input *p) {
	aoopus *a = (aoopus *) p->data;

	op_free(a->of);
	free(a);
	free(p);

	return 0;
}

input *input_new_opus(void) {
	input *p = input_alloc();

	p->open = opus_open;
	p->get_audio = opus_get_audio;
	p->get_pos = opus_get_pos;
	p->get_len = opus_get_len;
	p->get_seekable = opus_get_seekable;
	p->set_pos = opus_set_pos;
	p->close = opus_close;

	return p;
}
