/***************************************************************************
            qgsspatialiteprovider.h Data provider for SpatiaLite DBMS
begin                : Dec 2008
copyright            : (C) 2008 Sandro Furieri
email                : a.furieri@lqt.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef QGSSPATIALITEPROVIDER_H
#define QGSSPATIALITEPROVIDER_H

extern "C"
{
#include <sys/types.h>
#include <sqlite3.h>
#include <spatialite/gaiageo.h>
#include <spatialite.h>
}

#include "qgsvectordataprovider.h"
#include "qgsrectangle.h"
#include "qgsvectorlayerexporter.h"
#include "qgsfields.h"
#include "qgswkbtypes.h"

#include <list>
#include <queue>
#include <fstream>
#include <set>

class QgsFeature;
class QgsField;

class QgsSqliteHandle;
class QgsSpatiaLiteFeatureIterator;

#include "qgsdatasourceuri.h"

/**
  \class QgsSpatiaLiteProvider
  \brief Data provider for SQLite/SpatiaLite layers.

  This provider implements the
  interface defined in the QgsDataProvider class to provide access to spatial
  data residing in a SQLite/SpatiaLite enabled database.
  */
class QgsSpatiaLiteProvider: public QgsVectorDataProvider
{
    Q_OBJECT

  public:
    //! Import a vector layer into the database
    static QgsVectorLayerExporter::ExportError createEmptyLayer(
      const QString &uri,
      const QgsFields &fields,
      QgsWkbTypes::Type wkbType,
      const QgsCoordinateReferenceSystem &srs,
      bool overwrite,
      QMap<int, int> *oldToNewAttrIdxMap,
      QString *errorMessage = nullptr,
      const QMap<QString, QVariant> *options = nullptr
    );

    /**
     * Constructor of the vector provider
     * \param uri  uniform resource locator (URI) for a dataset
     */
    explicit QgsSpatiaLiteProvider( QString const &uri = QString() );

    ~ QgsSpatiaLiteProvider() override;

    QgsAbstractFeatureSource *featureSource() const override;
    QString storageType() const override;
    QgsCoordinateReferenceSystem crs() const override;
    QgsFeatureIterator getFeatures( const QgsFeatureRequest &request ) const override;
    QString subsetString() const override;
    bool setSubsetString( const QString &theSQL, bool updateFeatureCount = true ) override;
    bool supportsSubsetString() const override { return true; }
    QgsWkbTypes::Type wkbType() const override;

    /**
     * Return the number of layers for the current data source
     *
     * \note Should this be subLayerCount() instead?
     */
    size_t layerCount() const;

    long featureCount() const override;
    QgsRectangle extent() const override;
    void updateExtents() override;
    QgsFields fields() const override;
    QVariant minimumValue( int index ) const override;
    QVariant maximumValue( int index ) const override;
    QSet<QVariant> uniqueValues( int index, int limit = -1 ) const override;
    QStringList uniqueStringsMatching( int index, const QString &substring, int limit = -1,
                                       QgsFeedback *feedback = nullptr ) const override;

    bool isValid() const override;
    bool isSaveAndLoadStyleToDatabaseSupported() const override { return true; }
    bool addFeatures( QgsFeatureList &flist, QgsFeatureSink::Flags flags = nullptr ) override;
    bool deleteFeatures( const QgsFeatureIds &id ) override;
    bool truncate() override;
    bool addAttributes( const QList<QgsField> &attributes ) override;
    bool changeAttributeValues( const QgsChangedAttributesMap &attr_map ) override;
    bool changeGeometryValues( const QgsGeometryMap &geometry_map ) override;
    QgsVectorDataProvider::Capabilities capabilities() const override;
    QVariant defaultValue( int fieldId ) const override;
    bool skipConstraintCheck( int fieldIndex, QgsFieldConstraints::Constraint constraint, const QVariant &value = QVariant() ) const override;
    bool createAttributeIndex( int field ) override;

    /**
     * The SpatiaLite provider does its own transforms so we return
     * true for the following three functions to indicate that transforms
     * should not be handled by the QgsCoordinateTransform object. See the
     * documentation on QgsVectorDataProvider for details on these functions.
     */
    // XXX For now we have disabled native transforms in the SpatiaLite
    //   (following the PostgreSQL provider example)
    bool supportsNativeTransform()
    {
      return false;
    }

    QString name() const override;
    QString description() const override;
    QgsAttributeList pkAttributeIndexes() const override;
    void invalidateConnections( const QString &connection ) override;
    QList<QgsRelation> discoverRelations( const QgsVectorLayer *self, const QList<QgsVectorLayer *> &layers ) const override;

    // static functions
    static void convertToGeosWKB( const unsigned char *blob, int blob_size,
                                  unsigned char **wkb, int *geom_size );
    static int computeMultiWKB3Dsize( const unsigned char *p_in, int little_endian,
                                      int endian_arch );
    static QString quotedIdentifier( QString id );
    static QString quotedValue( QString value );

    struct SLFieldNotFound {}; //! Exception to throw

    struct SLException
    {
        explicit SLException( char *msg ) : errMsg( msg )
        {
        }

        SLException( const SLException &e ) : errMsg( e.errMsg )
        {
        }

        ~SLException()
        {
          if ( errMsg )
            sqlite3_free( errMsg );
        }

        SLException &operator=( const SLException &other ) = delete;

        QString errorMessage() const
        {
          return errMsg ? QString::fromUtf8( errMsg ) : QStringLiteral( "unknown cause" );
        }

      private:
        char *errMsg = nullptr;

    };

    /**
     * sqlite3 handles pointer
     */
    QgsSqliteHandle *mHandle = nullptr;

  signals:

    /**
     *   This is emitted when this provider is satisfied that all objects
     *   have had a chance to adjust themselves after they'd been notified that
     *   the full extent is available.
     *
     *   \note  It currently isn't being emitted because we don't have an easy way
     *          for the overview canvas to only be repainted.  In the meantime
     *          we are satisfied for the overview to reflect the new extent
     *          when the user adjusts the extent of the main map canvas.
     */
    void repaintRequested();

  private:

    //! Loads fields from input file to member mAttributeFields
    void loadFields();

    //! For views, try to get primary key from a dedicated meta table
    void determineViewPrimaryKey();

    //! Check if a table/view has any triggers.  Triggers can be used on views to make them editable.
    bool hasTriggers();

    //! Check if a table has a row id (internal primary key)
    bool hasRowid();

    //! Convert a QgsField to work with SL
    static bool convertField( QgsField &field );

    QString geomParam() const;

    //! get SpatiaLite version string
    QString spatialiteVersion();

    /**
     * Search all the layers using the given table.
     */
    static QList<QgsVectorLayer *> searchLayers( const QList<QgsVectorLayer *> &layers, const QString &connectionInfo, const QString &tableName );

    QgsFields mAttributeFields;

    //! Flag indicating if the layer data source is a valid SpatiaLite layer
    bool mValid = false;

    //! Flag indicating if the layer data source is based on a query
    bool mIsQuery = false;

    //! Flag indicating if the layer data source is based on a plain Table
    bool mTableBased = false;

    //! Flag indicating if the layer data source is based on a View
    bool mViewBased = false;

    //! Flag indicating if the layer data source is based on a VirtualShape
    bool mVShapeBased = false;

    //! Flag indicating if the layer data source has ReadOnly restrictions
    bool mReadOnly = false;

    //! DB full path
    QString mSqlitePath;

    //! Name of the table with no schema
    QString mTableName;

    //! Name of the table or subquery
    QString mQuery;

    //! Name of the primary key column in the table
    QString mPrimaryKey;

    //! Flag indicating whether the primary key is auto-generated
    bool mPrimaryKeyAutoIncrement = false;

    //! List of primary key columns in the table
    QgsAttributeList mPrimaryKeyAttrs;

    //! Name of the geometry column in the table
    QString mGeometryColumn;

    //! Map of field index to default value
    QMap<int, QVariant> mDefaultValues;

    //! Name of the SpatialIndex table
    QString mIndexTable;

    //! Name of the SpatialIndex geometry column
    QString mIndexGeometry;

    //! Geometry type
    QgsWkbTypes::Type mGeomType = QgsWkbTypes::Unknown;

    //! SQLite handle
    sqlite3 *mSqliteHandle = nullptr;

    //! String used to define a subset of the layer
    QString mSubsetString;

    //! CoordDimensions of the layer
    int nDims;

    //! Spatial reference id of the layer
    int mSrid = -1;

    //! auth id
    QString mAuthId;

    //! proj4text
    QString mProj4text;

    //! Rectangle that contains the extent (bounding box) of the layer
    QgsRectangle mLayerExtent;

    //! Number of features in the layer
    long mNumberFeatures = 0;

    //! this Geometry is supported by an R*Tree spatial index
    bool mSpatialIndexRTree = false;

    //! this Geometry is supported by an MBR cache spatial index
    bool mSpatialIndexMbrCache = false;

    QgsVectorDataProvider::Capabilities mEnabledCapabilities = nullptr;

    QgsField field( int index ) const;

    //! SpatiaLite version string
    QString mSpatialiteVersionInfo;

    //! Are mSpatialiteVersionMajor, mSpatialiteVersionMinor valid?
    bool mGotSpatialiteVersion = false;

    //! SpatiaLite major version
    int mSpatialiteVersionMajor = 0;

    //! SpatiaLite minor version
    int mSpatialiteVersionMinor = 0;

    /**
     * internal utility functions used to handle common SQLite tasks
     */
    //void sqliteOpen();
    void closeDb();
    bool checkLayerType();
    bool getGeometryDetails();
    bool getTableGeometryDetails();
    bool getViewGeometryDetails();
    bool getVShapeGeometryDetails();
    bool getQueryGeometryDetails();
    bool getSridDetails();
    bool getTableSummary();
    bool checkLayerTypeAbstractInterface( gaiaVectorLayerPtr lyr );
    bool getGeometryDetailsAbstractInterface( gaiaVectorLayerPtr lyr );
    bool getTableSummaryAbstractInterface( gaiaVectorLayerPtr lyr );
    void loadFieldsAbstractInterface( gaiaVectorLayerPtr lyr );
    void getViewSpatialIndexName();
    bool prepareStatement( sqlite3_stmt *&stmt,
                           const QgsAttributeList &fetchAttributes,
                           bool fetchGeometry,
                           QString whereClause );
    bool getFeature( sqlite3_stmt *stmt, bool fetchGeometry,
                     QgsFeature &feature,
                     const QgsAttributeList &fetchAttributes );

    void updatePrimaryKeyCapabilities();

    int computeSizeFromMultiWKB2D( const unsigned char *p_in, int nDims,
                                   int little_endian,
                                   int endian_arch );
    int computeSizeFromMultiWKB3D( const unsigned char *p_in, int nDims,
                                   int little_endian,
                                   int endian_arch );
    void convertFromGeosWKB2D( const unsigned char *blob, int blob_size,
                               unsigned char *wkb, int geom_size,
                               int nDims, int little_endian, int endian_arch );
    void convertFromGeosWKB3D( const unsigned char *blob, int blob_size,
                               unsigned char *wkb, int geom_size,
                               int nDims, int little_endian, int endian_arch );
    void convertFromGeosWKB( const unsigned char *blob, int blob_size,
                             unsigned char **wkb, int *geom_size,
                             int dims );
    int computeSizeFromGeosWKB3D( const unsigned char *blob, int size,
                                  QgsWkbTypes::Type type, int nDims, int little_endian,
                                  int endian_arch );
    int computeSizeFromGeosWKB2D( const unsigned char *blob, int size,
                                  QgsWkbTypes::Type type, int nDims, int little_endian,
                                  int endian_arch );

    void fetchConstraints();

    void insertDefaultValue( int fieldIndex, QString defaultVal );

    /**
     * Handles an error encountered while executing an sql statement.
     */
    void handleError( const QString &sql, char *errorMessage, bool rollback = false );

    friend class QgsSpatiaLiteFeatureSource;

};

// clazy:excludeall=qstring-allocations

#endif
