/*
 * This file is part of roccat-tools.
 *
 * roccat-tools is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Macro Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * roccat-tools is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Macro Public License for more details.
 *
 * You should have received a copy of the GNU Macro Public License
 * along with roccat-tools. If not, see <http://www.gnu.org/licenses/>.
 */

#include "sova_keys_easyzone.h"
#include "roccat_helper.h"
#include "sova_device.h"
#include <string.h>

static guint8 const default_keys_easyzone[sizeof(SovaKeysEasyzone)] = {
	0x0B, 0x26, 0x01, 0x00, 0x1E, 0x00, 0x00, 0x1F, 0x00, 0x00, 0x20, 0x00, 0x00, 0x21, 0x00, 0x00,
	0x22, 0x00, 0x00, 0x23, 0x00, 0x00, 0x24, 0x00, 0x00, 0x25, 0x00, 0x00, 0x26, 0x00, 0x00, 0x27,
	0x00, 0x00, 0x14, 0x00, 0x00, 0x1A, 0x00, 0x00, 0x08, 0x00, 0x00, 0x15, 0x00, 0x00, 0x17, 0x00,
	0x00, 0x1C, 0x00, 0x00, 0x18, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x12, 0x00, 0x00, 0x13, 0x00, 0x00,
	0x04, 0x00, 0x00, 0x16, 0x00, 0x00, 0x07, 0x00, 0x00, 0x09, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x0B,
	0x00, 0x00, 0x0D, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x0F, 0x00, 0x00, 0x33, 0x00, 0x00, 0x1D, 0x00,
	0x00, 0x1B, 0x00, 0x00, 0x06, 0x00, 0x00, 0x19, 0x00, 0x00, 0x05, 0x00, 0x00, 0x11, 0x00, 0x00,
	0x10, 0x00, 0x00, 0x36, 0x00, 0x00, 0x37, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x43, 0x00, 0x00,
	0x44, 0x00, 0x00, 0x45, 0x00, 0x00, 0x46, 0x00, 0x00, 0x49, 0x00, 0x00, 0x4C, 0x00, 0x00, 0xD2,
	0x00, 0x00, 0xD1, 0x00, 0x00, 0xD0, 0x00, 0x00, 0xC1, 0x00, 0x00, 0xCC, 0x00, 0x00, 0xCF, 0x00,
	0x00, 0xCE, 0x00, 0x00, 0xCD, 0x00, 0x00, 0xC2, 0x00, 0x00, 0xC3, 0x00, 0x00, 0xA3, 0x00, 0x00,
	0xA7, 0x00, 0x00, 0x48, 0x00, 0x00, 0x53, 0x00, 0x00, 0x47, 0x00, 0x00, 0x57, 0x0E, 0x07, 0x5F,
	0x02, 0x3A, 0x00, 0x00, 0x3B, 0x00, 0x00, 0x3C, 0x00, 0x00, 0x3D, 0x00, 0x00, 0x3E, 0x00, 0x00,
	0x3F, 0x00, 0x00, 0x40, 0x00, 0x00, 0x41, 0x00, 0x00, 0x42, 0x00, 0x00, 0x43, 0x00, 0x00, 0x44,
	0x00, 0x00, 0x45, 0x00, 0x00, 0x46, 0x00, 0x00, 0x49, 0x00, 0x00, 0x4C, 0x00, 0x00, 0xD2, 0x00,
	0x00, 0xD1, 0x00, 0x00, 0xD0, 0x00, 0x00, 0xC1, 0x00, 0x00, 0xCC, 0x00, 0x00, 0xCF, 0x00, 0x00,
	0xCE, 0x00, 0x00, 0xCD, 0x00, 0x00, 0xC2, 0x00, 0x00, 0xC3, 0x00, 0x00, 0xA3, 0x00, 0x00, 0xA7,
	0x00, 0x00, 0x48, 0x00, 0x00, 0x53, 0x00, 0x00, 0x47, 0x00, 0x00, 0x58, 0x0E, 0x07, 0x5F, 0x03,
	0x3A, 0x00, 0x00, 0x3B, 0xD3, 0x1F
};

SovaKeysEasyzone const *sova_keys_easyzone_default(void) {
	return (SovaKeysEasyzone const *)default_keys_easyzone;
}

static guint16 sova_keys_easyzone_calc_checksum(SovaKeysEasyzone const *keys_easyzone) {
	return ROCCAT_BYTESUM_PARTIALLY(keys_easyzone, SovaKeysEasyzone, report_id, checksum);
}

static void sova_keys_easyzone_set_checksum(SovaKeysEasyzone *keys_easyzone, guint16 new_value) {
	keys_easyzone->checksum = GUINT16_TO_LE(new_value);
}

static void sova_keys_easyzone_finalize(SovaKeysEasyzone *keys_easyzone, guint profile_index) {
	keys_easyzone->report_id = SOVA_REPORT_ID_KEYS_EASYZONE;
	keys_easyzone->size = sizeof(SovaKeysEasyzone);
	keys_easyzone->profile_index = profile_index;
	sova_keys_easyzone_set_checksum(keys_easyzone, sova_keys_easyzone_calc_checksum(keys_easyzone));
}

gboolean sova_keys_easyzone_write(RoccatDevice *device, guint profile_index, SovaKeysEasyzone *keys_easyzone, GError **error) {
	g_assert(profile_index < SOVA_PROFILE_NUM);
	sova_keys_easyzone_finalize(keys_easyzone, profile_index);
	return sova_device_write(device, (gchar const *)keys_easyzone, sizeof(SovaKeysEasyzone), error);
}

SovaKeysEasyzone *sova_keys_easyzone_read(RoccatDevice *device, guint profile_index, GError **error) {
	SovaKeysEasyzone *keys_easyzone;

	g_assert(profile_index < SOVA_PROFILE_NUM);

	gaminggear_device_lock(GAMINGGEAR_DEVICE(device));

	if (!sova_select(device, profile_index, SOVA_CONTROL_REQUEST_KEYS_EASYZONE, error)) {
		gaminggear_device_unlock(GAMINGGEAR_DEVICE(device));
		return NULL;
	}

	keys_easyzone = (SovaKeysEasyzone *)sova_device_read(device, SOVA_REPORT_ID_KEYS_EASYZONE, sizeof(SovaKeysEasyzone), error);

	gaminggear_device_unlock(GAMINGGEAR_DEVICE(device));

	return keys_easyzone;
}

gboolean sova_keys_easyzone_equal(SovaKeysEasyzone const *left, SovaKeysEasyzone const *right) {
	gboolean equal;
	equal = ROCCAT_MEMCMP_PARTIALLY(left, right, SovaKeysEasyzone, keys[0], checksum);
	return equal ? FALSE : TRUE;
}

void sova_keys_easyzone_copy(SovaKeysEasyzone *destination, SovaKeysEasyzone const *source) {
	memcpy(destination, source, sizeof(SovaKeysEasyzone));
}

guint sova_keys_easyzone_index_to_macro_index(guint easyzone_index) {
	return easyzone_index + 0x10;
}

guint sova_macro_index_to_keys_easyzone_index(guint macro_index) {
	return macro_index - 0x10;
}

gboolean sova_macro_index_is_keys_easyzone(guint macro_index) {
	return macro_index >= 0x10 && macro_index <= 0x6f;
}
