/*
 * Copyright 2020 - 2021 José Expósito <jose.exposito89@gmail.com>
 *
 * This file is part of Touché.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation,  either version 3 of the License,  or (at your option)  any later
 * version.
 *
 * This program is distributed in the hope that it will be useful,  but  WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the  GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
const fs = require('fs');
const path = require('path');
const readline = require('readline');
const { exec } = require('child_process');

const rootPath = process.cwd();
const srcPath = path.resolve(rootPath, 'src');
const i18nPath = path.resolve(rootPath, 'po');

const askForLocale = async () => new Promise((resolve, reject) => {
  console.log('This tool will help you to generate or update [locale].po files.');
  console.log('Locale is a 2 or 3 digit code as defined in the "ISO 639-1 Code" column here:');
  console.log('https://www.loc.gov/standards/iso639-2/php/code_list.php\n');
  console.log('You can also select the locale by running this script like:');
  console.log('$ npm run translate -- [locale]\n');

  let locale = null;

  const rl = readline.createInterface({
    input: process.stdin,
    output: process.stdout,
  });

  rl.question('Enter the name of the locale to create of update: ', (input) => {
    locale = input;
    rl.close();
  });

  rl.on('close', () => {
    console.log();
    if (locale) {
      resolve(locale);
    } else {
      reject();
    }
  });
});

const getLinguasLocales = () => {
  const linguasPath = path.resolve(i18nPath, 'LINGUAS');

  if (!fs.existsSync(linguasPath)) {
    return [];
  }

  const content = fs.readFileSync(linguasPath, 'utf-8');
  return content.replace(/#.*/g, '') // Remove comments
    .replace(/(\r\n|\r|\n)/g, ' ') // Replace new lines with spaces
    .replace(/\s+/g, ' ') // Replace multiple spaces with a single space
    .split(' ')
    .filter((item) => item !== '');
};

const addLocaleToLinguas = (locale) => {
  const linguasHeader = '# This file contains the list of available translation.\n'
    + '# For more information please visit:\n'
    + '# https://www.gnu.org/software/gettext/manual/html_node/po_002fLINGUAS.html\n'
    + '# Do NOT edit this file manually. Any change will be overridden.\n'
    + '# Use the translation command instead:\n'
    + '# $ npm run translate\n\n';
  const locales = getLinguasLocales();
  locales.push(locale);
  const linguasLocales = Array.from(new Set(locales)).join(' ');
  const linguasContent = `${linguasHeader}${linguasLocales}\n`;
  fs.writeFileSync(path.resolve(i18nPath, 'LINGUAS'), linguasContent, 'utf-8');

  console.log('File po/LINGUAS updated');
};

const generatePoFile = async (locale) => new Promise((resolve, reject) => {
  const poPath = path.resolve(i18nPath, `${locale}.po`);
  const exists = fs.existsSync(poPath);

  console.log(`${exists ? 'Updating' : 'Creating'} po/${locale}.po`);

  const command = `xgettext ${exists ? '--join-existing --omit-header --no-location' : '--no-location'} --output-dir=${i18nPath} --output=${poPath} --from-code=UTF-8 \`find ${srcPath} -name "*js"\``;
  exec(command, { cwd: rootPath },
    (error) => {
      if (error) {
        console.error(`Error generating po/${locale}.po: ${error}`);
        reject(error);
      } else {
        console.log(`File po/${locale}.po ${exists ? 'updated' : 'created'} successfully`);
        resolve();
      }
    });
});

const main = async () => {
  const locale = (process.argv.length > 2)
    ? process.argv[2]
    : await askForLocale();

  console.log(`Generating translations for locale "${locale}"`);
  await generatePoFile(locale);
  addLocaleToLinguas(locale);
  console.log('Translations generated successfully 🎉');
};

main();
