// $Id: NE.cc 1.7 Fri, 18 Jul 1997 15:53:23 -0700 wlee $
// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu (cengiz@isi.edu)

#include "config.hh"
#include <iostream.h>
#include "NE.hh"
#include "debug.hh"

#define DBG_REDUCE 3
#define DBG_REDUCE_DETAIL 4
#define DBG_OR 5
#define DBG_NOT 6

CLASS_DEBUG_MEMORY_CC(NormalExpression);

NormalExpression::NormalExpression(NormalExpression& a) {
   singleton_flag = a.singleton_flag;
   for (Pix i = a.terms.first(); i; a.terms.next(i))
      terms.append(new NormalTerm(*a.terms(i)));
}

ostream& operator<<(ostream& stream, NormalExpression& ne) {
   NormalTerm *term;
   Pix i;

   switch (ne.any()) {
   case ANY: 
      stream << "ANY";
      break;
   case NOT_ANY:
      stream << "NOT ANY";
      break;
   default:
      for (i = ne.terms.first(); i; ) {
	 term = ne.terms(i);
	 stream << "(" << *term << ")";
	 ne.terms.next(i);
	 if (i)
	    stream << "\n OR ";
      }
   }
   return stream;
}

void NormalExpression::reduce() {
   NormalTerm *term, *otherterm;
   int change = 1;
   Pix pixi, pixj, nextpixj;
   int j;

   // worst case O(N^3) Sigh.
   // can be improved by keeping sorted lists on set sizes
   while (change) {
      change = 0;
      for (pixi = terms.first(); pixi; terms.next(pixi)) {
	 term = terms(pixi);
	 for (pixj = pixi, terms.next(pixj); pixj; pixj = nextpixj) {
	    otherterm = terms(pixj);
	    nextpixj = pixj;
	    terms.next(nextpixj);

	    Debug(Channel(DBG_REDUCE_DETAIL) << *this << "\n");
	    
	    j = term->find_diff(*otherterm); // returns -1 if more than 1 diff
	    if (j >= 0) { 
	       // term and otherterm are identical 
	       // except in the set indexed by j
	       if (j != NT_SET_COUNT) // o/w *term == *otherterm
		  (*term)[j] |= (*otherterm)[j];
		  
	       delete otherterm;
	       terms.del(pixj);

	       change = 1;

	       int jj;
	       for (jj = 0; jj < NT_SET_COUNT; jj++)
		  if (!(*term)[jj].universal())
		     break;
	       if (jj == NT_SET_COUNT) { 
		  // we became ANY
		  become_universal();
		  return;
	       }
	    }
	 }
	 Debug(Channel(DBG_REDUCE) << *this << "\n");
      }
   }
}

void NormalExpression::do_or(NormalExpression &other) {
   NormalTerm *term, *otherterm;

   if (any() == ANY)
      return;
   if (other.any() == NOT_ANY)
      return;

   if (other.any() == ANY) { // become ANY
      become_universal();
      return;
   }

   if (any() == NOT_ANY) { // become other
      terms.join(other.terms);
      singleton_flag = other.singleton_flag;
      return;
   }

   ASSERT(any() == NEITHER && other.any() == NEITHER);

   if (singleton_flag >= 0 && other.singleton_flag == singleton_flag) { 
      // special case for speedup
      term = terms(terms.first());
      otherterm = other.terms(other.terms.first());
      (*term)[singleton_flag] |= (*otherterm)[singleton_flag];
      if ((*term)[singleton_flag].universal())
	 become_universal();
      return;
   } 

   // GENERAL CASE

   singleton_flag = -1;

   // add his terms to my terms
   terms.join(other.terms);

   Debug(Channel(DBG_OR) << *this << "\n");
 
   // get rid of duplicate terms
   reduce();
}

void NormalExpression::do_and(NormalExpression &other) {
   NormalTerm *term, *newt, *otherterm;
   NormalExpression result;

   if (any() == NOT_ANY)
      return;
   if (other.any() == ANY)
      return;

   if (other.any() == NOT_ANY) { // become NOT ANY
      become_not_any();
      return;
   }

   if (any() == ANY) { // become other
      terms.clear();
      terms.join(other.terms);
      singleton_flag = other.singleton_flag;
      return;
   }

   ASSERT(any() == NEITHER && other.any() == NEITHER);

   if (singleton_flag >= 0 && other.singleton_flag == singleton_flag) { 
      // special case for speedup
      term = terms(terms.first());
      otherterm = other.terms(other.terms.first());
      (*term)[singleton_flag] &= (*otherterm)[singleton_flag];
      if ((*term)[singleton_flag].empty())
	 become_not_any();
      return;
   } 

   singleton_flag = -1;

   if (terms.length() == 1 && other.terms.length() == 1) { 
      // special case for speedup
      term = terms(terms.first());
      otherterm = other.terms(other.terms.first());
      for (int i = 0; i < NT_SET_COUNT; i++) {
	 (*term)[i] &= (*otherterm)[i];
	 if ((*term)[i].empty()) {
	    become_not_any();
	    break;
	 }
      }
      return;
   }

   // GENERAL CASE

   // do a cartesian product
   for (Pix pixi = terms.first(); pixi;  terms.next(pixi)) {
      term = terms(pixi);
      for (Pix pixj = other.terms.first(); pixj; other.terms.next(pixj)) {
	 otherterm = other.terms(pixj);
	 newt = new NormalTerm;

	 for (int i=0; i < NT_SET_COUNT; i++) {
	    (*newt)[i] = (*term)[i];
	    (*newt)[i] &= (*otherterm)[i];
	    if ((*newt)[i].empty()) {
	       delete newt;
	       goto skip_otherterm;
	    }
	 }

	 result.terms.append(newt);
	       
	skip_otherterm: ;
      }
   }
   
   terms.clear();
   terms.join(result.terms);
   if (!terms.empty())
      reduce();   // get rid of duplicate terms
}

void NormalExpression::do_not() {
// the value of singleton_flag is not affected by do_not
   NormalTerm *term, *newt;
   NormalExpression result;
   NormalExpression tmpexp;

   if (any() == NOT_ANY) { // become ANY
      become_universal();
      return;
   }
   if (any() == ANY) { // become NOT_ANY
      become_not_any();
      return;
   }

   if (singleton_flag >= 0) { // special case for speedup
      ~(*terms(terms.first()))[singleton_flag];
      return;
   } 


   // GENERAL CASE
   result.become_universal();
 
   for (Pix pixi = terms.first(); pixi;  terms.next(pixi)) {
      term = terms(pixi);
      for (int i = 0; i < NT_SET_COUNT; i++) {
	 if (!(*term)[i].universal()) {
	    newt = new NormalTerm;
	    (*newt)[i] = (*term)[i];
	    ~(*newt)[i];

	    newt->make_universal(i);
	    
	    tmpexp.terms.append(newt);
	 }
      }

      Debug(Channel(DBG_NOT) << tmpexp << "\n"); 
      result.do_and(tmpexp);
      tmpexp.terms.clear(); // this also free's elements' memory
   }

   terms.clear();
   terms.join(result.terms);
}

void NormalExpression::evaluate(int expand) {
   Pix pixi, pixj;

   for (pixi = pixj = terms.first(); pixi;  pixi = pixj) {
      terms(pixi)->evaluate(expand);
      terms.next(pixj);
      if (terms(pixi)->empty()) {
	 delete terms(pixi);
	 terms.del(pixi);
      }
   }

   reduce();
   singleton_flag = -1;
}
