// $Id: pmatch.cc 1.6 Fri, 18 Jul 1997 15:53:23 -0700 wlee $
// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu (cengiz@isi.edu)

#include "config.hh"
#include <cstdlib>

extern "C" {
#if HAVE_UNISTD_H
#   include <unistd.h>
#endif
}
#include <iostream.h>
#include <iomanip.h>
#include "dbase.hh"
#include "Node.h"
#include "debug.hh"
#include "trace.hh"
#include "rusage.hh"
#include "pmatch.hh"
#include "Argv.hh"
#include "version.hh"

Rusage ru;
Route rt;
Pix my_as_no;
AutNum *my_autnum;

int  opt_rusage    = 0;
char *opt_prompt   = "pmatch> ";
char *opt_my_as    = NULL;
int  opt_as_out    = 0;
int  opt_echo_as   = 0;
int  opt_interactive = 1;
int  opt_mms       = 0;

int start_tracing(char *dst, char *key, char *nextArg) {
   if (nextArg) {
      trace.enable(nextArg);
      return 1; // return 1 to signify nextArg is used by us
   }
   return 0; 
}

int start_debugging(char *dst, char *key, char *nextArg) {
   if (nextArg) {
      Debug(dbg.enable(atoi(nextArg)));
      return 1; // return 1 to signify nextArg is used by us
   }
   return 0;
}

void init_and_set_options (int argc, char **argv, char **envp) {
   ArgvInfo argTable[] = {
      // RAToolSet common arguments
      // key, type, src, dst, help
      {"-T", ARGV_FUNC, (char *) &start_tracing,      (char *) NULL, 
       "Start tracing the next argument"},
      {"-D", ARGV_FUNC, (char *) &start_debugging,    (char *) NULL, 
       "Start debugging the next argument"},
      {"-version", ARGV_FUNC, (char *) &version,      (char *) NULL,
       "Show version"},
      {"-h", ARGV_FUNC, (char *) &Whois::ArgvHost,    (char *) NULL,
       "Host name of the RAWhoisd server"},
      {"-p", ARGV_FUNC, (char *) &Whois::ArgvPort,    (char *) NULL,
       "Port number of the RAWhoisd server"},
      {"-s", ARGV_FUNC, (char *) &Whois::ArgvSources, (char *) NULL,
       "Order of databases"},
      {"-rusage", ARGV_CONSTANT, (char *) 1,          (char *) &opt_rusage,
       "On termination print resource usage"},
      {"-prompt", ARGV_STRING,  (char *) NULL,        (char *) &opt_prompt,
       "Prompt"},
      {"-ignore_errors", ARGV_FUNC, (char *)&Whois::IgnoreErrors, (char *)NULL,
       "Ignore IRR error and warning messages"},
      {"-report_errors", ARGV_FUNC, (char *)&Whois::ReportErrors, (char *)NULL,
       "Print IRR error and warning messages"},

      // pmatch specific arguments
      {"-as",  ARGV_STRING,    (char *) NULL,    (char *) &opt_my_as,
       "AS number of the aut-num object to use."},
      {"-mms",  ARGV_CONSTANT,    (char *) 1,    (char *) &opt_mms,
       "Route filters match more specific routes"},
      {"-as_out",  ARGV_CONSTANT, (char *) 1,    (char *) &opt_as_out,
       "Match against as-out policy as well"},
      {"-echo_as", ARGV_CONSTANT, (char *) 1,    (char *) &opt_echo_as,
       "Print the autnum object"},

      {(char *) NULL, ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
   };

   for (char **p = envp; *p != NULL; p++) {
      if (strncmp(*p, "IRR_HOST=", 9) == 0)  {
	 whois.SetDefaultHost(*p + 9);
         continue;
      }
      if (strncmp(*p, "IRR_PORT=", 9) == 0)  {
	 whois.SetDefaultPort(atoi(*p + 9));
         continue;
      }
      if (strncmp(*p, "IRR_SOURCES=", 12) == 0)  {
	 whois.SetDefaultSources(*p + 12);
         continue;
      }
   }

   if (ParseArgv(&argc, argv, argTable, 0) != ARGV_OK) {
      cerr << endl;
      exit(1);
   }

   // if there are remaining arguments
   // the first one is my as number
   // the second one is a destination prefix
   switch (argc) {
   case 3 :
      opt_my_as = argv[1];
   case 2:
      extern char *xx_input_string;
      xx_input_string = argv[argc - 1];
      opt_interactive = 0;
   case 1:
      break;
   default:
      cerr << "Wrong number of arguments..." << endl;
      exit(-1);
   }

   // have a prompt only if the input is coming from a tty
   if (!opt_interactive || !isatty(fileno(stdin)) || !isatty(fileno(stdout)))
      opt_prompt = NULL;
}

void main (int argc, char **argv, char **envp) {
   Pix my_rt;
   extern int xxparse();

   init_and_set_options(argc, argv, envp);

   if (opt_my_as)
      my_as_no = AS_map.add_entry(opt_my_as);
   else {
      ipAddr myip;
      myip.setToMyHost();
      my_as_no = AS_map.add_entry(myip);
   }      
   
   my_autnum = AS_map.define(my_as_no);

   if (opt_echo_as) {
      cout << AS_map(my_as_no) << endl;
      my_autnum->print();
   }

   cout << "Matching against aut-num: " << AS_map(my_as_no) << endl;

   xxparse();
   
   if (opt_prompt)
      cout << endl;
	
   if (opt_rusage)
      clog << ru;
}

void match(AutNum *as, Route &r) {
   int (FilterNode::*match_ptr)(Route &r);

   if (opt_mms)
      match_ptr = &FilterNode::match_exact_or_more_specific;
   else
      match_ptr = &FilterNode::match;

   if (as->asno)
      cout << "AS" << AS_map.key(as->asno) << "\n";

   ASPolicy *p;
   Filter_Action *fap;
   InterASPolicy *ip;
   int as_match;

   for (p = as->peers.head(); p; p = as->peers.next(p->peers)) {
      as_match = 0;
      for (fap = p->in.head(); fap; fap = p->in.next(fap->falist))
	 if ((fap->filter->*match_ptr)(r)) {
	    cout << "as-in: " << AS_map.key(p->peeras) << " ";
	    fap->action->InOrderPrint();
	    cout << "\n";
	    as_match = 1;
	 }

      if (as_match)
	 for (ip = p->interas.head(); ip; ip = p->interas.next(ip->interas))
	    for (fap = ip->in.head(); fap; fap = ip->in.next(fap->falist))
	       	 if ((fap->filter->*match_ptr)(r)) {
		    cout << "   interas-in: " << AS_map.key(p->peeras) << " "
			 << Prefask_map(ip->laddress) << " "
			 << Prefask_map(ip->raddress) << " ";
		    fap->action->InOrderPrint();
		    cout << "\n";
		 }

      for (fap = p->dflt.head(); fap; fap = p->dflt.next(fap->falist))
	 if ((fap->filter->*match_ptr)(r)) {
	    cout << "default: " << AS_map.key(p->peeras) << " ";
	    fap->action->InOrderPrint();
	    cout << "\n";
	 }

      if (opt_as_out) {
	 as_match = 0;
	 for (fap = p->out.head(); fap; fap = p->out.next(fap->falist))
	    if ((fap->filter->*match_ptr)(r)) {
	       cout << "as-out: " << AS_map.key(p->peeras) << " ";
	       fap->action->InOrderPrint();
	       cout << "\n";
	       as_match = 1;
	    }

	 if (as_match)
	    for (ip = p->interas.head(); ip; ip = p->interas.next(ip->interas))
	       for (fap = ip->out.head(); fap; fap = ip->out.next(fap->falist))
		  if ((fap->filter->*match_ptr)(r)) {
		     cout << "   interas-out: " << AS_map.key(p->peeras) << " "
			  << Prefask_map(ip->laddress) << " "
			  << Prefask_map(ip->raddress) << " ";
		     fap->action->InOrderPrint();
		     cout << "\n";
		  }
      }
   }
}
