#!/usr/bin/env python

# Copyright (C) 2010  Internet Systems Consortium, Inc. ("ISC")
#
# Permission to use, copy, modify, and/or distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES WITH
# REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
# AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
# LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
# OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
# PERFORMANCE OF THIS SOFTWARE.
#

"""
Client side of AFTR remote configuration

Paul_Selkirk@isc.org, June 2010

aftrconf-client <aftr-ipv6> <command>

commands:
create <user-ipv6> <protocol> <src-ipv4> <src-port> <nat-ipv4> <nat-port>
delete <user-ipv6> <protocol> <src-ipv4> <src-port>
delete <user-ipv6>
flush
get <user-ipv6>
get
"""

__version__ = "$Id: aftrconf-client.py 702 2010-07-02 10:47:16Z fdupont $"

transport = "http"
#transport = "socket"
#transport = "debug"

import sys
import socket
from lxml import etree

NSMAP = {'aftr': 'http://aftr.isc.org/mapping/1.0'}

sockport = 4146
httpport = 4148

def debug(msg):
    """debug helper"""
    #syslog.syslog(syslog.LOG_DEBUG, msg)
    #print msg
    pass

if transport == "socket":
    def send(addr, msg):
        """ send <rpc> request """
        addrlist = socket.getaddrinfo(addr, sockport)
        try:
            sock = socket.socket(addrlist[0][0],
                                 socket.SOCK_STREAM,
                                 socket.IPPROTO_TCP)
        except socket.error as err:
            print "ERROR:", err[1]
            exit()
        try:
            sock.connect(addrlist[0][4])
        except socket.error as err:
            print "ERROR:", err[1]
            exit()
        try:
            sock.sendall(etree.tostring(msg,
                                     encoding='UTF-8',
                                     xml_declaration=True,
                                     pretty_print=True))
        except socket.error as err:
            print "ERROR:", err[1]
            exit()

        response = ""
        while True:
            try:
                buf = sock.recv(1024)
                response += buf
                if len(buf) == 0:
                    break
            except socket.error:
                # host unreachable, connection refused, etc
                pass
        sock.close()
        processresponse(response)

elif transport == "http":
    import httplib
    def send(addr, msg):
        """ send <rpc> request """
        addrlist = socket.getaddrinfo(addr, httpport)
        # XXX error check
        conn = httplib.HTTPConnection(addrlist[0][4][0], addrlist[0][4][1])
        # XXX error check
        try:
            conn.request("POST", "", etree.tostring(msg, pretty_print=True))
        except socket.error as err:
            print "ERROR:", err[1]
            exit()

        # XXX For some reason, BaseHTTPServer does not pass any body text to
        # the handler until it gets EOF, even though we have the correct
        # Content-Length in the request headers, so we have to force the issue
        conn.sock.shutdown(socket.SHUT_WR)

        response = conn.getresponse()
        debug("%s %s" % (response.status, response.reason))
        buf = response.read()
        conn.close()
        processresponse(buf)

else:
    def send(addr, request):
        """ send <rpc> request """
        # debugging: just print the request
        # (this can be piped to the server)
        print etree.tostring(request,
                             encoding='UTF-8',
                             xml_declaration=True,
                             pretty_print=True)

def processresponse(response):
    parser = etree.XMLParser(remove_blank_text=True,
                             remove_comments=True)
    try:
        #debug(response)
        root = etree.fromstring(response, parser)
    except etree.XMLSyntaxError:
        # XXX syslog?
        print "error parsing response"
        raise

    if root.tag != 'rpc-reply':
        print 'invalid document from aftr:'
        print etree.tostring(root,
                             encoding='UTF-8',
                             xml_declaration=True,
                             pretty_print=True)
    else:
        for entry in root:
            if entry.tag == 'ok':
                print 'ok'
            elif entry.tag == 'rpc-error':
                print entry.text
            elif entry.tag == 'conf':
                for nat in entry:
                    if nat.tag != "natEntry":
                        print "unexpected <conf> content: ", nat.tag
                        continue
                    for details in nat:
                        print "%s=%s" % (details.tag, details.text)
                    print
            else:
                print 'unexpected rpc-reply:'
                print etree.tostring(entry,
                                     encoding='UTF-8',
                                     xml_declaration=True,
                                     pretty_print=True)

def parsecreate(args, msg):
    """ parse a create command """
    if len(args) != 9:
        print "usage: create <user-ipv6> <protocol>", \
            "<src-ipv4> <src-port> <nat-ipv4> <nat-port>"
        return
    prog, aftr, cmd, tunnel, proto, saddr, sport, naddr, nport = args
    # XXX sanity check args?

    op = etree.SubElement(msg, 'create')
    etree.SubElement(op, 'tunnel').text = tunnel
    etree.SubElement(op, 'protocol').text = proto
    etree.SubElement(op, 'sourceAddress').text = saddr
    etree.SubElement(op, 'sourcePort').text = sport
    etree.SubElement(op, 'nattedAddress').text = naddr
    etree.SubElement(op, 'nattedPort').text = nport

    send(aftr, msg)

def parsedelete(args, msg):
    """ parse a delete command """
    op = etree.SubElement(msg, 'delete')

    # multiple forms of delete:
    if len(args) == 9:
        prog, aftr, cmd, tunnel, proto, saddr, sport, naddr, nport = args
        # XXX sanity check args?
        etree.SubElement(op, 'tunnel').text = tunnel
        etree.SubElement(op, 'protocol').text = proto
        etree.SubElement(op, 'sourceAddress').text = saddr
        etree.SubElement(op, 'sourcePort').text = sport
        etree.SubElement(op, 'nattedAddress').text = naddr
        etree.SubElement(op, 'nattedPort').text = nport

    elif len(args) == 7:
        prog, aftr, cmd, tunnel, proto, saddr, sport = args
        # XXX sanity check args?
        etree.SubElement(op, 'tunnel').text = tunnel
        etree.SubElement(op, 'protocol').text = proto
        etree.SubElement(op, 'sourceAddress').text = saddr
        etree.SubElement(op, 'sourcePort').text = sport

    elif len(args) == 6:
        prog, aftr, cmd, proto, naddr, nport = args
        # XXX sanity check args?
        etree.SubElement(op, 'protocol').text = proto
        etree.SubElement(op, 'nattedAddress').text = naddr
        etree.SubElement(op, 'nattedPort').text = nport

    elif len(args) == 4:
        prog, aftr, cmd, tunnel = args
        # XXX sanity check args?
        etree.SubElement(op, 'tunnel').text = tunnel

    else:
        print "usage: delete <user-ipv6> <protocol>", \
            "<src-ipv4> <src-port> <nat-ipv4> <nat-port>"
        return

    send(aftr, msg)

def parseflush(args, msg):
    """ parse a flush command """
    if len(args) != 3:
        print "usage: flush"
        return
    prog, aftr, cmd = args
    # XXX sanity check args?

    etree.SubElement(msg, 'flush')

    send(aftr, msg)

def parseget(args, msg):
    """ parse a get command """
    op = etree.SubElement(msg, 'get')

    if len(args) == 4:
        prog, aftr, cmd, tunnel = args
        # XXX sanity check args?
        op.set("tunnel", tunnel)

    elif len(args) == 3:
        prog, aftr, cmd = args

    else:
        print "usage: get [tunnel]"
        return

    send(aftr, msg)

def main(args):
    """main"""
    if len(args) < 3:
        print "usage:", args[0], "<aftr> get|create|delete|flush"
        return

    msg = etree.Element('rpc', nsmap = NSMAP)
    # XXX have to maintain message-id outside the script

    if args[2] == 'create':
        parsecreate(args, msg)
    elif args[2] == 'delete':
        parsedelete(args, msg)
    elif args[2] == 'flush':
        parseflush(args, msg)
    elif args[2] == 'get':
        parseget(args, msg)
    else:
        print 'bad command', args[2]

main(sys.argv)
