/*
 * Copyright (C) 2010-2011  Internet Systems Consortium, Inc. ("ISC")
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/* $Id: pcpiwf.c 1253 2011-07-04 14:54:05Z fdupont $ */

/* pseudo-firewall stuff for PCP */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "upnpglobalvars.h"
#include "upnpredirect.h"
#include "pcpiwf.h"

static uint32_t lasttm;
static struct timeval last;

const char *err_app0_msg = "inet_pton failure";

/* OpenPCPServerSocket() :
 * setup the socket with the PCP server. */
int
OpenPCPServerSocket(struct transaction *t)
{
	struct sockaddr_in6 srv6;
	struct sockaddr_in6 loc6;

	memset(&srv6, 0, sizeof(srv6));
	srv6.sin6_family = AF_INET6;
	srv6.sin6_port = htons(PCP_PORT);
	if (inet_pton(AF_INET6, pcp_server, &srv6.sin6_addr) <= 0) {
		syslog(LOG_ERR,
		       "Failed to convert '%s' to local IPv6 address",
		       pcp_local);
		return PCP_ERR_APP0;
	}
	memset(&loc6, 0, sizeof(loc6));
	loc6.sin6_family = AF_INET6;
	if (inet_pton(AF_INET6, pcp_local, &loc6.sin6_addr) <= 0) {
		syslog(LOG_ERR,
		       "Failed to convert '%s' to server IPv6 address",
		       pcp_server);
		return PCP_ERR_APP0;		
	}
	return pcp_init((struct sockaddr *) &srv6,
			(struct sockaddr *) &loc6,
			&t->pcp);
}

static int
OpenPCPClientSocket(in_addr_t addr)
{
	int fd;
	struct sockaddr_in sin4;

	fd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if (fd < 0) {
		syslog(LOG_ERR, "socket(pcp): %m");
		goto bad;
	}
	memset(&sin4, 0, sizeof(sin4));
	sin4.sin_family = AF_INET;
	sin4.sin_addr.s_addr = addr;
	sin4.sin_port = htons(PCP_PORT);
	if (bind(fd, (struct sockaddr *) &sin4, sizeof(sin4)) < 0) {
		syslog(LOG_ERR, "bind(pcp): %m");
		goto bad;
	}
	return fd;
    bad:
	if (fd >= 0)
		(void)close(fd);
	return -1;
}

int
OpenPCPSockets(int *sockets)
{
	int i, j;

	for (i = 0; i < n_lan_addr; i++) {
		sockets[i] = OpenPCPClientSocket(lan_addr[i].addr.s_addr);
		if (sockets[i] < 0) {
			for (j = 0; j < i; j++) {
				(void)close(sockets[j]);
				sockets[j] = -1;
			}
			sockets[i] = -1;
			return -1;
		}
	}
	return 0;
}

static void
pcp_boot_resend(void *arg)
{
	struct transaction *t = (struct transaction *) arg;
	int ret;

	t->status = PCP_TRY_AGAIN;
	ret = pcp_sendrequest(&t->pcp);
	if (ret != PCP_OK)
		syslog(LOG_WARNING, "pcp_boot_resend: %s", pcp_strerror(ret));
	t->retrans.when.tv_sec = now.tv_sec + 60;
	t->retrans.when.tv_usec = now.tv_usec;
	LIST_INSERT_HEAD(&timeouts, &t->retrans, chain);
}

static void
pcp_boot_handler(struct transaction *t)
{
	if (t->status > PCP_OK)
		return;
	if (t->status == PCP_OK) {
		/* done */
		(void) pcp_close(&t->pcp);
		LIST_REMOVE(t, chain);
		free(t);
		return;
	}
	syslog(LOG_WARNING, "pcp_boot_handler: %s", pcp_strerror(t->status));
	/* reset retrans timeout */
	t->retrans.action = pcp_boot_resend;
	t->retrans.arg = t;
	t->retrans.when.tv_sec = now.tv_sec + 60;
	t->retrans.when.tv_usec = now.tv_usec;
	LIST_INSERT_HEAD(&timeouts, &t->retrans, chain);
}

void
pcp_boot(void)
{
	struct transaction *t;
	pcp_request_t req;
	pcp_option_t **options = NULL;

	/* must not fail */
	t = (struct transaction *) malloc(sizeof(*t));
	memset(t, 0, sizeof(*t));
	t->type = TTYPE_SYSTEM;
	if (OpenPCPServerSocket(t) != PCP_OK) {
		free(t);
		return;
	}
	(void) pcp_setreset(&t->pcp, 1, NULL);
	t->status = PCP_NOT_READY;

	memset(&req, 0, sizeof(req));
	/* opcode (map4) */
	req.opcode = PCP_OPCODE_MAP4;
	/* protocol (all=0) */
	/* lifetime (0) */
	/* internal address (192.0.0.2) */
	if (inet_pton(AF_INET, "192.0.0.2", &req.intaddr4) <= 0) {
		syslog(LOG_WARNING, "renew_redirect(inet_pton0)");
		return;
	}
	/* ports (0:0) */
	/* third party (0.0.0.0) */
	if (pcp_third_party4(&options, 0U) != PCP_OK) {
		syslog(LOG_WARNING, "renew_redirect(pcp_third_party)");
		return;
	}
	if (pcp_makerequest(&t->pcp, &req, options) != PCP_OK) {
		(void) pcp_close(&t->pcp);
		free(t);
		pcp_freeoptions(options);
		return;
	}
	pcp_freeoptions(options);

	LIST_INSERT_HEAD(&transactions, t, chain);
	t->handler = pcp_boot_handler;
	pcp_send_request(t);
}

struct client_request {
	struct transaction t;
	int s;
	struct in_addr client_addr;
	unsigned short client_port;
	struct timeout expire;
};

static void
pcp_request_expire(void *arg)
{
	struct client_request *c = (struct client_request *) arg;

	(void) pcp_close(&c->t.pcp);
	LIST_REMOVE(&c->t, chain);
	if (c->t.retrans.action != NULL)
		LIST_REMOVE(&c->t.retrans, chain);
	free(c);
}

static uint16_t
in_cksum(uint8_t *p, u_int l)
{
	u_int sum = 0;

	while (l > 1) {
		sum += *p++ << 8;
		sum += *p++;
		l -= 2;
	}
	if (l == 1)
		sum += *p << 8;
	sum = ((sum >> 16) & 0xffff) + (sum & 0xffff);
	sum += sum >> 16;
	return htons((uint16_t) ~sum);
}

static void
pcp_icmp(struct client_request *c)
{
	struct sockaddr_in addr;
	uint8_t buf[36];
	socklen_t l;
	uint16_t cksum;
	int r;

	r = socket(PF_INET, SOCK_RAW, IPPROTO_ICMP);
	if (r < 0) {
		syslog(LOG_ERR, "pcp_icmp: socket");
		return;
	}
	memset(&addr, 0, sizeof(addr));
	l = sizeof(addr);
	if (getsockname(c->s, (struct sockaddr *) &addr, &l) < 0) {
		syslog(LOG_ERR, "pcp_icmp: getsockname");
		return;
	}

	memset(buf, 0, sizeof(buf));
	/* type = UNREACH */
	buf[0] = 3;
	/* code = UNREACH_PORT */
	buf[1] = 3;
	/* checksum (use UDP one), pptr/gwaddr/void */
	/* ip[0] = version | hlen */
	buf[8] = 0x45;
	/* tos, len = 20+8+28 */
	buf[11] = 56;
	/* id, off, ttl */
	buf[16] = 64;
	/* protocol */
	buf[17] = IPPROTO_UDP;
	/* checksum, source */
	memcpy(buf + 20, &c->client_addr, 4);
	/* destination */
	memcpy(buf + 24, &addr.sin_addr, 4);
	/* UDP source port */
	memcpy(buf + 28, &c->client_port, 2);
	/* UDP destionation port */
	memcpy(buf + 30, &addr.sin_port, 2);
	/* UDP length = 8+28 */
	buf[33] = 36;
	/* IP checksum */
	cksum = in_cksum(buf + 8, 20);
	memcpy(buf + 18, &cksum, 2);
	/* UDP checksum (to adjust ICMP one) */
	cksum = in_cksum(buf, sizeof(buf));
	memcpy(buf + 34, &cksum, 2);

	addr.sin_port = 0;
	if (bind(r, (struct sockaddr *) &addr, l) < 0) {
		syslog(LOG_ERR, "pcp_icmp: bind");
		return;
	}
	addr.sin_addr = c->client_addr;
	if (sendto(r, buf, sizeof(buf), 0, (struct sockaddr *) &addr, l) < 0)
		syslog(LOG_ERR, "pcp_icmp: sendto");
	(void) close(r);
}

static void
pcp_request_handler(struct transaction *t)
{
	struct client_request *c = t->parent;
	uint8_t *resp = NULL;
	uint16_t resplen;
	unsigned int optidx;
	pcp_option_t *tp = NULL;
	struct sockaddr_in client;

	if (t->status > PCP_OK)
		return;
	/* special case for recv() syscall error */
	if (t->status == PCP_ERR_RECV) {
		pcp_icmp(c);
		return;
	}
	/* not protocol errors */
	if ((t->status < PCP_OK) && (t->status > PCP_ERR_PROTOBASE))
		return;
	if (_pcp_getresponse(&t->pcp, &resp, &resplen) != PCP_OK) {
		syslog(LOG_ERR, "pcp_request_handler(getresponse)");
		return;
	}
	/* get back client address */
	memset(&client, 0, sizeof(client));
	client.sin_family = AF_INET;
	client.sin_addr.s_addr = c->client_addr.s_addr;
	client.sin_port = c->client_port;
	/* pop third party */
	for (optidx = 0; optidx < t->response.optcnt; optidx++) {
		if (t->response.options[optidx].code != PCP_OPTION_THIRD_PARTY)
			continue;
		if (t->response.options[optidx].length != 4)
			continue;
		tp = &t->response.options[optidx];
		break;
	}
	if ((tp != NULL) && (memcmp(tp->data, &client.sin_addr, 4) == 0)) {
		memcpy(resp + PCP_CLIENT_ADDR, tp->data, 4);
		if (t->response.optoffs[optidx] + 8 > resplen) {
			memmove(resp + t->response.optoffs[optidx],
				resp + t->response.optoffs[optidx] + 8,
				resplen - (t->response.optoffs[optidx] + 8));
		}
		resplen -= 8;
	}
	/* send response back to the client */
	if (sendto(c->s, resp, (size_t) resplen, 0,
		   (struct sockaddr *) &client, sizeof(client)) < 0)
		syslog(LOG_ERR, "sendto(pcp): %m");
}

void
ProcessPCPRequest(int s)
{
	struct transaction *t;
	struct client_request *c;
	unsigned char req[1040];
	struct sockaddr_in client;
	socklen_t clen = sizeof(client);
	char clientstr[INET_ADDRSTRLEN];
	int cc, ret;

	memset(req, 0, sizeof(req));
	cc = recvfrom(s, req, sizeof(req), 0,
		      (struct sockaddr *) &client, &clen);
	if (cc < 0) {
		syslog(LOG_ERR, "recvfrom(pcp): %m");
		return;
	}
	if (cc < 4) {
		syslog(LOG_WARNING, "ProcessPCPRequest(underrun): %d", cc);
		return;
	}
	/* discarding PCP responses silently */
	if ((req[1] & PCP_OPCODE_RESPONSE) != 0)
		return;
	if (!inet_ntop(AF_INET, &client.sin_addr,
		       clientstr, INET_ADDRSTRLEN)) {
		syslog(LOG_ERR, "ProcessPCPRequest(inet_ntop)");
		return;
	}
	syslog(LOG_INFO, "PCP request received from %s:%hu %d bytes",
	       clientstr, ntohs(client.sin_port), cc);
	if (cc > 1024) {
		syslog(LOG_WARNING, "ProcessPCPRequest(overrun): %d", cc);
		/* just let 4 extra octets go through */
		if (cc > 1028)
			cc = 1028;
	}

	/* known? */
	LIST_FOREACH(t, &transactions, chain) {
		uint8_t *p = NULL;
		uint16_t l;

		if (t->type != TTYPE_PCP)
			continue;
		c = t->parent;
		if ((c->s == s) &&
		    (c->client_addr.s_addr == client.sin_addr.s_addr) &&
		    (c->client_port == client.sin_port) &&
		    (_pcp_getrequest(&t->pcp, &p, &l) == PCP_OK) &&
		    ((int) l == cc) &&
		    (memcmp(p, req, (size_t) l) == 0)) {
			if (t->status <= 0)
				pcp_request_handler(t);
			return;
		}
	}

	/* new one */
	c = (struct client_request *)malloc(sizeof(*c));
	if (c == NULL)
		return;
	memset(c, 0, sizeof(*c));
	/* fill client request structure */
	ret = OpenPCPServerSocket(&c->t);
	if (ret != PCP_OK) {
		syslog(LOG_WARNING,
		       "ProcessPCPRequest(init): %s",
		       pcp_strerror(ret));
		free(c);
		return;
	}
	c->t.parent = c;
	c->t.type = TTYPE_PCP;
	c->t.status = PCP_NOT_READY;
	c->s = s;
	c->client_addr.s_addr = client.sin_addr.s_addr;
	c->client_port = client.sin_port;
	if (memcmp(req + PCP_CLIENT_ADDR, &client.sin_addr, 4) == 0) {
		if (inet_pton(AF_INET, "192.0.0.2",
			      req + PCP_CLIENT_ADDR) <= 0) {
			(void) pcp_close(&c->t.pcp);
			free(c);
			return;
		}
		req[cc] = PCP_OPTION_THIRD_PARTY;
		req[cc + 3] = 4;
		memcpy(req + cc + 4, &client.sin_addr, 4);
		cc += 8;
	}
	if (_pcp_setrequest(&c->t.pcp, req, (uint16_t) cc) != PCP_OK) {
		(void) pcp_close(&c->t.pcp);
		free(c);
		return;
	}
	c->expire.when.tv_sec = now.tv_sec + 120;
	c->expire.when.tv_usec = now.tv_usec;
	c->expire.action = pcp_request_expire;
	c->expire.arg = c;
	LIST_INSERT_HEAD(&timeouts, &c->expire, chain);

	/* booting */
	if (external_addr == 0) {
		c->t.status = PCP_ERR_RECV;
		pcp_icmp(c);
	}

	/* forward */
	LIST_INSERT_HEAD(&transactions, &c->t, chain);
	c->t.handler = pcp_request_handler;
	pcp_send_request(&c->t);
}

void
get_external_address(uint8_t *buf, uint16_t len)
{
	uint32_t tm, delta;
	in_addr_t addr;
	struct in_addr in;

	if ((len < PCP_LEN_MAP4) ||
	    (buf[PCP_VERSION] != 1) ||
	    (buf[PCP_OPCODE] != (PCP_OPCODE_MAP4 | PCP_OPCODE_RESPONSE))) {
		syslog(LOG_WARNING, "get_external_address: skip");
		return;
	}
	memcpy(&tm, buf + PCP_EPOCH, 4);
	tm = ntohl(tm);
	/* protect against not initialized and clock going backward */
	if ((lasttm != 0) &&
	    ((now.tv_sec > last.tv_sec) ||
	     ((now.tv_sec == last.tv_sec) && (now.tv_usec >= last.tv_usec)))) {
		delta = now.tv_sec - last.tv_sec;
		if ((now.tv_usec - last.tv_usec) >= 500000)
			delta++;
		else if ((now.tv_usec - last.tv_usec) < -500000)
			delta--;
		delta = ((delta << 3) - delta) >> 3;
		if (lasttm + delta > tm + 1) {
			syslog(LOG_ERR, "epoch: server has rebootted");
			exit(0);
		}
	}
	lasttm = tm;
	last = now;
	memcpy(&addr, buf + PCP_MAP_EXTERNAL_ADDR, 4);
	if (addr == 0U) {
		syslog(LOG_WARNING, "no external address: skip");
		return;
	}
	if (external_addr != 0) {
		if (external_addr == addr)
			return;
		syslog(LOG_ERR, "external address: server has rebootted");
		exit(0);
	}
	external_addr = addr;
	in.s_addr = addr;
	syslog(LOG_NOTICE, "new external address '%s'", inet_ntoa(in));
	should_send_public_address_change_notif = 1;
}

void
ProcessPCPResponse(struct transaction *t)
{
	uint8_t *buf;
	uint16_t len;

	t->status = pcp_recvresponse(&t->pcp, &t->response);
	if (t->status == PCP_TRY_AGAIN)
		return;
	else if (t->status < PCP_OK)
		syslog(LOG_WARNING,
		       "ProcessPCPResponse: %s",
		       pcp_strerror(t->status));

	/* check server reboot */
	if (((t->status == PCP_OK) || (t->status <= PCP_ERR_PROTOBASE)) &&
	    (_pcp_getresponse(&t->pcp, &buf, &len) == PCP_OK))
		get_external_address(buf, len);

	/* cancel retrans */
	if (t->retrans.action)
		LIST_REMOVE(&t->retrans, chain);
	memset(&t->retrans, 0, sizeof(t->retrans));

	if (t->handler != NULL)
		(t->handler)(t);
}

static void
pcp_resend(void *arg)
{
	struct transaction *t = (struct transaction *) arg;
	int ret, trycnt;

	ret = pcp_gettries(&t->pcp, &trycnt);
	if (ret < PCP_OK) {
		syslog(LOG_WARNING, "pcp_resend(pcp_gettries)");
		t->status = ret;
		return;
	} else if (trycnt > PCP_MAX_TRIES) {
		memset(&t->retrans, 0, sizeof(t->retrans));
		t->status = PCP_ERR_FAILURE;
		if (t->handler != NULL)
			(t->handler)(t);
		return;
	}
	t->status = PCP_TRY_AGAIN;
	ret = pcp_sendrequest(&t->pcp);
	if (ret != PCP_OK)
		syslog(LOG_WARNING, "pcp_resend: %s", pcp_strerror(ret));
	/* can't fail */
	(void) pcp_gettimeout(&t->pcp, &t->retrans.when, 0);
	LIST_INSERT_HEAD(&timeouts, &t->retrans, chain);
}

void
pcp_send_request(struct transaction *t)
{
	int ret;

	/* set status */
	t->status = PCP_TRY_AGAIN;
	/* send */
	ret = pcp_sendrequest(&t->pcp);
	if (ret != PCP_OK)
		syslog(LOG_WARNING, "pcp_send_request: %s", pcp_strerror(ret));
	/* set retrans timeout */
	t->retrans.action = pcp_resend;
	t->retrans.arg = t;
	/* can't fail */
	(void) pcp_gettimeout(&t->pcp, &t->retrans.when, 0);
	LIST_INSERT_HEAD(&timeouts, &t->retrans, chain);
}
