/*
 * Copyright (C) 1998, 1999, 2000  Internet Software Consortium.
 * 
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM DISCLAIMS
 * ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL INTERNET SOFTWARE
 * CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Id: minfo_14.c,v 1.28 2000/05/22 12:37:44 marka Exp $ */

/* reviewed: Wed Mar 15 17:45:32 PST 2000 by brister */

#ifndef RDATA_GENERIC_MINFO_14_C
#define RDATA_GENERIC_MINFO_14_C

#define RRTYPE_MINFO_ATTRIBUTES (0)

static inline isc_result_t
fromtext_minfo(dns_rdataclass_t rdclass, dns_rdatatype_t type,
	       isc_lex_t *lexer, dns_name_t *origin,
	       isc_boolean_t downcase, isc_buffer_t *target)
{
	isc_token_t token;
	dns_name_t name;
	isc_buffer_t buffer;
	int i;

	REQUIRE(type == 14);

	UNUSED(rdclass);
	
	for (i = 0; i < 2 ; i++) {
		RETERR(gettoken(lexer, &token, isc_tokentype_string,
				ISC_FALSE));
		dns_name_init(&name, NULL);
		buffer_fromregion(&buffer, &token.value.as_region);
		origin = (origin != NULL) ? origin : dns_rootname;
		RETERR(dns_name_fromtext(&name, &buffer, origin,
					 downcase, target));
	}
	return (ISC_R_SUCCESS);
}

static inline isc_result_t
totext_minfo(dns_rdata_t *rdata, dns_rdata_textctx_t *tctx, 
	     isc_buffer_t *target) 
{
	isc_region_t region;
	dns_name_t rmail;
	dns_name_t email;
	dns_name_t prefix;
	isc_boolean_t sub;

	REQUIRE(rdata->type == 14);

	dns_name_init(&rmail, NULL);
	dns_name_init(&email, NULL);
	dns_name_init(&prefix, NULL);

	dns_rdata_toregion(rdata, &region);

	dns_name_fromregion(&rmail, &region);
	isc_region_consume(&region, rmail.length);

	dns_name_fromregion(&email, &region);
	isc_region_consume(&region, email.length);

	sub = name_prefix(&rmail, tctx->origin, &prefix);

	RETERR(dns_name_totext(&prefix, sub, target));

	RETERR(str_totext(" ", target));

	sub = name_prefix(&email, tctx->origin, &prefix);
	return (dns_name_totext(&prefix, sub, target));
}

static inline isc_result_t
fromwire_minfo(dns_rdataclass_t rdclass, dns_rdatatype_t type,
	       isc_buffer_t *source, dns_decompress_t *dctx,
	       isc_boolean_t downcase, isc_buffer_t *target)
{
        dns_name_t rmail;
        dns_name_t email;
        
	REQUIRE(type == 14);

	dns_decompress_setmethods(dctx, DNS_COMPRESS_GLOBAL14);

	UNUSED(rdclass);

        dns_name_init(&rmail, NULL);
        dns_name_init(&email, NULL);

        RETERR(dns_name_fromwire(&rmail, source, dctx, downcase, target));
        return (dns_name_fromwire(&email, source, dctx, downcase, target));
}

static inline isc_result_t
towire_minfo(dns_rdata_t *rdata, dns_compress_t *cctx, isc_buffer_t *target)
{
	isc_region_t region;
	dns_name_t rmail;
	dns_name_t email;

	REQUIRE(rdata->type == 14);

	dns_compress_setmethods(cctx, DNS_COMPRESS_GLOBAL14);

	dns_name_init(&rmail, NULL);
	dns_name_init(&email, NULL);

	dns_rdata_toregion(rdata, &region);

	dns_name_fromregion(&rmail, &region);
	isc_region_consume(&region, name_length(&rmail));

	RETERR(dns_name_towire(&rmail, cctx, target));

	dns_name_fromregion(&rmail, &region);
	isc_region_consume(&region, rmail.length);

	return (dns_name_towire(&rmail, cctx, target));
}

static inline int
compare_minfo(dns_rdata_t *rdata1, dns_rdata_t *rdata2)
{
	isc_region_t region1;
	isc_region_t region2;
	dns_name_t name1;
	dns_name_t name2;
	int order;

	REQUIRE(rdata1->type == rdata2->type);
	REQUIRE(rdata1->rdclass == rdata2->rdclass);
	REQUIRE(rdata1->type == 14);

	dns_name_init(&name1, NULL);
	dns_name_init(&name2, NULL);

	dns_rdata_toregion(rdata1, &region1);
	dns_rdata_toregion(rdata2, &region2);

	dns_name_fromregion(&name1, &region1);
	dns_name_fromregion(&name2, &region2);

	order = dns_name_rdatacompare(&name1, &name2);
	if (order != 0)
		return (order);

	isc_region_consume(&region1, name_length(&name1));
	isc_region_consume(&region2, name_length(&name2));

	dns_name_init(&name1, NULL);
	dns_name_init(&name2, NULL);

	dns_name_fromregion(&name1, &region1);
	dns_name_fromregion(&name2, &region2);

	order = dns_name_rdatacompare(&name1, &name2);
	return (order);
}

static inline isc_result_t
fromstruct_minfo(dns_rdataclass_t rdclass, dns_rdatatype_t type, void *source,
		 isc_buffer_t *target)
{
	dns_rdata_minfo_t *minfo = source;
	isc_region_t region;

	REQUIRE(type == 14);
	REQUIRE(source != NULL);
	REQUIRE(minfo->common.rdtype == type);
	REQUIRE(minfo->common.rdclass == rdclass);

	dns_name_toregion(&minfo->rmailbox, &region);
	RETERR(isc_buffer_copyregion(target, &region));
	dns_name_toregion(&minfo->emailbox, &region);
	return (isc_buffer_copyregion(target, &region));
}

static inline isc_result_t
tostruct_minfo(dns_rdata_t *rdata, void *target, isc_mem_t *mctx)
{
	dns_rdata_minfo_t *minfo = target;
	isc_region_t region;
	dns_name_t name;
	isc_result_t result;

	REQUIRE(rdata->type == 14);
	REQUIRE(target != NULL);

	minfo->common.rdclass = rdata->rdclass;
	minfo->common.rdtype = rdata->type;
	ISC_LINK_INIT(&minfo->common, link);

	dns_name_init(&name, NULL);
	dns_rdata_toregion(rdata, &region);
	dns_name_fromregion(&name, &region);
	dns_name_init(&minfo->rmailbox, NULL);
	RETERR(name_duporclone(&name, mctx, &minfo->rmailbox));
	isc_region_consume(&region, name_length(&name));

	dns_name_fromregion(&name, &region);
	dns_name_init(&minfo->emailbox, NULL);
	result = name_duporclone(&name, mctx, &minfo->emailbox);
	if (result != ISC_R_SUCCESS)
		goto cleanup;
	minfo->mctx = mctx;
	return (ISC_R_SUCCESS);

 cleanup:
	if (mctx != NULL)
		dns_name_free(&minfo->rmailbox, mctx);
	return (ISC_R_NOMEMORY);
}

static inline void
freestruct_minfo(void *source)
{
	dns_rdata_minfo_t *minfo = source;

	REQUIRE(source != NULL);
	REQUIRE(minfo->common.rdtype == 14);

	if (minfo->mctx == NULL)
		return;

	dns_name_free(&minfo->rmailbox, minfo->mctx);
	dns_name_free(&minfo->emailbox, minfo->mctx);
	minfo->mctx = NULL;
}

static inline isc_result_t
additionaldata_minfo(dns_rdata_t *rdata, dns_additionaldatafunc_t add,
		     void *arg)
{
	REQUIRE(rdata->type == 14);

	UNUSED(rdata);
	UNUSED(add);
	UNUSED(arg);

	return (ISC_R_SUCCESS);
}

static inline isc_result_t
digest_minfo(dns_rdata_t *rdata, dns_digestfunc_t digest, void *arg)
{
	isc_region_t r;
	dns_name_t name;
	isc_result_t result;

	REQUIRE(rdata->type == 14);

	dns_rdata_toregion(rdata, &r);
	dns_name_init(&name, NULL);
	dns_name_fromregion(&name, &r);
	result = dns_name_digest(&name, digest, arg);
	if (result != ISC_R_SUCCESS)
		return (result);
	isc_region_consume(&r, name_length(&name));
	dns_name_init(&name, NULL);
	dns_name_fromregion(&name, &r);

	return (dns_name_digest(&name, digest, arg));
}

#endif	/* RDATA_GENERIC_MINFO_14_C */
