/*
** articles in cans, Tobi
*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "configdata.h"
#include "clibrary.h"
#include "libinn.h"
#include "logging.h"
#include "macros.h"
#include "mydir.h"
#include <sys/param.h>
#include <sys/mount.h>
#include "tree.h"
#include "can.h"


ARTCAN	*openCans[2][MAX(MAXOPENREADCANS,MAXOPENWRITECANS)];
int	openCansPos[2];
int	maxOpenCans[2]={MAXOPENREADCANS, MAXOPENWRITECANS};

tree	*canTree[2];


void
CANsetup()
{
    int i;
    
    for( i=0; i<MAX(MAXOPENREADCANS,MAXOPENWRITECANS); i++ )
    {	openCans[0][i]=0;
    	openCans[1][i]=0;
    }

    tree_init(&canTree[READ]);
    tree_init(&canTree[WRITE]);
}

void
CANshutdown()
{
    CANclose();

    tree_mung(&canTree[READ], CANfree);
    tree_mung(&canTree[WRITE], CANnop);
}

int
CANlookup(can)
    ARTCAN *can;
{
    struct stat buf;

    if( can->Name && stat(can->Name, &buf)<0 )
    {	syslog(L_TRACE, "can't stat %s any more, expired?", can->Name);
    	free(can->Name);
    	can->Name=NULL;
    }
    
    return TRUE;
}

void
CANrestart()
{
    tree_trav(&canTree[READ], CANlookup);
}
   
void
CANnop()
{
}

void
CANfree(can)
    ARTCAN	*can;
{
    if( can->Name )
    	free(can->Name);

    free(can);
}

void
CANclose()
{
    int i;
    
    for( i=0; i<maxOpenCans[WRITE]; i++ )
    {	if( openCans[WRITE][i] && openCans[WRITE][i]->fd[WRITE] )
    	    CANcloseCan(WRITE, openCans[WRITE][i]);
    }
    for( i=0; i<maxOpenCans[READ]; i++ )
    {	if( openCans[READ][i] && openCans[READ][i]->fd[READ] )
	    CANcloseCan(READ, openCans[READ][i]);
    }
}

void
CANsync()
{
    int i;
    
    for( i=0; i<maxOpenCans[WRITE]; i++ )
    {	if( openCans[WRITE][i] && openCans[WRITE][i]->dirty )
    	    CANsyncCan(openCans[WRITE][i]);
    }
}

int
compareCanByName(can1, can2)
    ARTCAN	*can1;
    ARTCAN	*can2;
{
    return strcmp(can1->Name, can2->Name);
}

int compareWriteCan(can1, can2)
    ARTCAN	*can1, *can2;
{
    if(can1->Delete<can2->Delete)
    	return -1;
    else if(can1->Delete>can2->Delete)
    	return 1;
    else
    	return 0;
}

int compareReadCan(can1, can2)
    ARTCAN	*can1, *can2;
{
    if(can1->Delete<can2->Delete)
    	return -1;
    else if(can1->Delete>can2->Delete)
    	return 1;
    else if(can1->Number<can2->Number)
    	return -1;
    else if(can1->Number>can2->Number)
    	return 1;
    else
    	return 0;
}

int
CANrecover(fd)
    int fd;
{
    struct 	stat statBuf;
    long	canSize, fileSize, artSize;
    int		artCount;
    int		i;
    char 	buffer[30];
    char	*artBuffer=NULL;
    long	artBufferSize=0;
    
    lseek(fd, 0, SEEK_SET);

    read(fd, buffer, 23);
    if( strncmp(buffer, "Hennerich's articleCan\n", 23)!=0 )
	return -1;
    
    read(fd, buffer, 12);
    if( strncmp(buffer, "Version 1.0\n", 12)!=0 )
	return -2;

    read(fd, buffer, 12);
    canSize=atol(buffer);
    
    fstat(fd, &statBuf);	
    fileSize=statBuf.st_size;
    
    if( canSize!=fileSize )
    {	if( canSize<fileSize )
    	    lseek(fd, canSize, SEEK_SET);
	else
	    lseek(fd, 47, SEEK_SET);			/* !!! */

    	artCount=0;
	
	for( ;; )
	{   if( read(fd, buffer, 12)!=12 )
	    	break;
	
	    if( *buffer!='-' && *buffer!='+' )
		return -3;
	
	    for( i=1; i<11; i++ )
	    {	if( buffer[i]<'0' || buffer[i]>'9' )
		    return -3;
	    }

	    if( i!=11 || buffer[11]!='\n' )
	    	return -3;

	    artSize=atol(&buffer[1]);
	    
	    if( artBufferSize==0 )
	    {	if( (artBuffer=malloc(artSize))<0 )
	    	    return -3;
	    }
	    else if( artBufferSize<artSize )
	    {	if( (artBuffer=realloc(artBuffer, artSize))<0 )
		{   free(artBuffer);
		    return -3;
		}
	    }
	    	
	    if( read(fd, artBuffer, artSize)<artSize )
	    	break;
	
	    canSize+=artSize+12;
	    artCount++;
	}
	
	if( ftruncate(fd, canSize)<0 )
	    return -3;
	
	lseek(fd, canSize, SEEK_SET);
	sprintf(buffer, "-%010d\n", 0);
	
	if( write(fd, buffer, strlen(buffer))<strlen(buffer) )
	    return -3;
	    
	lseek(fd, 23+12, SEEK_SET);
	sprintf(buffer, "+%010ld", canSize+12);

	if( write(fd, buffer, strlen(buffer))<strlen(buffer) )
	    return -3;
	    
	syslog(L_ERROR, "truncated can at bytepos %ld of %ld, recovered %d articles",
				canSize, fileSize, artCount);
	
	return 0;
    }
    
    return 4;
}

int
CANcheck(fd)
    int fd;
{
    struct 	stat statBuf;
    char 	buffer[30];

    lseek(fd, 0, SEEK_SET);

    read(fd, buffer, 23);
    if( strncmp(buffer, "Hennerich's articleCan\n", 23)!=0 )
	return -1;
    
    read(fd, buffer, 12);
    if( strncmp(buffer, "Version 1.0\n", 12)!=0 )
	return -2;

    read(fd, buffer, 12);
    fstat(fd, &statBuf);	
    if( atol(buffer)!=statBuf.st_size )
	return -3;
    
    return 0;
}

void
CANsyncCan(can)
    ARTCAN *can;
{
    char buffer[20];

    lseek(can->fd[WRITE], 35, SEEK_SET);
    read(can->fd[WRITE], buffer, 12);
    syslog(L_TRACE, "syncing can %s, wrote %ld bytes", 
	can->Name, can->Bytepos[WRITE]-atol(buffer));

    sprintf(buffer, "+%010ld\n", can->Bytepos[WRITE]);
    lseek(can->fd[WRITE], 35, SEEK_SET);
    write(can->fd[WRITE], buffer, strlen(buffer));
    
    lseek(can->fd[WRITE], can->Bytepos[WRITE], SEEK_SET);
    can->dirty=FALSE;
}

void
CANcloseCan(rwFlag, can)
    int		rwFlag;
    ARTCAN	*can;
{
    if( rwFlag==WRITE )
    {	if( can->dirty )
	    CANsyncCan(can);
    	syslog(L_TRACE, "closing can %s", can->Name);
    }
    else
    	syslog(L_TRACE, "closing can %s (read)", can->Name);

    close(can->fd[rwFlag]);
    can->fd[rwFlag]=0;
    can->AccessFlag[rwFlag]=FALSE;
}

time_t
CANdelete(canName)
   char *canName;
{
    static struct tm	canTime;
    static time_t theTime=0;
    char	*p;
    static char	buffer[11]="";
    char	tmp;
    
    if( (p=strchr(canName, '/'))!=NULL )
	canName=p+1;

    if( strncmp(canName, buffer, 10)!=0 )
    {   strncpy(buffer, canName, 10);

	if( !theTime )
	{   time(&theTime);
	    canTime=*localtime(&theTime);
	    canTime.tm_min=0;
	    canTime.tm_sec=0;
	    canTime.tm_isdst=-1;
	    buffer[10]='\0';
	}

	tmp=buffer[4];
	buffer[4]='\0';
	canTime.tm_year=atoi(buffer)-1900;
	buffer[4]=tmp;
	
	tmp=buffer[6];
	buffer[6]='\0';
	canTime.tm_mon=atoi(buffer+4)-1;
	buffer[6]=tmp;

	tmp=buffer[8];
	buffer[8]='\0';
	canTime.tm_mday=atoi(buffer+6);
	buffer[8]=tmp;

	canTime.tm_hour=atoi(buffer+8);
    
	theTime=mktime(&canTime);
    }
    
    return theTime;
}

long
CANartsize(can)
    ARTCAN	*can;
{
    char	artsize[12];
    char	*p;
    
    if( read(can->fd[READ], artsize, 12)!=12 )
    {   can->Bytepos[READ]=0;
	return 0;
    }
    else
    	can->Bytepos[READ]+=12;

    if( *artsize=='-' )				/* canceled */
    	return 0;
    else if( *artsize!='+' )			/* uh? */
    	return 0;

    for( p=artsize+1; p-artsize<11; p++ )
    {	if( *p<'0' || *p>'9' )			/* doesn't match header   */
	    return 0;				/* [+-][0-9][0-9]...[0-9] */
    }
    
    if( artsize[11]!='\n' )
    	return 0;
    
    return atol(artsize);
}

int
CANdescriptorCycle(rwFlag, can)
    int		rwFlag;
    ARTCAN	*can;
{
    struct 	stat statBuf;
    int		canCheck;
    int		ocp;
    int		oldOcp;
    int		fd;
    
    oldOcp=openCansPos[rwFlag];

    if( rwFlag==READ )
    {	if( (fd=open(can->Name, O_RDWR, 0666))<0 )
	{   if( CANdelete(can->Name)>time(NULL) )
		syslog(L_ERROR, "cant open %s %m", can->Name);
    
	    return -1;
	}
    }
    else
    {   if( (fd=open(can->Name, O_RDWR, 0666))<0 )
	{   syslog(L_ERROR, "cant open %s %m", can->Name);
	    exit(-1);						/* !!! */
	}
    }

    if( (canCheck=CANcheck(fd))!=0 )
    {	if( rwFlag!=READ || canCheck!=-3 )
	{   switch( canCheck )
	    {	case -1:
		    syslog(L_ERROR, "can %s corrupt, wrong magic", 
					can->Name);
		    break;
		case -2:
		    syslog(L_ERROR, "can %s corrupt, unknown version", 
					can->Name);
		    break;
		case -3:
		    syslog(L_ERROR, "can %s corrupt, wrong size, unclean shutdown?", can->Name);
	    }
	    return canCheck;
	}
    }
    
    do
    {   if( ++openCansPos[rwFlag]==maxOpenCans[rwFlag] )
	    openCansPos[rwFlag]=0;

	ocp=openCansPos[rwFlag];

	if( openCans[rwFlag][ocp]==NULL )
	    openCans[rwFlag][ocp]=can;
	
	if( openCans[rwFlag][ocp]->AccessFlag[rwFlag] )
	    openCans[rwFlag][ocp]->AccessFlag[rwFlag]=0;
	else
	{   if( openCans[rwFlag][ocp]->fd[rwFlag] )
		CANcloseCan(rwFlag, openCans[rwFlag][ocp]);

	    openCans[rwFlag][ocp]=can;	    
	    can->fd[rwFlag]=fd;
	    
	    if( rwFlag==READ )
		can->Bytepos[READ]=47;
	    else
	    {   fstat(fd, &statBuf);    
		lseek(fd, statBuf.st_size, SEEK_SET);
		can->Bytepos[rwFlag]=statBuf.st_size;
	    }
	    
	    can->AccessFlag[rwFlag]=TRUE;

	    if( rwFlag==READ )
	    	syslog(L_TRACE, "opening can %s for reading (oldOcp=%d, newOcp=%d)", can->Name, oldOcp, ocp);
	    else
		syslog(L_TRACE, "opening can %s, starting at %ld (oldOcp=%d, newOcp=%d)", 
		    can->Name, can->Bytepos[rwFlag], oldOcp, ocp);
	}
    } while( openCans[rwFlag][ocp]->AccessFlag[rwFlag]==0 );
    
    return 0;
}



ARTCAN *
CANopenByName(canName)
    char *canName;
{
    ARTCAN	*can;
    ARTCAN	canWON;	/* can With Only Name */
    
    canWON.Name=canName;
    
    if( (can=tree_srch(&canTree[READ], compareCanByName, &canWON))==NULL )
    {	can=NEW(ARTCAN, 1);
    
    	can->Version=1;
	can->Delete=0;
	can->Name=COPY(canName);
	can->Number=0;	/*!!!*/
	can->AccessFlag[READ]=FALSE;
	can->fd[READ]=0;
	can->Bytepos[READ]=0;
	
	tree_add(&canTree[READ], compareCanByName, can, CANnop);
    }
    
    if( can->fd[READ]==0 )
    {	if( CANdescriptorCycle(READ, can) )
	    return NULL;
    }

    can->AccessFlag[READ]=TRUE;
    
    return can;
}


ARTCAN *
CANopenForRead(canpos)
    char	*canpos;
{
    char	buffer[SPOOLNAMEBUFF];
    ARTCAN	*can;
    ARTCAN	canWOD;				/* can With Only Delete */
    long	byteposOfCan;
    char	*p;
					/* shelf.01/yyyymmddhhxx!1234 */
    if( (p=strchr(canpos, '!'))==NULL )
	return NULL;

    strncpy(buffer, canpos, p-canpos);
    buffer[p-canpos]='\0';

    if( (byteposOfCan=atol(++p))==0 )	/*			 1234 */
	return NULL;

    canWOD.Delete=CANdelete(buffer);	/* shelf.01/yyyymmddhhxx */
    p-=3;
    canWOD.Number=atoi(p);		/*                    xx */
    
    if( (can=tree_srch(&canTree[READ], compareReadCan, &canWOD))==NULL )
    	return NULL;
    else if( can->Name==NULL )
    	return NULL;

    if( can->fd[READ]==0 )
    	CANdescriptorCycle(READ, can);

    if( can->Bytepos[READ]!= byteposOfCan )
    {	lseek(can->fd[READ], byteposOfCan, SEEK_SET);
    	can->Bytepos[READ]=byteposOfCan;
    }

    can->AccessFlag[READ]=TRUE;
   
    return can;
}


ARTCAN *
CANopenByNameForRead(canpos)
    char	*canpos;
{
    ARTCAN	*can;
    char	buffer[SPOOLNAMEBUFF];
    long	byteposOfCan;
    char	*p;

    if( (p=strchr(canpos, '!'))==NULL )
	return NULL;

    strncpy(buffer, canpos, p-canpos);
    buffer[p-canpos]='\0';

    if( (byteposOfCan=atol(++p))==0 )
	return NULL;

    if( (can=CANopenByName(buffer))==NULL )
    	return NULL;

    if( can->Bytepos[READ]!=byteposOfCan )
    {	lseek(can->fd[READ], byteposOfCan, SEEK_SET);
    	can->Bytepos[READ]=byteposOfCan;
    }

    return can;
}
   